// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <iostream>
#include "prelude.hpp"

using namespace boost::fcpp;
using std::cout;
using std::endl;

/* A functoid to test the fixpoint operator. Implements the Fibonacci
   function. */
template <class X>
struct FibGenHelper : public c_fun_type<int, int> {
  X x;

  FibGenHelper (const X& in_x) : x(in_x) {}
  
  int operator () (const int& a) const {
    if (a == 0 || a == 1)
      return 1;
    else
      return x(a - 1) + x(a - 2);
  }
};

struct FibGen {
  template <class X>
  struct sig : public fun_type<FibGenHelper<X> > {};

  template <class X>
  FibGenHelper<X> operator () (const X& x) const {
    return FibGenHelper<X>(x);
  }
} fibgen;

// An example of a polymorphic function, to show that Y isn't limited to
// working on monomorphic functions.
template <class Self>
struct MapGenHelper {
   Self self;
   MapGenHelper( const Self& s ) : self(s) {}

   template <class F, class L> struct sig : public fun_type<
      odd_list<typename RT<F,typename L::value_type>::result_type> > {};

   template <class F, class L>
   typename sig<F,L>::result_type
   operator()( const F& f, const L& l ) const {
      if( null(l) )
         return NIL;
      else
         return cons( f(head(l)), thunk2(self,f,tail(l)) );
   }
};

struct MapGen {
   template <class Self>
   struct sig : public fun_type<MapGenHelper<Self> > {};

   template <class Self>
   MapGenHelper<Self> operator()( const Self& s ) const
   { return MapGenHelper<Self>(s); }
} mapgen;

template <class F>
class YHelper {
  F function;
public:
  YHelper(const F& f) : function(f) {}

  template <class A>
  struct sig : public fun_type<
    typename RT<typename RT<F,YHelper<F> >::result_type,A>::result_type> {};

  template <class A> 
  typename sig<A>::result_type operator () (const A& a) const {
    return function(*this)(a);
  }
};

struct Y {
  template <class F>
  struct sig : public fun_type<YHelper<F> > {};

  template <class F>
  typename sig<F>::result_type operator () (const F& f) const {
    return YHelper<F>(f);
  }
} y;

// This is kinda shoddy.  It is not too hard to make a single Y
// combinator which can be applied to any functoid, regardless of the
// number of arguments it expects.  Here we just define two versions:
// "y" which works for one-args and "y2" which works for two-args.  If Y
// is ever move into the library, I'll do it right; as an example file,
// I don't care enough to bother.  :)
template <class F>
class Y2Helper {
  F function;
public:
  Y2Helper(const F& f) : function(f) {}

  template <class A, class B>
  struct sig : public fun_type<
    typename RT<typename RT<F,Y2Helper<F> >::result_type,A,B>::result_type> {};

  template <class A, class B> 
  typename sig<A,B>::result_type 
  operator()( const A& a, const B& b ) const {
    return function(*this)(a,b);
  }
};

struct Y2 {
  template <class F>
  struct sig : public fun_type< Y2Helper<F> > {};

  template <class F>
  typename sig<F>::result_type operator () (const F& f) const {
    return Y2Helper<F>(f);
  }
} y2;

int main () {
  cout <<
    "Testing the Y combinator." << endl <<
    "Should return a finonacci sequence, actually returned: ";
  for (int i = 0; i < 10; i++)
    cout << y(fibgen)(i) << " ";
  cout << endl << endl;

   cout << "Y test to show it works for polymorphic functions" << endl;
   cout << "Should return list of ints starting at 4..." << endl;
   list<int> l = y2(mapgen)( boost::fcpp::plus(3), enum_from(1) );
   l = take(10,l);
   while( !null(l) ) {
      cout << head(l) << " ";
      l = tail(l);
   }
   cout << endl;
}
