// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <iostream>
#include <utility>
#include "prelude.hpp"
#include <boost/test/minimal.hpp>

using namespace boost::fcpp;

using std::pair;
using std::ostream;
using std::cout;
using std::endl;

// This client tries to exhibit basic coverage of the "prelude".

struct Double : public c_fun_type<int,int> {
   int operator()( int x ) const {
      return 2*x;
   }
};

template <class Kind, bool b>
struct test_lazy {
   static void go() {
      // cannot use these unless lazy
      BOOST_CHECK( take(5,iterate(Double(),1))==list_with<Kind>()(1,2,4,8,16) );
      BOOST_CHECK( take(5,repeat(1)) == list_with<Kind>()(1,1,1,1,1) );
      BOOST_CHECK( take(5,cycle(list_with<Kind>()(1,2))) == 
                                                list_with<Kind>()(1,2,1,2,1) );
   }
};
template <class Kind>
struct test_lazy<Kind,false> {
   static void go() { 
   }
};

template <class Kind>
void test() {
   typedef typename Kind::template List<int>::type LI;
   typedef typename Kind::template List<LI>::type LLI;
   typedef typename Kind::template List<bool>::type LB;
   typedef typename Kind::template List<pair<int,int> >::type LPII;

   // Basic list stuff
   LI empty = NIL;
   LI part1 = cons(2,cons(5,cons(7,empty)));
   LI part2 = cons(11,cons(15,cons(18,empty)));
   LI l = cat( list_with<Kind>()(2,5,7), list_with<Kind>()(11,15,18) );
   BOOST_CHECK( l == cat(part1,part2) );
   l = tail(l);
   BOOST_CHECK( l == list_with<Kind>()(5,7,11,15,18) );

   // Prelude list functions
   LLI ll;
   ll = cons( l, ll );
   ll = cons( l, ll );
   BOOST_CHECK( thunk1(concat,ll)() == cat(l,l) );
   BOOST_CHECK( thunk1(init,l)() == list_with<Kind>()(5,7,11,15) );
   BOOST_CHECK( thunk1(last,l)() == 18 );
   BOOST_CHECK( thunk1(length,l)() == 5 );
   BOOST_CHECK( reverse(l) == list_with<Kind>()(18,15,11,7,5) );
   BOOST_CHECK( take_while(less(_,9),l) == list_with<Kind>()(5,7) );
   BOOST_CHECK( drop_while(less(_,9),l) == list_with<Kind>()(11,15,18) );
   BOOST_CHECK( at(l,2) == 11 );
   BOOST_CHECK( filter( bind2of2( greater, 10 ))(l) == 
                                               list_with<Kind>()(11,15,18) );
   BOOST_CHECK( foldr(plus,0,l) == 56 );
   BOOST_CHECK( foldr1(plus,l) == 56 );
   BOOST_CHECK( foldl(plus,0,l) == 56 );
   BOOST_CHECK( foldl1(plus,l) == 56 );
   BOOST_CHECK( scanr(plus,0,l) == cons(56,list_with<Kind>()(51,44,33,18,0)) );
   BOOST_CHECK( scanr1(plus,l) == list_with<Kind>()(56,51,44,33,18) );
   BOOST_CHECK( scanl(plus,0,l) == cons(0,list_with<Kind>()(5,12,23,38,56)) );
   BOOST_CHECK( scanl1(plus,l) == list_with<Kind>()(5,12,23,38,56) );
   BOOST_CHECK( drop(3,l) == list_with<Kind>()(15,18) );

   test_lazy<Kind,LI::is_lazy>::go();

   pair<LI,LI> p = split_at(3)( l );
   BOOST_CHECK( p.first == list_with<Kind>()(5,7,11) );
   BOOST_CHECK( p.second == list_with<Kind>()(15,18) );

   p = span( less(_,7), l );
   BOOST_CHECK( p.first == list_with<Kind>()(5) );
   BOOST_CHECK( p.second == list_with<Kind>()(7,11,15,18) );

   p = break_( greater(_,10), l );
   BOOST_CHECK( fst(p) == list_with<Kind>()(5,7) );
   BOOST_CHECK( snd(p) == list_with<Kind>()(11,15,18) );
   LB b = map( odd_type(), l );

   BOOST_CHECK( and_(b) == false );
   BOOST_CHECK( or_(b) == true );
   BOOST_CHECK( and_(take(1,b)) == true );
   BOOST_CHECK( or_(take(1,b)) == true );

   BOOST_CHECK( any( bind2of2(greater,10))( l) == true );
   BOOST_CHECK( all( bind2of2(greater,10))( l) == false );
   BOOST_CHECK( any( bind2of2(greater,-10))( l) == true );
   BOOST_CHECK( all( bind2of2(greater,-10))( l) == true );
   BOOST_CHECK( elem(10,l) == false );
   BOOST_CHECK( not_elem(10,l) == true );
   BOOST_CHECK( sum(l) == 56 );
   BOOST_CHECK( product(tail(l)) == 20790 );
   BOOST_CHECK( maximum(l) == 18 );
   BOOST_CHECK( minimum(l) == 5 );

   LPII lp = zip(l,repeat(1));
   BOOST_CHECK( lp == list_with<Kind>()( make_pair(5,1), make_pair(7,1), 
      make_pair(11,1), make_pair(15,1), make_pair(18,1) ) );

   p = unzip( lp );
   BOOST_CHECK( p.first == list_with<Kind>()(5,7,11,15,18) );
   BOOST_CHECK( p.second == list_with<Kind>()(1,1,1,1,1) );

   BOOST_CHECK( zip_with(plus,l,repeat(1)) == list_with<Kind>()(6,8,12,16,19) );
   BOOST_CHECK( take(3,enum_from(1)) == list_with<Kind>()(1,2,3) );
   BOOST_CHECK( enum_from_to(1,3) == list_with<Kind>()(1,2,3) );
   BOOST_CHECK( list_until(greater(_,3),plus(1),1)==list_with<Kind>()(1,2,3) );

   // Other prelude functions
   BOOST_CHECK( id( 3.4 ) == 3.4 );
   BOOST_CHECK( compose( plus(1), plus )(2,3) == 6 );
   BOOST_CHECK( of( plus(1), plus )(2,3) == 6 );
   BOOST_CHECK( compose2( plus, negate, id )(3) == 0 );
   BOOST_CHECK( until( greater(_,5), plus(1), 1 ) == 6 );
   BOOST_CHECK( flip(minus)(2,3) == 1 );
   BOOST_CHECK( gcd(30,18) == 6 );
   BOOST_CHECK( odd(1) );
   BOOST_CHECK( even(2) );
   BOOST_CHECK( h_curry( fst, 3, 4 ) == 3 );
   BOOST_CHECK( h_uncurry( plus, make_pair(3,4) ) == 7 );

#ifdef BOOST_FCPP_SAFE_LIST
   {
      list<int> tmp = enum_from_to(1,20000);
      length(tmp);   // force evaluation
      LI l( tmp.begin(), tmp.end() );
      length(l);   // force evaluation
      while( tmp ) tmp = tail(tmp);
   }  // ensure l destructor doesn't blow up
#endif

#ifdef BOOST_FCPP_DEBUG
   {
      LI l;
      try {
         head(l);
      }
      catch( const std::exception& e ) {
         cout << "Successfully caught: " << e.what() << endl;
      }
      try {
         tail(l);
      }
      catch( const std::exception& e ) {
         cout << "Successfully caught: " << e.what() << endl;
      }
   }
#endif

}
   
int test_main(int,char**) {
   cout << "list test" << endl;
   test<UseList>();
   cout << "odd_list test" << endl;
   test<UseOddList>();
   cout << "strict_list test" << endl;
   test<UseStrictList>();
   return 0;
}
