// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_PRE_LAMBDA_HPP
#define BOOST_FCPP_PRE_LAMBDA_HPP

// Some of the stuff here is needed as forward declarations for full.hpp.
// A lot of the stuff here is just handy helpers for lambda.hpp.  

#ifdef BOOST_FCPP_ENABLE_LAMBDA
namespace boost {
namespace fcpp {

// Here is a 'cute' kludge to make "_*_" represent a 'nothing' argument
struct placeholder_for_zero_arguments {};
inline placeholder_for_zero_arguments 
operator*( auto_curry_type, auto_curry_type ) {
   return placeholder_for_zero_arguments();
}

namespace lambda_impl {

//////////////////////////////////////////////////////////////////////
// See operator, overloads in lambda.hpp for namespace explanation
namespace exp {
   struct NIL {};
   template <class H, class T> struct CONS {
      H head;
      T tail;
      CONS( const H& h ) : head(h) {}
      CONS( const H& h, const T& t ) : head(h), tail(t) {}
   };
}
using exp::NIL;
using exp::CONS;
//////////////////////////////////////////////////////////////////////
template <class LA, class LB> struct AppendList;
template <class LB> struct AppendList<NIL,LB> { 
   typedef LB Result; 
   static inline Result go( const NIL&, const LB& x ) { return x; }
};
template <class H, class T, class LB> struct AppendList<CONS<H,T>,LB> { 
   typedef CONS<H,typename AppendList<T,LB>::Result> Result; 
   static inline Result go( const CONS<H,T>& x, const LB& y ) 
   { return Result( x.head, AppendList<T,LB>::go(x.tail,y) ); }
};
//////////////////////////////////////////////////////////////////////
template <class L> struct length_type;
template <> struct length_type<NIL> { static const int value = 0; };
template <class H, class T> struct length_type< CONS<H,T> >
{ static const int value = 1 + length_type<T>::value; };
//////////////////////////////////////////////////////////////////////
// remove every T element from the list A
template <class A, class T> struct Remove;
template <class T> struct Remove<NIL,T> { typedef NIL Result; };
template <class AH, class AT, class T> struct Remove<CONS<AH,AT>,T> 
{ typedef CONS<AH,typename Remove<AT,T>::Result> Result; };
template <class AT, class T> struct Remove<CONS<T,AT>,T> 
{ typedef typename Remove<AT,T>::Result Result; };
///////////////////////////////////////////////////////////////////////
// for each element in the list B, remove that element from list A
template <class A, class B> struct ListDifference;
template <class A> struct ListDifference<A,NIL> { typedef A Result; };
template <class A, class T, class Rest> 
struct ListDifference<A,CONS<T,Rest> > {
   typedef typename Remove<A,T>::Result APrime;
   typedef typename ListDifference<APrime,Rest>::Result Result;
};
//////////////////////////////////////////////////////////////////////
template <class L> struct RemoveDuplicates;
template <> struct RemoveDuplicates<NIL> { typedef NIL Result; };
template <class H, class Rest> struct RemoveDuplicates<CONS<H,Rest> > 
{ typedef CONS<H,typename Remove<Rest,H>::Result> Result; };
//////////////////////////////////////////////////////////////////////
struct LEBase {};     // Base type for all LEs
//////////////////////////////////////////////////////////////////////
#ifdef BOOST_FCPP_LAMBDA_DEBUG
   template <class T, bool b> struct EnsureLEHelper 
   { static inline void go() {} };
   template <class T> struct EnsureLEHelper<T,false> {};
   template <class T> struct EnsureLE { 
      static inline void go() 
      { EnsureLEHelper<T,boost::is_base_and_derived<LEBase,T>::value>::go(); } 
   };
   //////////////////////////////////////////////////////////////////////
   template <class LEL> struct EnsureLEList;
   template <> struct EnsureLEList<NIL> { static inline void go() {} };
   template <class H, class T> struct EnsureLEList<CONS<H,T> > 
   { static inline void go() { EnsureLE<H>::go(); EnsureLEList<T>::go(); } };
#endif
//////////////////////////////////////////////////////////////////////
template <int i, class TypeThunk> struct TEPair {
   static const int my_lv = i;
   typedef TypeThunk MyTypeThunk;
};
// TE is a type environment; a list of TEPair<i,T>
template <int i, class LE> struct BEPair { 
   static const int my_lv = i;
   typedef LE MyLE;
   LE value;  
   BEPair( const LE& x ) : value(x) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<LE>::go(); 
#endif
   } 
};
// BE is a value environment; a list of BEPair<i,LE>
//////////////////////////////////////////////////////////////////////
template <class LEL> struct AccumFreeVars;
template <> struct AccumFreeVars<NIL> { typedef NIL Result; };
template <class H, class T> struct AccumFreeVars<CONS<H,T> > { 
   typedef typename AppendList<typename H::FreeVars,
      typename AccumFreeVars<T>::Result>::Result Result; 
};
//////////////////////////////////////////////////////////////////////
// forward decls
namespace exp { 
   template <class T> struct Value; 
   template <class Fun, class Args> struct Call;
}
//////////////////////////////////////////////////////////////////////
template <int i, class LE> struct Binder { static const int lvnum = i; 
   LE exp; Binder( const LE& e ) : exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<LE>::go(); 
#endif
   } 
};
//////////////////////////////////////////////////////////////////////
#ifdef BOOST_FCPP_LAMBDA_DEBUG
   template <class BL> struct EnsureBinderList;
   template <> struct EnsureBinderList<NIL> { static inline void go() {} };
   template <int i, class LE, class Rest> 
   struct EnsureBinderList<CONS<Binder<i,LE>,Rest> > 
   { static inline void go() { EnsureBinderList<Rest>::go(); } };
#endif
//////////////////////////////////////////////////////////////////////
template <int i, class LE> struct Gets { static const int lvnum = i; 
   LE exp; Gets( const LE& e ) : exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<LE>::go(); 
#endif
   } 
};
//////////////////////////////////////////////////////////////////////
template <class X, class Y> struct AlwaysFirst { typedef X type; };
//////////////////////////////////////////////////////////////////////
typedef enum { IfNormal, IfTrue, IfFalse } IfKind;
//////////////////////////////////////////////////////////////////////
template <bool b, class T>
struct LEifyHelper {
   typedef T type;
   static inline type go( const T& x ) { return x; }
};
template <class T>
struct LEifyHelper<false,T> {
   typedef exp::Value<T> type;
   static inline type go( const T& x ) { return type(x); }
};
template <class T>
struct LEify {
   static const bool b = boost::is_base_and_derived<LEBase,T>::value;
   typedef typename LEifyHelper<b,T>::type type;
   static inline type go( const T& x ) { return LEifyHelper<b,T>::go(x); }
};
//////////////////////////////////////////////////////////////////////
// 4 cases: normal value (LE or otherwise), _*_, NIL, CONS
template <class T> struct LEListify {
   typedef typename LEify<T>::type LE;
   typedef CONS<LE,NIL> type;
   static inline type go( const T& x ) { return type( LEify<T>::go(x) ); }
};
template <> struct LEListify<placeholder_for_zero_arguments> {
   typedef NIL type;
   static inline type go( const placeholder_for_zero_arguments& ) 
   { return type(); }
};
template <> struct LEListify<NIL> {
   typedef NIL type;
   static inline type go( const NIL& ) { return type(); }
};
template <class LE, class Rest> struct LEListify<CONS<LE,Rest> > {
   typedef CONS<LE,Rest> type;
   static inline type go( const type& x ) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLEList<type>::go(); 
#endif
      return x; 
   }
};
//////////////////////////////////////////////////////////////////////
template <class B> struct BinderListify;
template <int i, class LE> struct BinderListify<Binder<i,LE> > {
   typedef CONS<Binder<i,LE>,NIL> type;
   static inline type go( const Binder<i,LE>& b ) { return type(b); }
};
template <class B, class Rest> struct BinderListify<CONS<B,Rest> > {
   typedef CONS<B,Rest> type;
   static inline type go( const type& x ) { return x; }
};
//////////////////////////////////////////////////////////////////////
template <class This, class Arg>
struct BracketCallable {
   typedef typename LEify<This>::type ThisLE;
   typedef exp::Call<ThisLE,typename LEListify<Arg>::type> Result;
   static inline Result go( const This& me, const Arg& arg ) 
   { return Result( LEify<This>::go(me), LEListify<Arg>::go(arg) ); }
};
//////////////////////////////////////////////////////////////////////
template <bool b, class list> struct FilterHelp;
template <class H, class T> struct FilterHelp<true,CONS<H,T> > {
   typedef CONS<H,T> Result;
   static inline Result go( const CONS<H,T>& x ) { return x; }
};
template <class H, class T> struct FilterHelp<false,CONS<H,T> > {
   typedef T Result;
   static inline Result go( const CONS<H,T>& x ) { return x.tail; }
};
template <class PredThunk, class list> struct Filter;
template <class PT> struct Filter<PT,NIL> {
   typedef NIL Result;
   static inline Result go( const NIL& x ) { return x; }
};
template <class PT, class H, class T> struct Filter<PT,CONS<H,T> > {
   static const bool b = PT::template Go<H>::value;
   typedef FilterHelp<b,CONS<H,T> > Help;
   typedef typename Help::Result Result;
   static inline Result go( const CONS<H,T>& x ) { return Help::go(x); }
};
//////////////////////////////////////////////////////////////////////
template <class list, class E> struct Contains;
template <class E> struct Contains<NIL,E> 
{ static const bool value = false; };
template <class H, class T, class E> struct Contains<CONS<H,T>,E> 
{ static const bool value = Contains<T,E>::value; };
template <class T, class E> struct Contains<CONS<E,T>,E> 
{ static const bool value = true; };
//////////////////////////////////////////////////////////////////////
template <class F, class list> struct map_type;
template <class F> struct map_type<F,NIL> { 
   typedef NIL Result;
   static inline Result go( const NIL& x ) { return x; }
};
template <class F, class H, class T> struct map_type<F,CONS<H,T> > { 
   typedef CONS<typename F::template Go<H>::Result,
                typename map_type<F,T>::Result> Result;
   static inline Result go( const CONS<H,T>& x ) {
      return Result( F::template Go<H>::go( x.head ),
                     map_type<F,T>::go( x.tail ) );
   }
};
//////////////////////////////////////////////////////////////////////
template <class Op, class E, class list> struct foldr_type;
template <class Op, class E> struct foldr_type<Op,E,NIL> 
{ typedef E Result; };
template <class Op, class E, class H, class T> 
struct foldr_type<Op,E,CONS<H,T> > { 
   typedef typename foldr_type<Op,E,T>::Result Tmp;
   typedef typename Op::template Go<H,Tmp>::Result Result;
};
//////////////////////////////////////////////////////////////////////
   
} // end namespace lambda_impl
} // end namespace fcpp
} // end namespace boost

#endif
#endif
