// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

// This program is based on the the paper
//    "Monadic Parser Combinators"
// by Graham Hutton and Erik Meijer.

#include <string>
#include <iostream>
#define BOOST_FCPP_ENABLE_LAMBDA
#include "prelude.hpp"

using std::cout;
using std::endl;
using std::string;
using std::cin;
using std::ostream;
using std::flush;

using namespace boost::fcpp;

lambda_var< 1> S;
lambda_var< 2> P;
lambda_var< 3> A;
lambda_var< 4> AS;
lambda_var< 5> C;
lambda_var< 6> F;
lambda_var< 7> X;
lambda_var< 8> Y;
lambda_var< 9> OP;
lambda_var<10> M;
lambda_var<11> N;
lambda_var<12> XS;

typedef list<char> String;

struct ParserM {
   // M a = String -> [(a,String)]

   // We use indirect functoids as a representation type, since we will
   // often need two functions with different behaviors (but the same
   // signatures) to appear to have the same type.
   template <class A> struct rep 
      { typedef fun1<String,list<std::pair<A,String> > > type; };
   template <class MA> struct unrep { 
      typedef typename RT<MA,String>::result_type Tmp;
      typedef typename Tmp::value_type Tmp2;
      typedef typename Tmp2::first_type type; };

   // ReRep is a type-identity in Haskell; here it "indirect"ifies the
   // type, so direct functoids are turned into indirect ones so that
   // only the signature information appears in the type.
   template <class MA> struct ReRep 
      { typedef typename rep<typename unrep<MA>::type>::type type; };

   struct XUnit {
      template <class A> struct sig : public fun_type<
         typename rep<A>::type> {};
      template <class A>
      typename sig<A>::result_type
      operator()( const A& a ) const {
         return lambda(S)[ cons[make_pair[a,S],NIL] ];
      }
   };
   typedef full1<XUnit> unit_type;
   static unit_type unit;

   struct XBind {
      template <class M, class K> struct sig : public fun_type<typename 
         ReRep<typename RT<K,typename unrep<M>::type>::result_type>::type> {};
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         return lambda(S)[ concat[ comp_m<list_m>()
            [ k[fst[P]][snd[P]] | P <= m[S] ] ] ];
      }
   };
   typedef full2<XBind> bind_type;
   static bind_type bind;

   typedef fun1<String,a_unique_type_for_nil> zero_type;
   static zero_type zero;
};
ParserM::unit_type ParserM::unit;
ParserM::bind_type ParserM::bind;
ParserM::zero_type ParserM::zero = ignore( const_(NIL) );

// item :: Parser char
struct XItem : public c_fun_type<String,list<std::pair<char,String> > > {
   // Note that I have "uncurried" the String arg from the parser
   list<std::pair<char,String> > operator()( const String& s ) const {
      if( null(s) )
         return NIL;
      else
         return cons( make_pair( head(s), tail(s) ), NIL );
   }
};
typedef ParserM::rep<char>::type Item;
Item item = XItem();

// plusP :: Parser a -> Parser a -> Parser a
struct XPlusP {
   // Note that I have "uncurried" the String arg from the parser
   template <class P, class Q, class S> struct sig : public fun_type<
      typename RT<P,String>::result_type> {};
   template <class P, class Q>
   typename sig<P,Q,String>::result_type
   operator()( const P& p, const Q& q, const String& s ) const {
      return p(s) ^cat^ thunk1(q,s);
   }
};
typedef full3<XPlusP> PlusP;
PlusP plusP;

template <class Monad>
struct XManyM {
   // Monad a -> Monad [a]
   template <class MA>
   struct sig : public fun_type<typename RT<typename
      plus_m_type<Monad>::type, typename LE<LAM<COMP<CALL<cons_type,LV<1>,LV<2>
      >,GETS<1,MA>,GETS<2,CALL<full1<XManyM<Monad> >,MA> > > > >::type,
      typename RT<typename unit_m_type<Monad>::type, list<typename
      Monad::template unrep<MA>::type> >::result_type>::result_type> {};

   template <class MA>
   typename sig<MA>::result_type operator()( const MA& ma ) const {
      typedef typename Monad::template unrep<MA>::type AA;
      return fcomp_m<Monad>()[ cons[A,AS] | 
         A <= ma, AS <= make_full1(*this)[ma] ]
         ^plus_m<Monad>()^ unit_m<Monad>()( list<AA>() );
   }
};
// many :: Parser a -> Parser [a]
typedef full1<XManyM<ParserM> > Many;
Many many;

struct XSat {
   template <class P> struct sig 
      : public fun_type<ParserM::rep<char>::type> {};
   template <class P>
   typename sig<P>::result_type
   operator()( const P& p ) const {
      return fcomp_m<ParserM>()[ C | C<=item, guard[p[C]] ];
   }
};
// sat :: (char -> bool) -> Parser char
typedef full1<XSat> Sat;
Sat sat;

struct XCharP : public c_fun_type<char, 
   RT<Sat,RT<equal_type,char>::result_type>::result_type> {
   RT<Sat,RT<equal_type,char>::result_type>::result_type
   operator()( char c ) const {
      return sat( equal(c) );
   }
};
// charP :: char -> Parser char
typedef full1<XCharP> CharP;
CharP charP;

typedef ParserM::rep<char>::type Digit;
Digit digit = sat( between('0','9') );

typedef ParserM::rep<char>::type Lower;
Lower lower = sat( between('a','z') );

typedef ParserM::rep<char>::type Upper;
Upper upper = sat( between('A','Z') );

typedef ParserM::rep<char>::type Letter;
Letter letter = lower ^plusP^ upper;

typedef ParserM::rep<char>::type AlphaNum;
AlphaNum alphaNum = letter ^plusP^ digit;

template <class Monad>
struct XMany1M {
   // Monad a -> Monad [a]
   template <class MA>
   struct sig : public fun_type<typename LE<LAM<COMP<
      CALL<delay_type,CALL<cons_type,LV<1>,LV<2> > >,GETS<1,MA>,GETS<2,
      CALL<full1<XManyM<Monad> >,MA> > > > >::type> {};

   template <class MA>
   typename sig<MA>::result_type
   operator()( const MA& ma ) const {
      return fcomp_m<Monad>()[ delay[cons[A,AS]] | 
         A <= ma, AS <= make_full1(XManyM<Monad>())[ma] ];
   }
};
// many1 :: Parser a -> Parser [a]
typedef full1<XMany1M<ParserM> > Many1;
Many1 many1;

struct XChainl1 {
   // Parser a -> Parser (a->a->a) -> Parser a
   // parses a series of items separated by left-associative operators

   typedef bind_m_type<ParserM>::type BIND;
   typedef unit_m_type<ParserM>::type UNIT;
   template <class P, class O>
   struct XRest {
      P p;
      O op;
      XRest( const P& pp, const O& oo ) : p(pp), op(oo) {}

      template <class X> struct sig : public fun_type<
         typename ParserM::rep<X>::type> {};
      template <class X>
      typename sig<X>::result_type
      operator()( const X& x ) const {
         return (op ^BIND()^ lambda(F)[ 
                 p  %BIND()% lambda(Y)[
                 make_full1(*this)[ F[x,Y] ] ] ]) ^plusP^ UNIT()(x);
      }
   };

   template <class P, class O> struct sig : public fun_type<
      typename RT<BIND,P,full1<XRest<P,O> > >::result_type> {};
   template <class P, class O>
   typename sig<P,O>::result_type
   operator()( const P& p, const O& op ) const {
      return p ^BIND()^ make_full1(XRest<P,O>(p,op));
   }
};
typedef full2<XChainl1> Chainl1;
Chainl1 chainl1;

struct XChainr1 {
   // Parser a -> Parser (a->a->a) -> Parser a
   // parses a series of items separated by right-associative operators

   template <class P, class O> struct sig : public fun_type<
      typename ParserM::rep<typename ParserM::unrep<P>::type>::type> {};
   template <class P, class O>
   typename sig<P,O>::result_type
   operator()( const P& p, const O& op ) const {
      return p ^bind_m<ParserM>()^ lambda(X)[ comp_m<ParserM>()
         [ F[X,Y] | F <= op, Y <= make_full2(*this)[p][op] ]
         %plusP% unit_m<ParserM>()[X] ];
   }
};
typedef full2<XChainr1> Chainr1;
Chainr1 chainr1;

struct XChainl {
   template <class P, class O, class V> struct sig : public fun_type<
      typename RT<PlusP,typename RT<Chainl1,P,O>::result_type, typename 
      RT<typename unit_m_type<ParserM>::type,V>::result_type>::result_type> {};
   template <class P, class O, class V>
   typename sig<P,O,V>::result_type
   operator()( const P& p, const O& op, const V& v ) const {
      return (p ^chainl1^ op) ^plusP^ unit_m<ParserM>()(v);
   }
};
typedef full3<XChainl> Chainl;
Chainl chainl;

struct XChainr {
   template <class P, class O, class V> struct sig : public fun_type<
      typename RT<PlusP,typename RT<Chainr1,P,O>::result_type, typename 
      RT<typename unit_m_type<ParserM>::type,V>::result_type>::result_type> {};
   template <class P, class O, class V>
   typename sig<P,O,V>::result_type
   operator()( const P& p, const O& op, const V& v ) const {
      return (p ^chainr1^ op) ^plusP^ unit_m<ParserM>()(v);
   }
};
typedef full3<XChainr> Chainr;
Chainr chainr;

typedef ParserM::rep<int>::type Nat;
Nat nat = lambda()[ let[ 
      OP == lambda(M,N)[ plus[multiplies[10,M],N] ]
      ].in[ comp_m<ParserM>()[ construct1<int>()[minus[X,'0']] | X<=digit ]
            %chainl1% unit_m<ParserM>()[OP] ] ]();

typedef fun1<int,int> f1ii;
typedef ParserM::rep<f1ii>::type MaybeNegate;
MaybeNegate maybeNegate = lambda()[ 
   comp_m<ParserM>()[ f1ii(negate) | charP['-'] ]
   %plusP% unit_m<ParserM>()[ f1ii(id) ] ](); 

typedef ParserM::rep<int>::type IntP;
IntP intP = lambda()[ comp_m<ParserM>()[ OP[N] | OP<=maybeNegate, N<=nat ] ]();

struct XSepBy1 {
   // Parser a -> Parser b -> Parser [a]
   // parses "p (sep p)*", throwing away the separators
   template <class P, class S> struct sig : public fun_type<
      typename RT<typename LE<LAM<COMP<ParserM,CALL<delay_type,
      CALL<cons_type,LV<1>,LV<2> > >,GETS<1,P>,GETS<2,CALL<Many,
      COMP<ParserM,LV<3>,S,GETS<3,P> > > > > > >::type>::result_type> {};
   template <class P, class S>
   typename sig<P,S>::result_type
   operator()( const P& p, const S& sep ) const {
      return fcomp_m<ParserM>()[ delay[cons[X,XS]] |
         X <= p, XS <= many[ comp_m<ParserM>()[ Y | sep, Y <= p ] ] ];
   }
};
typedef full2<XSepBy1> SepBy1;
SepBy1 sepBy1;

// Parser a -> Parser b -> Parser c -> Parser b
struct XBracket {
   template <class O, class P, class C> struct sig : public fun_type<
      typename RT<typename LE<LAM<COMP<ParserM,LV<1>,O,GETS<1,P>,C>
      > >::type>::result_type> {};
   template <class O, class P, class C>
   typename sig<O,P,C>::result_type
   operator()( const O& open, const P& p, const C& close ) const {
      return fcomp_m<ParserM>()[ X | open, X<=p, close ];
   }
};
// bracket :: Parser a -> Parser b -> Parser c -> Parser b
typedef full3<XBracket> Bracket;
Bracket bracket;

// Parser a -> Parser b -> Parser [a]
struct XSepBy {
   template <class P, class S> struct sig : public fun_type<
      typename RT<PlusP,typename RT<SepBy1,P,S>::result_type,
      typename RT<unit_m_type<ParserM>::type,list<typename
      ParserM::unrep<P>::type> >::result_type>::result_type> {};
   template <class P, class S>
   typename sig<P,S>::result_type
   operator()( const P& p, const S& sep ) const {
      typedef typename ParserM::unrep<P>::type A;
      list<A> l = NIL;
      return (p ^sepBy1^ sep) ^plusP^ unit_m<ParserM>()( l );
   }
};
// sepBy :: Parser a -> Parser b -> Parser [a]
typedef full2<XSepBy> SepBy;
SepBy sepBy;

struct XOps {
   // [(Parser a, b)] -> Parser b
   // given a list of pair<parser to parse op,op>, returns a parser
   template <class X> struct sig : public fun_type<
      typename ParserM::rep<typename X::value_type::second_type>::type> {};
   template <class X>
   typename sig<X>::result_type
   operator()( const X& xs ) const {
      list<typename sig<X>::result_type> tmp =   
         map( lambda(P)[ comp_m<ParserM>()[ snd[P] | fst[P] ] ], xs );
      return foldr1( plusP, tmp );
   }
};
// ops :: [(Parser a, b)] -> Parser b
typedef full1<XOps> Ops;
Ops ops;

//////////////////////////////////////////////////////////////////////

ostream& operator<<( ostream& o, const String& s ) {
   string ss( s.begin(), s.end() );
   return o << "\"" << ss << "\"";
}

template <class A, class B>
ostream& operator<<( ostream& o, const std::pair<A,B>& p ) {
   return o << "(" << p.first << "," << p.second << ")";
}

template <class T>
ostream& operator<<( ostream& o, odd_list<T> l ) {
   o << "[" << flush;
   if(l)
      for(;;) {
         o << head(l) << flush;
         l = tail(l);
         if(l)
            o << "," << flush;
         else 
            break;
      }
   return o << "]" << flush;
}

template <class T>
ostream& operator<<( ostream& o, list<T> l ) {
   return o << l.force();
}

//////////////////////////////////////////////////////////////////////

int my_pow( int x, int y ) {
   int r = 1;
   while(y) {
      r *= x;
      --y;
   }
   return r;
}

typedef ParserM::rep<int>::type ExprP;
extern ExprP exprP;

typedef fun2<int,int,int> f2iii;
list<std::pair<ParserM::rep<char>::type,f2iii> > add_op_list = list_with<>()( 
      boost::fcpp::make_pair( charP('+'), f2iii(plus)  ), 
      boost::fcpp::make_pair( charP('-'), f2iii(minus) )   );

typedef ParserM::rep<f2iii>::type AddOp;
AddOp addOp = ops( add_op_list );

typedef ParserM::rep<f2iii>::type ExpOp;
ExpOp expOp = ops( list_with<>()( 
   boost::fcpp::make_pair( charP('^'), f2iii(ptr_to_fun(&my_pow)) )    ) );
   
//////////////////////////////////////////////////////////////////////
// Here I just want to show the straightforward way using indirect
// functoid types:

typedef ParserM::rep<int>::type P_int;
P_int dummy = ParserM::zero;
P_int group=dummy, factor=dummy, term=dummy, expression=dummy;

int main() {
   cout << "Starting program" << endl;

   factor     = lambda(S)[ (intP %plusP% dereference[&group])[S] ];
   term       = factor ^chainr1^ expOp;
   expression = term ^chainl1^ addOp;
   group      = bracket( charP('('), expression, charP(')') );

   string ss;
   cout << "Enter a string to parse (e.g. '11+2^3-(2+-1)^5'):" << endl;
   getline(cin,ss);
   String s( ss.begin(), ss.end() );
   length(s);  // force evaluation

   typedef ParserM P;
   cout << "item: " << item(s) << endl;
   cout << "digit: " << digit(s) << endl;
   cout << "nat: " << nat(s) << endl;
   cout << "list of expression parses:" << endl << expression(s) << endl;
}
