// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#include <vector>
#include <algorithm>
#include <functional>
#include <iostream>
#include "prelude.hpp"

using namespace boost::fcpp;

using std::cout;
using std::endl;

void print( int x ) {
   cout << x << endl;
}
void print2( int x, int y ) {
   cout << "(" << x << "," << y << ")" << endl;
}

struct Foo {
   Foo() { cout << "0-arg" << endl; }
   Foo(int) { cout << "1-arg" << endl; }
   Foo(int,int) { cout << "2-arg" << endl; }
   Foo(int,int,int) { cout << "3-arg" << endl; }
};

int main() {
   using boost::fcpp::plus;
   using boost::fcpp::compose;
   using boost::fcpp::compose2;

   fun1<int,void> pr = ptr_to_fun(&print);
   fun1<int,void> pr_p10 = compose(pr,plus(10));

   // Some tests of the before() effect combinator:
   // Output for this section: 1, 2, 1, 2, 2, 1, 2, 6, 6
   before( thunk1(pr,1), thunk1(pr,2) )();
   before( thunk1(pr,1), pr )(2);
   before( thunk1(pr,2), before( thunk1(pr,1), pr ) )(2);
   cout << before( no_op, const_(6) )() << endl;
   cout << before( no_op, before( no_op, const_(6) ) )() << endl;

cout << "-----" << endl;

   // Ways to define functions like
   //    h(f,g)(x,y)  = { f(x); g(y); }
   // without lambda.  
   // Output for this section: 1, 2, 1, 2, 1, 12
   compose( before(_,pr), thunk2(thunk1,pr) )(1)(2);
   //compose( before(_,pr), thunk2(thunk1,pr) )(1,2); // not split_args-able
   split_args(compose( before(_,pr), thunk2(thunk1,pr) ))(1,2);
   split_args(compose( before(_,pr_p10), thunk2(thunk1,pr) ))(1,2);

cout << "-----" << endl;

   int x = 10, y = 0;
   fun2<int,int,void> f = ptr_to_fun(&print2);
   fun1<int,int>      g = plus(1);
   fun1<int,int>      h = plus(2);

   // Ways to define functions like...
   // ...i(x,y) = f(g(x),h(y))
   // Output for this section: (11,2), (11,2), (11,12), (11,12), (11,12)
   compose(flip(compose)(h),compose(f,g))(x)(y);
   compose(compose(_,h),compose(f,g))(x)(y);

   // ...i(x) = f(g(x),h(x))
   duplicate(compose(flip(compose)(h),compose(f,g)))(x);
   duplicate(compose(compose(_,h),compose(f,g)))(x);
   compose2(f,g,h)(x);

cout << "-----" << endl;

   // Output for this section: 123, 5, 5, 1, 5

   // Testing list_until().
   list<int> l = list_until( ignore(const_(false)), plus(1), 1 );
   cout << at(l,0) << at(l,1) << at(l,2) << endl;

   // Ways to ignore certain arguments
   cout << ignore(plus)(1)(2)(3) << endl;
   cout << ignore(plus)(1,2,3) << endl;
   // ignore second arg
   cout << flip(ignore(length)) (list_with<>()(1),4) << endl;
   // ignore third arg
   cout << split_args(make_full3(bind2and3of3)(ignore(plus))) (2,3,l) << endl;

cout << "-----" << endl;

   // Some tests of the after() effect combinator.
   // Output for this section: 3,0,3,5,6,7
   cout << after(const_(0),thunk1(pr,3))() << endl;
   cout << after(plus,thunk1(pr,3))(2,3) << endl;
   after( emptify(thunk1(pr,6)), thunk1(pr,7) )();

cout << "-----" << endl;

   // Testing maybe datatype
   // Output for this section: nothing,3,nothing,3
   maybe<int> xx = maybe<int>();
   maybe<int> yy = maybe<int>(3);
   if( !xx.is_nothing() ) 
      cout << xx.value() << endl; 
   else 
      cout << "nothing" << endl;
   if( !yy.is_nothing() ) 
      cout << yy.value() << endl; 
   else 
      cout << "nothing" << endl;

   xx = NOTHING;
   yy = just(3);
   if( !xx.is_nothing() ) 
      cout << xx.value() << endl; 
   else 
      cout << "nothing" << endl;
   if( !yy.is_nothing() ) 
      cout << yy.value() << endl; 
   else 
      cout << "nothing" << endl;

cout << "-----" << endl;

   // Tests constructN
   // Output for this section: 0-arg,1-arg,2-arg,3-arg
   before( no_op, construct0<Foo>() )();
   before( no_op, construct1<Foo>() )(1);
   before( no_op, construct2<Foo>() )(1,1);
   before( no_op, construct3<Foo>() )(1,1,1);

cout << "-----" << endl;

   // Tests force, delay, dereference, address_of
   // Output for this section: 0, 0, 3
   list<int> lll = repeat(0);
   cout << head( compose(force,tail)(lll) ) << endl;
   cout << head( compose(delay,compose(force,tail))(lll) ) << endl;
   int xyz = 3;
   cout << compose(dereference,address_of)(xyz) << endl;
 
   return 0;
}
