// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_REUSE_HPP
#define BOOST_FCPP_REUSE_HPP

#include "function.hpp"

namespace boost {
namespace fcpp {

// This is really a "list.hpp" thing, but Reusers borrow NIL to do some of
// their dirty work.
struct a_unique_type_for_nil {
   bool operator==( a_unique_type_for_nil ) const { return true; }
   bool operator< ( a_unique_type_for_nil ) const { return false; }
};
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN a_unique_type_for_nil NIL;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// "Reuser"s are effectively special-purpose versions of curry() that
// enable recursive list functoids to reuse the thunk of the curried
// recursive call.  See 
//    http://www.cc.gatech.edu/~yannis/fc++/New/reusers.html
// for a more detailed description.
//////////////////////////////////////////////////////////////////////

// For efficiency, we mark parameters as either "VAR"iant or "INV"ariant.
struct INV {};
struct VAR {};

template <class V, class X> struct Maybe_Var_Inv;
template <class X>
struct Maybe_Var_Inv<VAR,X> {
   static void remake( X& x, const X& val ) {
      x.~X();
      new (&x) X(val);
   }
   static X clone( const X& x ) { return X(x); }
};
template <class X>
struct Maybe_Var_Inv<INV,X> {
   static void remake( X&, const X& ) {}
   static const X& clone( const X& x ) { return x; }
};

//////////////////////////////////////////////////////////////////////

template <class R, class ThunkRef>
const Fun0Impl<R>* to_fun0( ThunkRef x ) {
   return &*x;
}

//////////////////////////////////////////////////////////////////////

template <class V1, class F> 
struct reuser0;

template <class V1, class F, class R>
struct Thunk0 : public Fun0Impl<R> {
   mutable F f;
   Thunk0( const F& ff ) : f(ff) {}
   void init( const F& ff ) const {
      Maybe_Var_Inv<V1,F>::remake( f, ff );
   }
   R operator()() const {
      return Maybe_Var_Inv<V1,F>::clone(f)( reuser0<V1,F>(this) );
   }
};

template <class V1, class F>
struct reuser0 {
   typedef typename RT<F>::result_type R;
   typedef Thunk0<V1,F,R> M;
   boost::intrusive_ptr<const M> ref;
   reuser0(a_unique_type_for_nil) {}
   reuser0(const M* m) : ref(m) {}
   fun0<R> operator()( const F& f ) {
      if( !ref )   ref = boost::intrusive_ptr<const M>( new M(f) );
      else         ref->init(f);
      return fun0<R>( 1, to_fun0<R>(ref) );
   }
   void iter( const F& f ) {
      if( ref )    ref->init(f);
   }
};

//////////////////////////////////////////////////////////////////////

template <class V1, class V2, class F, class X> 
struct reuser1;

template <class V1, class V2, class F, class X, class R>
struct Thunk1 : public Fun0Impl<R> {
   mutable F f;
   mutable X x;
   Thunk1( const F& ff, const X& xx ) : f(ff), x(xx) {}
   void init( const F& ff, const X& xx ) const {
      Maybe_Var_Inv<V1,F>::remake( f, ff );
      Maybe_Var_Inv<V2,X>::remake( x, xx );
   }
   R operator()() const {
      return Maybe_Var_Inv<V1,F>::clone(f)( 
         Maybe_Var_Inv<V2,X>::clone(x), 
         reuser1<V1,V2,F,X>(this) );
   }
};

template <class V1, class V2, class F, class X>
struct reuser1 {
   typedef typename RT<F,X>::result_type R;
   typedef Thunk1<V1,V2,F,X,R> M;
   boost::intrusive_ptr<const M> ref;
   reuser1(a_unique_type_for_nil) {}
   reuser1(const M* m) : ref(m) {}
   fun0<R> operator()( const F& f, const X& x ) {
      if( !ref )   ref = boost::intrusive_ptr<const M>( new M(f,x) );
      else         ref->init(f,x);
      return fun0<R>( 1, to_fun0<R>(ref) );
   }
   void iter( const F& f, const X& x ) {
      if( ref )    ref->init(f,x);
   }
};

//////////////////////////////////////////////////////////////////////

template <class V1, class V2, class V3, class F, class X, class Y> 
struct reuser2;

template <class V1, class V2, class V3, class F, class X, class Y, class R>
struct Thunk2 : public Fun0Impl<R> {
   mutable F f;
   mutable X x;
   mutable Y y;
   Thunk2( const F& ff, const X& xx, const Y& yy ) : f(ff), x(xx), y(yy) {}
   void init( const F& ff, const X& xx, const Y& yy ) const {
      Maybe_Var_Inv<V1,F>::remake( f, ff );
      Maybe_Var_Inv<V2,X>::remake( x, xx );
      Maybe_Var_Inv<V3,Y>::remake( y, yy );
   }
   R operator()() const {
      return Maybe_Var_Inv<V1,F>::clone(f)( 
         Maybe_Var_Inv<V2,X>::clone(x), 
         Maybe_Var_Inv<V3,Y>::clone(y), 
         reuser2<V1,V2,V3,F,X,Y>(this) );
   }
};

template <class V1, class V2, class V3, class F, class X, class Y>
struct reuser2 {
   typedef typename RT<F,X,Y>::result_type R;
   typedef Thunk2<V1,V2,V3,F,X,Y,R> M;
   boost::intrusive_ptr<const M> ref;
   reuser2(a_unique_type_for_nil) {}
   reuser2(const M* m) : ref(m) {}
   fun0<R> operator()( const F& f, const X& x, const Y& y ) {
      if( !ref )   ref = boost::intrusive_ptr<const M>( new M(f,x,y) );
      else         ref->init(f,x,y);
      return fun0<R>( 1, to_fun0<R>(ref) );
   }
   void iter( const F& f, const X& x, const Y& y ) {
      if( ref )    ref->init(f,x,y);
   }
};

//////////////////////////////////////////////////////////////////////

template <class V1, class V2, class V3, class V4, 
          class F, class X, class Y, class Z> 
struct reuser3;

template <class V1, class V2, class V3, class V4, 
          class F, class X, class Y, class Z, class R>
struct Thunk3 : public Fun0Impl<R> {
   mutable F f;
   mutable X x;
   mutable Y y;
   mutable Z z;
   Thunk3( const F& ff, const X& xx, const Y& yy, const Z& zz ) 
      : f(ff), x(xx), y(yy), z(zz) {}
   void init( const F& ff, const X& xx, const Y& yy, const Z& zz ) const {
      Maybe_Var_Inv<V1,F>::remake( f, ff );
      Maybe_Var_Inv<V2,X>::remake( x, xx );
      Maybe_Var_Inv<V3,Y>::remake( y, yy );
      Maybe_Var_Inv<V4,Z>::remake( z, zz );
   }
   R operator()() const {
      return Maybe_Var_Inv<V1,F>::clone(f)( 
         Maybe_Var_Inv<V2,X>::clone(x), 
         Maybe_Var_Inv<V3,Y>::clone(y), 
         Maybe_Var_Inv<V4,Z>::clone(z), 
         reuser3<V1,V2,V3,V4,F,X,Y,Z>(this) );
   }
};

template <class V1, class V2, class V3, class V4,
          class F, class X, class Y, class Z>
struct reuser3 {
   typedef typename RT<F,X,Y,Z>::result_type R;
   typedef Thunk3<V1,V2,V3,V4,F,X,Y,Z,R> M;
   boost::intrusive_ptr<const M> ref;
   reuser3(a_unique_type_for_nil) {}
   reuser3(const M* m) : ref(m) {}
   fun0<R> operator()( const F& f, const X& x, const Y& y, const Z& z ) {
      if( !ref )   ref = boost::intrusive_ptr<const M>( new M(f,x,y,z) );
      else         ref->init(f,x,y,z);
      return fun0<R>( 1, to_fun0<R>(ref) );
   }
   void iter( const F& f, const X& x, const Y& y, const Z& z ) {
      if( ref )    ref->init(f,x,y,z);
   }
};

}  // namespace fcpp
}  // namespace boost

#endif
