// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_OPERATOR_HPP
#define BOOST_FCPP_OPERATOR_HPP

//////////////////////////////////////////////////////////////////////
// The goal here is to provide functoids for most C++ operators (e.g.
// plus_type, greater_type, ...) as well as conversions between representations.
// The conversions include ptr_to_fun, for turning function pointers into
// functoids, stl_to_funN, for turning STL functoids into our functoids,
// and monomophizeN, for converting polymorphic direct functoids into
// monomorphic ones.
//
// There's also some miscellaneous stuff at both the beginning and the 
// end of this file, for lack of a better place to put it.
//////////////////////////////////////////////////////////////////////

#include <utility>
#include <iterator>
#include <iostream>
#include "lambda.hpp"
#include "boost/utility.hpp"

namespace boost {
namespace fcpp {

// Will be defined in function.hpp
template <class R> class fun0;
template <class A1, class R> class fun1;
template <class A1, class A2, class R> class fun2;
template <class A1, class A2, class A3, class R> class fun3;

//////////////////////////////////////////////////////////////////////
// syntactic sugar for infix operators
//////////////////////////////////////////////////////////////////////
// The syntax
//    arg1 ^fun^ arg2     means     fun( arg1, arg2 )
// like Haskell's backquotes.
//
// I feel justified in this convenient abuse of operator overloading in
// that it's complete nonsense for someone to try to XOR a value with a
// 2-argument full functoid.  Put another way, I own full2<F>s; I can do 
// whatever I want with them, darn it!  :)
//
// Note that it also works on Full3s (with currying).

template <class LHS, class Fun>
struct InfixOpThingy {
   // Note that storing const&s here relies on the fact that temporaries
   // are guaranteed to live for the duration of the full-expression in
   // which they are created.  There's no need to create copies.
   const LHS& lhs;
   const Fun& f;
   InfixOpThingy( const LHS& l, const Fun& ff ) : lhs(l), f(ff) {}
};

template <class LHS, class F>
inline InfixOpThingy<LHS,full2<F> >
operator^( const LHS& lhs, const full2<F>& f ) {
   return InfixOpThingy<LHS,full2<F> >(lhs,f);
}

template <class LHS, class A1, class A2, class R>
inline InfixOpThingy<LHS,fun2<A1,A2,R> >
operator^( const LHS& lhs, const fun2<A1,A2,R>& f ) {
   return InfixOpThingy<LHS,fun2<A1,A2,R> >(lhs,f);
}

template <class LHS, class F>
inline InfixOpThingy<LHS,full3<F> >
operator^( const LHS& lhs, const full3<F>& f ) {
   return InfixOpThingy<LHS,full3<F> >(lhs,f);
}

template <class LHS, class A1, class A2, class A3, class R>
inline InfixOpThingy<LHS,fun3<A1,A2,A3,R> >
operator^( const LHS& lhs, const fun3<A1,A2,A3,R>& f ) {
   return InfixOpThingy<LHS,fun3<A1,A2,A3,R> >(lhs,f);
}

template <class LHS, class FF, class RHS>
inline typename RT<FF,LHS,RHS>::result_type
operator^( const InfixOpThingy<LHS,FF>& x, const RHS& rhs ) {
   return x.f( x.lhs, rhs );
}

// FIX infix doesn't always work; need disable_if to do it well (see below)

// Ugh, I need overloads to disambiguate (e.g. "plus(1) ^of^ plus").
// Really, I need disable_if to ensure LHS never matches InfixOpThingy.


// Furthermore, I just can't help myself from making
//    arg1 %fun% arg2     mean     fun[ arg1 ][ arg2 ]
// for use in lambda expressions.  % is a good choice because it binds
// more tightly than <=, so it's less likely to interfere with "gets"
// bindings.

#ifdef BOOST_FCPP_ENABLE_LAMBDA

template <class LHS, class Fun>
struct InfixOpWhatzit {
   // See comment in InfixOpThingy
   const LHS& lhs;
   const Fun& f;
   InfixOpWhatzit( const LHS& l, const Fun& ff ) : lhs(l), f(ff) {}
};

template <class LHS, class F>
inline InfixOpWhatzit<LHS,full2<F> >
operator%( const LHS& lhs, const full2<F>& f ) {
   return InfixOpWhatzit<LHS,full2<F> >(lhs,f);
}

template <class LHS, class A1, class A2, class R>
inline InfixOpWhatzit<LHS,fun2<A1,A2,R> >
operator%( const LHS& lhs, const fun2<A1,A2,R>& f ) {
   return InfixOpWhatzit<LHS,fun2<A1,A2,R> >(lhs,f);
}

template <class LHS, class F>
inline InfixOpWhatzit<LHS,full3<F> >
operator%( const LHS& lhs, const full3<F>& f ) {
   return InfixOpWhatzit<LHS,full3<F> >(lhs,f);
}

template <class LHS, class A1, class A2, class A3, class R>
inline InfixOpWhatzit<LHS,fun3<A1,A2,A3,R> >
operator%( const LHS& lhs, const fun3<A1,A2,A3,R>& f ) {
   return InfixOpWhatzit<LHS,fun3<A1,A2,A3,R> >(lhs,f);
}

template <class LHS, class FF, class RHS>
inline typename LE<CALL<FF,LHS,RHS> >::type
operator%( const InfixOpWhatzit<LHS,FF>& x, const RHS& rhs ) {
   // line below like
   //     x.f[ x.lhs, rhs ]
   // only without possibility of built-in comma operator biting us
   return x.f[ lambda_impl::exp::explicit_comma(x.lhs,rhs) ];
}

#endif

//////////////////////////////////////////////////////////////////////
// operators
//////////////////////////////////////////////////////////////////////

namespace impl {
struct XMakePair {
   template <class A, class B>
   struct sig : public fun_type<std::pair<A,B> > {};

   template <class A, class B>
   std::pair<A,B> operator()( const A& a, const B& b ) const {
      return std::make_pair(a,b);
   }
};
}
typedef full2<impl::XMakePair> make_pair_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN make_pair_type make_pair;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XPlus {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( const T& x, const T& y ) const {
      return std::plus<T>()( x, y );
   }
};
}
typedef full2<impl::XPlus> plus_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN plus_type plus;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XMinus {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( const T& x, const T& y ) const {
      return std::minus<T>()( x, y );
   }
};
}
typedef full2<impl::XMinus> minus_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN minus_type minus;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XMultiplies {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( const T& x, const T& y ) const {
      return std::multiplies<T>()( x, y );
   }
};
}
typedef full2<impl::XMultiplies> multiplies_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN multiplies_type multiplies;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XDivides {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( const T& x, const T& y ) const {
      return std::divides<T>()( x, y );
   }
};
}
typedef full2<impl::XDivides> divides_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN divides_type divides;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XModulus {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( const T& x, const T& y ) const {
      return std::modulus<T>()( x, y );
   }
};
}
typedef full2<impl::XModulus> modulus_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN modulus_type modulus;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XNegate {
   template <class T>
   struct sig : public fun_type<T> {};

   template <class T>
   T operator()( const T& x ) const {
      return std::negate<T>()( x );
   }
};
}
typedef full1<impl::XNegate> negate_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN negate_type negate;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XEqual {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x, const T&y ) const {
      return std::equal_to<T>()( x, y );
   }
};
}
typedef full2<impl::XEqual> equal_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN equal_type equal;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XNotEqual {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x, const T&y ) const {
      return std::not_equal_to<T>()( x, y );
   }
};
}
typedef full2<impl::XNotEqual> not_equal_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN not_equal_type not_equal;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XGreater {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x, const T&y ) const {
      return std::greater<T>()( x, y );
   }
};
}
typedef full2<impl::XGreater> greater_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN greater_type greater;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XLess {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x, const T&y ) const {
      return std::less<T>()( x, y );
   }
};
}
typedef full2<impl::XLess> less_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN less_type less;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XGreaterEqual {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x, const T&y ) const {
      return std::greater_equal<T>()( x, y );
   }
};
}
typedef full2<impl::XGreaterEqual> greater_equal_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN greater_equal_type greater_equal;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XLessEqual {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x, const T&y ) const {
      return std::less_equal<T>()( x, y );
   }
};
}
typedef full2<impl::XLessEqual> less_equal_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN less_equal_type less_equal;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XLogicalAnd {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x, const T&y ) const {
      return std::logical_and<T>()( x, y );
   }
};
}
typedef full2<impl::XLogicalAnd> logical_and_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN logical_and_type logical_and;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XLogicalOr {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x, const T&y ) const {
      return std::logical_or<T>()( x, y );
   }
};
}
typedef full2<impl::XLogicalOr> logical_or_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN logical_or_type logical_or;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XLogicalNot {
   template <class T>
   struct sig : public fun_type<bool> {};

   template <class T>
   bool operator()( const T&x ) const {
      return std::logical_not<T>()( x );
   }
};
}
typedef full1<impl::XLogicalNot> logical_not_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN logical_not_type logical_not;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XMin {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( const T& x, const T& y ) const {
      return less( x, y ) ? x : y;
   }
};
}
typedef full2<impl::XMin> min_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN min_type min;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XMax {
   template <class T, class U> struct sig;

   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( const T& x, const T& y ) const {
      return less( x, y ) ? y : x;
   }
};
}
typedef full2<impl::XMax> max_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN max_type max;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XDereference {
   template <class T> struct sig 
   : public fun_type<typename std::iterator_traits<T>::value_type> {};

   template <class T>
   typename sig<T>::result_type operator()( const T& p ) const {
      return *p;
   }
};
}
typedef full1<impl::XDereference> dereference_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN dereference_type dereference;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XAddressOf {
   template <class T>
   struct sig : public fun_type<const T*> {};

   template <class T>
   const T* operator()( const T& x ) const {
      return boost::addressof(x);
   }
};
}
typedef full1<impl::XAddressOf> address_of_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN address_of_type address_of;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
   struct XDelete_ {
      template <class T> struct sig : public fun_type<void> {};
      template <class T> void operator()( T* p ) const { delete p; }
   };
}
typedef full1<impl::XDelete_> Delete_;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN Delete_ delete_;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
   template <class Dest>
   struct XDynamicCast {
      template <class Src> struct sig : public fun_type<Dest> {};
      template <class Src>
      Dest operator()( const Src& s ) const {
         return dynamic_cast<Dest>(s);
      }
   };
}
template <class T>
struct dynamic_cast_type { typedef full1<impl::XDynamicCast<T> > type; };
template <class T> full1<impl::XDynamicCast<T> > dynamic_cast_()
{ return make_full1( impl::XDynamicCast<T>() ); }

// out_stream is the << stream operator, but takes a stream*
//    e.g.   &cout ^out_stream^ x
namespace impl {
   struct XOutStream {
      template <class StreamP, class Data> struct sig
         : public fun_type<StreamP> {};
      template <class StreamP, class Data>
      StreamP operator()( StreamP s, const Data& x ) const {
         (*s) << x;
         return s;
      }
   };
}
typedef full2<impl::XOutStream> out_stream_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN out_stream_type out_stream;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

// in_stream is the >> stream operator, but takes stream* and data*
//    e.g.   &cin ^in_stream^ &x
namespace impl {
   struct XInStream {
      template <class StreamP, class DataP> struct sig
         : public fun_type<StreamP> {};
      template <class StreamP, class DataP>
      StreamP operator()( StreamP s, DataP x ) const {
         (*s) >> (*x);
         return s;
      }
   };
}
typedef full2<impl::XInStream> in_stream_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN in_stream_type in_stream;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

// make_manip(aStream)(aManip) returns the manipulator for that stream
// (The C++ std stream manipulators have a crazy interface which
// necessitates this ugliness.)
//    e.g.  &cout ^out_stream^ make_manip(cout)(endl)
template <class C, class T>
struct ManipMaker {
   std::basic_ostream<C,T>& (*
   operator()( std::basic_ostream<C,T>& (*pfn)( std::basic_ostream<C,T>&) )
         const )( std::basic_ostream<C,T>& ) { return pfn; }
   std::basic_ios<C,T>& (*
   operator()( std::basic_ios<C,T>& (*pfn)( std::basic_ios<C,T>& ) )
         const )( std::basic_ios<C,T>& ) { return pfn; }
   std::ios_base& (*
   operator()( std::ios_base& (*pfn)( std::ios_base& ) )
         const )( std::ios_base& ) { return pfn; }
};
template <class C, class T>
ManipMaker<C,T> 
make_manip( std::basic_ios<C,T>& ) { return ManipMaker<C,T>(); }

//////////////////////////////////////////////////////////////////////
// STL conversions
//////////////////////////////////////////////////////////////////////
// Note that these are template functions, not functoids.  I'm lazy.  

namespace impl {
template <class Op>
class Xstl1 : public c_fun_type<typename Op::argument_type,
                       typename Op::result_type> {
   Op f;
public:
   Xstl1( const Op& o ) : f(o) {}
   typename Op::result_type 
   operator()( const typename Op::argument_type& x ) const {
      return f(x);
   }
};
}
template <class Op>
full1<impl::Xstl1<Op> > stl_to_fun1( const Op& o ) {
   return make_full1( impl::Xstl1<Op>(o) );
}

namespace impl {
template <class Op>
class Xstl2 : public c_fun_type<typename Op::first_argument_type,
                       typename Op::second_argument_type,
                       typename Op::result_type> {
   Op f;
public:
   Xstl2( const Op& o ) : f(o) {}
   typename Op::result_type 
   operator()( const typename Op::first_argument_type& x, 
               const typename Op::second_argument_type& y ) const {
      return f(x,y);
   }
};
}
template <class Op>
full2<impl::Xstl2<Op> > stl_to_fun2( const Op& o ) {
   return make_full2(impl::Xstl2<Op>(o));
}

//////////////////////////////////////////////////////////////////////
// monomorphizing conversions
//////////////////////////////////////////////////////////////////////
// Note that these are template functions, not functoids.  I'm lazy.  

namespace impl {
template <class Arg1, class Arg2, class Arg3, class Res, class F>
class XMonomorphicWrapper3 : public c_fun_type<Arg1,Arg2,Arg3,Res> {
   F f;
public:
   XMonomorphicWrapper3( const F& g ) : f(g) {}
   Res operator()( const Arg1& x, const Arg2& y, const Arg3& z ) const {
      return f(x,y,z);
   }
};
}
template <class Arg1, class Arg2, class Arg3, class Res, class F>
full3<impl::XMonomorphicWrapper3<Arg1,Arg2,Arg3,Res,F> > 
monomorphize3( const F& f ) {
   return make_full3(impl::XMonomorphicWrapper3<Arg1,Arg2,Arg3,Res,F>( f ));
}

namespace impl {
template <class Arg1, class Arg2, class Res, class F>
class XMonomorphicWrapper2 : public c_fun_type<Arg1,Arg2,Res> {
   F f;
public:
   XMonomorphicWrapper2( const F& g ) : f(g) {}
   Res operator()( const Arg1& x, const Arg2& y ) const {
      return f(x,y);
   }
};
}
template <class Arg1, class Arg2, class Res, class F>
full2<impl::XMonomorphicWrapper2<Arg1,Arg2,Res,F> > 
monomorphize2( const F& f ) {
   return make_full2(impl::XMonomorphicWrapper2<Arg1,Arg2,Res,F>( f ));
}

namespace impl {
template <class Arg1, class Res, class F>
class XMonomorphicWrapper1 : public c_fun_type<Arg1,Res> {
   F f;
public:
   XMonomorphicWrapper1( const F& g ) : f(g) {}
   Res operator()( const Arg1& x ) const {
      return f(x);
   }
};
}
template <class Arg1, class Res, class F>
full1<impl::XMonomorphicWrapper1<Arg1,Res,F> > monomorphize1( const F& f ) {
   return make_full1( impl::XMonomorphicWrapper1<Arg1,Res,F>( f ) );
}

namespace impl {
template <class Res, class F>
class XMonomorphicWrapper0 : public c_fun_type<Res> {
   F f;
public:
   XMonomorphicWrapper0( const F& g ) : f(g) {}
   Res operator()() const {
      return f();
   }
};
}
template <class Res, class F>
full0<impl::XMonomorphicWrapper0<Res,F> > monomorphize0( const F& f ) {
   return make_full0( impl::XMonomorphicWrapper0<Res,F>( f ) );
}

//////////////////////////////////////////////////////////////////////
// ptr_fun
//////////////////////////////////////////////////////////////////////
// ptr_to_fun is now a functoid -- hurray!

namespace impl {

template <class Result>
class Xptr_to_nullary_function : public c_fun_type<Result> {
    Result (*ptr)();
public:
    explicit Xptr_to_nullary_function(Result (*x)()) : ptr(x) {}
    Result operator()() const { return ptr(); }
};

template <class Arg, class Result>
class Xptr_to_unary_function : public c_fun_type<Arg, Result> {
    Result (*ptr)(Arg);
public:
    explicit Xptr_to_unary_function(Result (*x)(Arg)) : ptr(x) {}
    Result operator()(Arg x) const { return ptr(x); }
};

template <class Arg1, class Arg2, class Result>
class Xptr_to_binary_function : public c_fun_type<Arg1, Arg2, Result> {
    Result (*ptr)(Arg1, Arg2);
public:
    explicit Xptr_to_binary_function(Result (*x)(Arg1, Arg2)) : ptr(x) {}
    Result operator()(Arg1 x, Arg2 y) const { return ptr(x, y); }
};

template <class Arg1, class Arg2, class Arg3, class Result>
class Xptr_to_ternary_function : public c_fun_type<Arg1, Arg2, Arg3, Result> {
    Result (*ptr)(Arg1, Arg2, Arg3);
public:
    explicit 
    Xptr_to_ternary_function(Result (*x)(Arg1, Arg2, Arg3)) : ptr(x) {}
    Result operator()(Arg1 x, Arg2 y, Arg3 z) const { return ptr(x,y,z); }
};

//////////////////////////////////////////////////////////////////////
// Turn member functions into normal functions which take a Receiver*
// (or a smart pointer) as their first (extra) argument.  Note that we 
// disallow reference parameters.
//////////////////////////////////////////////////////////////////////

template <class Arg1, class Arg2, class Arg3, class Result>
class Xptr_to_mem_binary_function {
    Result (Arg1::*ptr)(Arg2,Arg3);
public:
    explicit Xptr_to_mem_binary_function(Result (Arg1::*x)(Arg2,Arg3)) 
       : ptr(x) {}
    template <class P, class Y, class Z> 
    struct sig : public fun_type<Result> {};
    template <class P> 
    Result operator()(const P& x, const Arg2& y, const Arg3& z) const 
       //{ return (x->*ptr)(y,z); }
       { return ((*x).*ptr)(y,z); }
};

template <class Arg1, class Arg2, class Arg3, class Result>
class Xptr_to_const_mem_binary_function {
    Result (Arg1::*ptr)(Arg2,Arg3) const;
public:
    explicit Xptr_to_const_mem_binary_function(
          Result (Arg1::*x)(Arg2,Arg3) const) : ptr(x) {}
    template <class P, class Y, class Z> 
    struct sig : public fun_type<Result> {};
    template <class P> 
    Result operator()(const P& x, const Arg2& y, const Arg3& z) const 
       //{ return (x->*ptr)(y,z); }
       { return ((*x).*ptr)(y,z); }
};

template <class Arg1, class Arg2, class Result>
class Xptr_to_mem_unary_function {
    Result (Arg1::*ptr)(Arg2);
public:
    explicit Xptr_to_mem_unary_function(Result (Arg1::*x)(Arg2)) : ptr(x) {}
    template <class P, class Y> 
    struct sig : public fun_type<Result> {};
    template <class P> 
    Result operator()(const P& x, const Arg2& y) const 
    //{ return (x->*ptr)(y); }
    { return ((*x).*ptr)(y); }
};

template <class Arg1, class Arg2, class Result>
class Xptr_to_const_mem_unary_function {
    Result (Arg1::*ptr)(Arg2) const;
public:
    explicit Xptr_to_const_mem_unary_function(Result (Arg1::*x)(Arg2) const) 
       : ptr(x) {}
    template <class P, class Y> 
    struct sig : public fun_type<Result> {};
    template <class P> 
    Result operator()(const P& x, const Arg2& y) const 
       //{ return (x->*ptr)(y); }
       { return ((*x).*ptr)(y); }
};

template <class Arg1, class Result>
class Xptr_to_mem_nullary_function {
    Result (Arg1::*ptr)();
public:
    explicit Xptr_to_mem_nullary_function(Result (Arg1::*x)()) : ptr(x) {}
    template <class P> 
    struct sig : public fun_type<Result> {};
    template <class P>
    //Result operator()(const P& x) const { return (x->*ptr)(); }
    Result operator()(const P& x) const { return ((*x).*ptr)(); }
};

template <class Arg1, class Result>
class Xptr_to_const_mem_nullary_function {
    Result (Arg1::*ptr)() const;
public:
    explicit Xptr_to_const_mem_nullary_function(Result (Arg1::*x)() const) 
       : ptr(x) {}
    template <class P> 
    struct sig : public fun_type<Result> {};
    template <class P>
    //Result operator()(const P& x) const { return (x->*ptr)(); }
    Result operator()(const P& x) const { return ((*x).*ptr)(); }
};

struct XPtrToFun {
   template <class P> struct sig;

   // non-member functions
   template <class Result>
   struct sig< Result (*)() > : public fun_type<
      full0<Xptr_to_nullary_function<Result> > > {};
   template <class A1, class Result>
   struct sig< Result (*)(A1) > : public fun_type<
      full1<Xptr_to_unary_function<A1,Result> > > {};
   template <class A1, class A2, class Result>
   struct sig< Result (*)(A1,A2) > : public fun_type<
      full2<Xptr_to_binary_function<A1,A2,Result> > > {};
   template <class A1, class A2, class A3, class Result>
   struct sig< Result (*)(A1,A2,A3) > : public fun_type<
      full3<Xptr_to_ternary_function<A1,A2,A3,Result> > > {};

   // member functions
   template <class A1, class A2, class A3, class Result>
   struct sig< Result (A1::*)(A2,A3) > : public fun_type<
      full3<Xptr_to_mem_binary_function<A1, A2, A3, Result> > > {};
   template <class A1, class A2, class A3, class Result>
   struct sig< Result (A1::*)(A2,A3) const > : public fun_type<
      full3<Xptr_to_const_mem_binary_function<A1, A2, A3, Result> > > {};
   template <class A1, class A2, class Result>
   struct sig< Result (A1::*)(A2) > : public fun_type<
      full2<Xptr_to_mem_unary_function<A1, A2, Result> > > {};
   template <class A1, class A2, class Result>
   struct sig< Result (A1::*)(A2) const > : public fun_type<
      full2<Xptr_to_const_mem_unary_function<A1, A2, Result> > > {};
   template <class A1, class Result>
   struct sig< Result (A1::*)() > : public fun_type<
      full1<Xptr_to_mem_nullary_function<A1, Result> > > {};
   template <class A1, class Result>
   struct sig< Result (A1::*)() const > : public fun_type<
      full1<Xptr_to_const_mem_nullary_function<A1, Result> > > {};

   // non-member functions
   template <class Result>
   inline full0<Xptr_to_nullary_function<Result> >
   operator()(Result (*x)()) const {
     return make_full0( Xptr_to_nullary_function<Result>(x) );
   }
   template <class A, class Result>
   inline full1<Xptr_to_unary_function<A, Result> >
   operator()(Result (*x)(A)) const {
     return make_full1( Xptr_to_unary_function<A, Result>(x) );
   }
   template <class A1, class A2, class Result>
   inline full2<Xptr_to_binary_function<A1, A2, Result> >
   operator()(Result (*x)(A1, A2)) const {
     return make_full2( Xptr_to_binary_function<A1, A2, Result>(x) );
   }
   template <class A1, class A2, class A3, class Result>
   inline full3<Xptr_to_ternary_function<A1, A2, A3, Result> >
   operator()(Result (*x)(A1, A2, A3)) const {
     return make_full3( Xptr_to_ternary_function<A1,A2,A3,Result>(x) );
   }

   // member functions
   template <class A1, class A2, class A3, class Result>
   inline full3<Xptr_to_mem_binary_function<A1, A2, A3, Result> >
   operator()(Result (A1::*x)(A2,A3)) const {
     return make_full3(
               Xptr_to_mem_binary_function<A1, A2, A3, Result>(x) );
   }
   template <class A1, class A2, class A3, class Result>
   inline full3<Xptr_to_const_mem_binary_function<A1, A2, A3, Result> >
   operator()(Result (A1::*x)(A2,A3) const) const {
     return make_full3(
        Xptr_to_const_mem_binary_function<A1, A2, A3, Result>(x) );
   }
   template <class A1, class A2, class Result>
   inline full2<Xptr_to_mem_unary_function<A1, A2, Result> >
   operator()(Result (A1::*x)(A2)) const {
     return make_full2( Xptr_to_mem_unary_function<A1, A2, Result>(x) );
   }
   template <class A1, class A2, class Result>
   inline full2<Xptr_to_const_mem_unary_function<A1, A2, Result> >
   operator()(Result (A1::*x)(A2) const) const {
     return make_full2( 
       Xptr_to_const_mem_unary_function<A1, A2, Result>(x) );
   }
   template <class A1, class Result>
   inline full1<Xptr_to_mem_nullary_function<A1, Result> >
   operator()(Result (A1::*x)()) const {
     return make_full1( Xptr_to_mem_nullary_function<A1, Result>(x) );
   }
   template <class A1, class Result>
   inline full1<Xptr_to_const_mem_nullary_function<A1, Result> >
   operator()(Result (A1::*x)() const) const {
     return make_full1( Xptr_to_const_mem_nullary_function<A1, Result>(x) );
   }
};
   
}  // end namespace impl
typedef full1<impl::XPtrToFun> ptr_to_fun_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN ptr_to_fun_type ptr_to_fun;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// funify is an identity for FullNs, but calls ptr_to_fun otherwise
//////////////////////////////////////////////////////////////////////

namespace impl {
struct XFunify {
   template <class P> struct sig : public fun_type<
      typename RT<ptr_to_fun_type,P>::result_type> {};
   template <class F> struct sig< full0<F> > 
   : public fun_type< full0<F> > {};
   template <class F> struct sig< full1<F> > 
   : public fun_type< full1<F> > {};
   template <class F> struct sig< full2<F> > 
   : public fun_type< full2<F> > {};
   template <class F> struct sig< full3<F> > 
   : public fun_type< full3<F> > {};

   template <class P>
   typename sig<P>::result_type
   operator()( const P& p ) const { return ptr_to_fun(p); }
   template <class F>
   typename sig<full0<F> >::result_type
   operator()( const full0<F>& f ) const { return f; }
   template <class F>
   typename sig<full1<F> >::result_type
   operator()( const full1<F>& f ) const { return f; }
   template <class F>
   typename sig<full2<F> >::result_type
   operator()( const full2<F>& f ) const { return f; }
   template <class F>
   typename sig<full3<F> >::result_type
   operator()( const full3<F>& f ) const { return f; }
};
}
typedef full1<impl::XFunify> funify_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN funify_type funify;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// ++ and -- operators as functoids
//////////////////////////////////////////////////////////////////////

namespace impl {
struct XInc {
    template <class T>
    struct sig : public fun_type<T> {};
   
    template <class T>
    T operator()(const T& x) const { T y = x; return ++y; }
};
}
typedef full1<impl::XInc> inc_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN inc_type inc;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XDec {
    template <class T>
    struct sig : public fun_type<T> {};
   
    template <class T>
    T operator()(const T& x) const { T y = x; return --y; }
};
}
typedef full1<impl::XDec> dec_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN dec_type dec;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

// Note: always1 and never1 were deprecated and have been removed.

} // end namespace fcpp
} // end namespace boost

#endif
