// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_MONAD_HPP
#define BOOST_FCPP_MONAD_HPP

namespace boost {
namespace fcpp {

template <class rep> struct ThisTypeIsNotAnInstanceOfAnInferrableMonad {};
template <class T> struct MonadError { inline static void error() {} };
template <class T> 
struct MonadError<ThisTypeIsNotAnInstanceOfAnInferrableMonad<T> > {};
template <class rep> struct monad_traits_specializer {
   typedef ThisTypeIsNotAnInstanceOfAnInferrableMonad<rep> Monad;
};
template <class rep> struct MonadTraits {
   typedef typename monad_traits_specializer<rep>::Monad Monad;
   inline static void ensure_is_instance_of_monad() {
      MonadError<Monad>::error();
   }
};

//////////////////////////////////////////////////////////////////////
// If a monad has a type constructor which is not a type synonym, then
// we call it an "inferrable monad".  
// For any monad M, you can conjure up one of its functoids via, e.g. 
//    unit_m<M>()     bind_m<M>()
// whereas for an inferrable monad, you can just refer to e.g.
//    bind
// which will use the monad_traits_specializer to infer "M" from "M a".
//
// (Section 4.3.2 in the Haskell Report explains that an instance of a
// type class's type constructor cannot be a type synonym.)

namespace impl {
   template <class Monad>
   struct XUnitM {
      template <class A> struct sig : public fun_type<
         typename Monad::unit_type::template sig<A>::result_type> {};

      template <class A>
      typename sig<A>::result_type
      operator()( const A& a ) const {
         return Monad::unit(a);
      }
   };
}
template <class Monad> full1<impl::XUnitM<Monad> > unit_m()
{ return make_full1( impl::XUnitM<Monad>() ); }
template <class Monad> struct unit_m_type
{ typedef full1<impl::XUnitM<Monad> > type; };

namespace impl {
   template <class Monad>
   struct XBindM {
      template <class M, class K> struct sig : public fun_type<
         typename Monad::bind_type::template sig<M,K>::result_type> {};
   
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         return Monad::bind(m,k);
      }
   };
}
template <class Monad> full2<impl::XBindM<Monad> > bind_m()
{ return make_full2( impl::XBindM<Monad>() ); }
template <class Monad> struct bind_m_type 
{ typedef full2<impl::XBindM<Monad> > type; };

namespace impl {
   template <class Monad>
   struct XBindM_ {
      template <class M, class K> struct sig : public fun_type<
         typename Monad::bind_type::template sig<M,typename RT<konst_type,K>
            ::result_type>::result_type> {};
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         return Monad::bind( m, konst(k) );
      }
   };
}
template <class Monad> full2<impl::XBindM_<Monad> > bind_m_()
{ return make_full2( impl::XBindM_<Monad>() ); }
template <class Monad> struct bind_m_x_type 
{ typedef full2<impl::XBindM_<Monad> > type; };

template <class Monad> typename Monad::zero_type zero_m()
{ return Monad::zero; }
template <class Monad> struct zero_m_type 
{ typedef typename Monad::zero_type type; };

namespace impl {
   template <class Monad>
   struct XPlusM {
      template <class MA, class MA2> struct sig : public fun_type<
         typename Monad::plus_type::template sig<MA,MA2>::result_type> {};
   
      template <class MA, class MA2>
      typename sig<MA,MA2>::result_type
      operator()( const MA& x, const MA2& y ) const {
         return Monad::plus(x,y);
      }
   };
}
template <class Monad> full2<impl::XPlusM<Monad> > plus_m()
{ return make_full2( impl::XPlusM<Monad>() ); }
template <class Monad> struct plus_m_type 
{ typedef full2<impl::XPlusM<Monad> > type; };

namespace impl {
   template <bool b> struct BindError {};
   template <> struct BindError<true> { static inline void error() {} };
   template <class M, class K>
   struct BindHelper {
      typedef typename MonadTraits<M>::Monad MonadA;
      typedef typename MonadA::template unrep<M>::type A;
      typedef typename MonadA::bind_type::template sig<M,K>::result_type 
         result_type;
      typedef typename MonadTraits<result_type>::Monad MonadB;
      inline static void ensure_m_and_k_are_in_the_same_monad_instance() {
         BindError<boost::is_same<MonadA,MonadB>::value>::error();
      }
   };
   struct XBind {
      template <class M, class K> struct sig 
      : public fun_type<typename BindHelper<M,K>::result_type> {};
   
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         BindHelper<M,K>::ensure_m_and_k_are_in_the_same_monad_instance();
         typedef typename BindHelper<M,K>::MonadA MM;
         return MM::bind(m,k);
      }
   };
}
typedef full2<impl::XBind> bind_type;
BOOST_FCPP_MAYBE_EXTERN bind_type bind;

namespace impl {
   struct XBind_ {
      template <class M, class K> struct sig : public fun_type<
         typename RT<bind_type,M,typename RT<
         konst_type,K>::result_type>::result_type> {};
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         return bind( m, konst(k) );
      }
   };
}
typedef full2<impl::XBind_> bind_x_type;
BOOST_FCPP_MAYBE_EXTERN bind_x_type bind_;

namespace impl {
   template <class Monad>
   struct XMapM {
      template <class F, class M> struct sig : fun_type<
         typename RT<typename bind_m_type<Monad>::type,M,typename LE<
            LAM<LV<1>,CALL<typename unit_m_type<Monad>::type,
            CALL<F,LV<1> > > > >::type>::result_type> {};
      template <class F, class M>
      typename sig<F,M>::result_type
      operator()( const F& f, const M& m ) const {
         lambda_var<1> A;
         return bind_m<Monad>()( m, lambda(A)[ unit_m<Monad>()[f[A]] ] );
      }
   };
}
template <class Monad> full2<impl::XMapM<Monad> > map_m()
{ return make_full2( impl::XMapM<Monad>() ); }
template <class Monad> struct map_m_type 
{ typedef full2<impl::XMapM<Monad> > type; };

// If you can't beat 'em, join_m_type! :) :) :)
// This comment was much funnier before global-search-and-replace
// changed JoinM to join_m_type.  :)
namespace impl {
   template <class Monad>
   struct XJoinM {
      template <class Z> struct sig : public fun_type<typename 
         RT<typename bind_m_type<Monad>::type,Z,id_type>::result_type> {};
      template <class Z>
      typename sig<Z>::result_type
      operator()( const Z& z ) const {
         return bind_m<Monad>()( z, id );
      }
   };
}
template <class Monad> full1<impl::XJoinM<Monad> > join_m()
{ return make_full1( impl::XJoinM<Monad>() ); }
template <class Monad> struct join_m_type 
{ typedef full1<impl::XJoinM<Monad> > type; };

//////////////////////////////////////////////////////////////////////

namespace lambda_impl {

// Gets is not really an LE, but making it appear so makes op,() and
// LEify work properly automagically.  So we fake it to make the
// implementation easier.
namespace exp {
   template <int i, class LEa> struct Gets : public LEBase {
      static const int my_lv = i;
      LEa exp; Gets( const LEa& e ) : exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
         EnsureLE<LEa>::go(); 
#endif
      };
   };
}
template <int i, class E>
exp::Gets<i,typename LEify<E>::type>
operator<=( const lambda_var<i>&, const E& e ) {
   return exp::Gets<i,typename LEify<E>::type>( LEify<E>::go(e) );
}

//////////////////////////////////////////////////////////////////////
// Next on the agenda...
//    do_m[ LE ]                 ==>  LE
//    do_m[ LE, stmts ]          ==>  bind_[ LE, do_m[stmts] ]
//    do_m[ X<=LE, stmts ]  ==>  bind[ LE, lambda(X)[ do_m[stmts] ] ]
// "do_m" is an unfortunate name, as everything else monad-ly I've
// written uses the 'M' suffix to say "I'm a template, pass me a monad
// as a parameter, please", whereas this isn't true here.  But I still
// like "do_m" better than "do_", and just "do" is a C++ keyword.

struct DoM_Lambdaoid {
   template <class Dummy, class FinalLE, class Stuff> struct Helper2;
   template <class D, class FinalLE>
   struct Helper2<D,FinalLE,NIL> {
      typedef FinalLE Result;
      static inline Result go( const FinalLE& le, const NIL& )
      { return le; }
   };
   template <class D, class FinalLE, class LEa, class Rest>
   struct Helper2<D,FinalLE,CONS<LEa,Rest> > {
      typedef typename LE<CALL<bind_x_type,LEa,FinalLE> >::type Inner;
      typedef typename Helper2<D,Inner,Rest>::Result Result;
      static inline Result go( const FinalLE& le, const CONS<LEa,Rest>& rest ) 
      { return Helper2<D,Inner,Rest>::go( bind_[ rest.head, le ], rest.tail ); }
   };
   template <class D, class FinalLE, int i, class LEa, class Rest>
   struct Helper2<D,FinalLE,CONS<exp::Gets<i,LEa>,Rest> > {
      typedef typename LE<CALL<bind_type,LEa,LAM<LV<i>,FinalLE> > >::type 
         Inner;
      typedef typename Helper2<D,Inner,Rest>::Result Result;
      static inline Result 
      go( const FinalLE& le, const CONS<exp::Gets<i,LEa>,Rest>& rest ) { 
         lambda_var<i> X;
         return Helper2<D,Inner,Rest>::go( 
            bind[ rest.head.exp, lambda(X)[ le ] ],    rest.tail ); 
      }
   };

   template <class Dummy, class X> struct Helper;
   template <class D, class LEa, class Rest>
   struct Helper<D,CONS<LEa,Rest> > {
      typedef typename Helper2<D,LEa,Rest>::Result Result;
      static inline Result go( const CONS<LEa,Rest>& l ) 
      { return Helper2<D,LEa,Rest>::go( l.head, l.tail ); }
   };

   template <class X> struct RT 
   { typedef typename DoM_Lambdaoid::Helper<DUMMY,typename
      LEListify<X>::type>::Result type; };

   template <class X>
   typename RT<X>::type
   operator[]( const X& x ) const {
      typedef typename LEListify<X>::type XP;
      return Helper<DUMMY,XP>::go( LEListify<X>::go(x) );
   }
};

template <class A, class B=void, class C=void, class D=void> struct DOM;
template <int i, class E> struct GETS;

template <int i, class E>
struct LE< GETS<i,E> > {
   typedef exp::Gets<i,typename LEify<typename LE<E>::type>::type> type;
};

template <class A, class B, class C, class D> 
struct LE< DOM<A,B,C,D> > {
   typedef typename LET_LEListify<A,B,C,D>::type L;
   typedef typename DoM_Lambdaoid::template RT<L>::type type;
};
template <class A, class B, class C>
struct LE< DOM<A,B,C,void> > {
   typedef typename LET_LEListify<A,B,C>::type L;
   typedef typename DoM_Lambdaoid::template RT<L>::type type;
};
template <class A, class B>
struct LE< DOM<A,B,void,void> > {
   typedef typename LET_LEListify<A,B>::type L;
   typedef typename DoM_Lambdaoid::template RT<L>::type type;
};
template <class A>
struct LE< DOM<A,void,void,void> > {
   typedef typename LET_LEListify<A>::type L;
   typedef typename DoM_Lambdaoid::template RT<L>::type type;
};

//////////////////////////////////////////////////////////////////////
// Finally, the mother-of-all-syntactic-sugar: comprehensions!
//    comp_m<M>()[ LE ]
//       ==>  unit<M>()[ LE ]
//    comp_m<M>()[ LE | LEa, stmts ]
//       ==>  bind_m_[ LEa, comp<M>()[LE|stmts] ]
//    comp_m<M>()[ LE | X<=LEa, stmts ]
//       ==>  bind_m[ LEa, lambda(X)[ comp<M>()[LE|stmts] ] ]
//    comp_m<M>()[ LE | guard[LEa], stmts ]
//       ==>  if1[ LEa, comp<M>()[LE|stmts], zero<M>() ]
// The vert (|) will be a bit of a pain, as it binds tighter than the
// comma (but thankfully less tightly that <=), and so I'm gonna have to
// make it return a VertPair or something as a new "exp" type which
// operator,() can LEListify, and then have comp_m unwrap it on the back
// end.  Hmm, and just to ensure that we don't accidentally call C++'s
// built-in operator|(), I think I'll outlaw the first form of comp_m
// above, and require that there always be a vert and a right-hand-side
// in user calls.  Yup.

namespace exp {
   template <class LHS, class RHS>
   struct VertPair : public LEBase {
      LHS lhs;
      RHS rhs;
      VertPair( const LHS& l, const RHS& r ) : lhs(l), rhs(r) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
         EnsureLE<LHS>::go(); 
         EnsureLE<RHS>::go(); 
#endif
      }
   };
   template <class LHS, class RHS>
   VertPair<typename LEify<LHS>::type,typename LEify<RHS>::type> 
   operator|( const LHS& l, const RHS& r ) {
      return VertPair<typename LEify<LHS>::type,typename LEify<RHS>::type>
         ( LEify<LHS>::go(l), LEify<RHS>::go(r) );
   }
   template <class BFLE> struct Guard : public LEBase { 
      BFLE fun; 
      Guard(const BFLE&f) : fun(f) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
         EnsureLE<BFLE>::go(); 
#endif
      } 
   };
   template <class T> struct CompCast : public LEBase { };
}

struct Guard_Lambdaoid {
   template <class BF>
   exp::Guard<typename LEify<BF>::type> operator[]( const BF& f ) const {
      return exp::Guard<typename LEify<BF>::type>( LEify<BF>::go(f) );
   }
};

template <class Monad>
struct comp_m {
   // Dig walks all the way down the list and gets to the LHS of the
   // VertPair at the end, and returns a _reference_ to it.  This is
   // especially happy because it should translate to a no-op.
   template <class Dummy, class L> struct Dig;
   template <class D, class LHS, class RHS> 
   struct Dig<D,CONS<exp::VertPair<LHS,RHS>,NIL> > {
      typedef const LHS& Result;
      typedef LHS type;
      static inline Result go( const CONS<exp::VertPair<LHS,RHS>,NIL>& l )
      { return l.head.lhs; }
   };
   template <class D, class E, class Rest> 
   struct Dig<D,CONS<E,Rest> > {
      typedef typename Dig<D,Rest>::Result Result;
      typedef typename Dig<D,Rest>::type type;
      static inline Result go( const CONS<E,Rest>& l )
      { return Dig<D,Rest>::go( l.tail ); }
   };

   template <class FinalLE, class Stuff> struct Help2;
   template <class FinalLE>
   struct Help2<FinalLE,NIL> {
      typedef FinalLE Result;
      static inline Result go( const FinalLE& le, const NIL& ) 
      { return le; }
   };
   template <class FinalLE, class LHS, class RHS> 
   struct Help2<FinalLE,CONS<exp::VertPair<LHS,RHS>,NIL> > {
      typedef typename Help2<FinalLE,CONS<RHS,NIL> >::Result Result;
      static inline Result 
      go( const FinalLE& le, const CONS<exp::VertPair<LHS,RHS>,NIL>& x ) {
         return Help2<FinalLE,CONS<RHS,NIL> >::go
            ( le, CONS<RHS,NIL>(x.head.rhs) );
      }
   };
   template <class FinalLE, class LEa, class Rest> 
   struct Help2<FinalLE,CONS<LEa,Rest> > {
      typedef typename LE<CALL<typename bind_m_x_type<Monad>::type,LEa,
         FinalLE> >::type NextLE;
      typedef typename Help2<NextLE,Rest>::Result Result;
      static inline Result 
      go( const FinalLE& le, const CONS<LEa,Rest>& x ) {
         return Help2<NextLE,Rest>::go(
            bind_m_<Monad>()[ x.head, le ], x.tail );
      }
   };
   template <class FinalLE, int i, class LEa, class Rest> 
   struct Help2<FinalLE,CONS<exp::Gets<i,LEa>,Rest> > {
      typedef typename LE<CALL<typename bind_m_type<Monad>::type,LEa,
         LAM<LV<i>,FinalLE> > >::type NextLE;
      typedef typename Help2<NextLE,Rest>::Result Result;
      static inline Result 
      go( const FinalLE& le, const CONS<exp::Gets<i,LEa>,Rest>& x ) {
         lambda_var<i> X;
         return Help2<NextLE,Rest>::go(
            bind_m<Monad>()[ x.head.exp, lambda(X)[ le ] ], x.tail );
      }
   };
   template <class FinalLE, class BF, class Rest> 
   struct Help2<FinalLE,CONS<exp::Guard<BF>,Rest> > {
      typedef typename LE<IF1<BF,FinalLE,
         typename zero_m_type<Monad>::type> >::type NextLE;
      typedef typename Help2<NextLE,Rest>::Result Result;
      static inline Result 
      go( const FinalLE& le, const CONS<exp::Guard<BF>,Rest>& x ) {
         return Help2<NextLE,Rest>::go(
            if1[ x.head.fun, le, zero_m<Monad>() ], x.tail );
      }
   };
   template <class FinalLE, class CT, class Rest> 
   struct Help2<FinalLE,CONS<exp::CompCast<CT>,Rest> > {
      typedef typename LE<CALL<typename construct1_type<CT>::type,
         FinalLE> >::type NextLE;
      typedef typename Help2<NextLE,Rest>::Result Result;
      static inline Result 
      go( const FinalLE& le, const CONS<exp::CompCast<CT>,Rest>& x ) {
         return Help2<NextLE,Rest>::go( construct1<CT>()[le], x.tail );
      }
   };

   template <class L> struct Helper {
      typedef typename Dig<DUMMY,L>::type DigLE;
      typedef typename LE<CALL<typename unit_m_type<Monad>::type,
         DigLE> >::type FinalLE;
      typedef typename Help2<FinalLE,L>::Result Result;
      static inline Result go( const L& l ) { 
         return Help2<FinalLE,L>::go( unit_m<Monad>()[Dig<DUMMY,L>::go(l)], l );
      }
   };

   template <class X> struct RT {
      typedef typename LEListify<X>::type XP;
      typedef typename Helper<XP>::Result type;
   };
   template <class X>
   typename RT<X>::type
   operator[]( const X& x ) const {
      return Helper<typename RT<X>::XP>::go( LEListify<X>::go(x) );
   }
};

template <class T> struct GUARD;
template <class T> struct COMP_CAST;

template <class T> struct LE< GUARD<T> > {
   typedef exp::Guard< typename LEify<typename LE<T>::type>::type > type;
};
template <class T> struct LE< COMP_CAST<T> > {
   typedef exp::CompCast< typename LE<T>::type > type;
};

template <class A, class B, class C, class D=void, class E=void,
          class F=void> struct COMP {};

template <class M, class LHS, class A>
struct LE< COMP<M,LHS,A,void,void,void> > {
   typedef typename LEify<typename LE<LHS>::type>::type LHST;
   typedef typename LEify<typename LE<A>::type>::type AT;
   typedef exp::VertPair<LHST,AT> T;
   typedef typename LET_LEListify<T>::type TE;
   typedef typename comp_m<M>::template RT<TE>::type type;
};
template <class M, class LHS, class A, class B>
struct LE< COMP<M,LHS,A,B,void,void> > {
   typedef typename LEify<typename LE<LHS>::type>::type LHST;
   typedef typename LEify<typename LE<A>::type>::type AT;
   typedef exp::VertPair<LHST,AT> T;
   typedef typename LET_LEListify<T,B>::type TE;
   typedef typename comp_m<M>::template RT<TE>::type type;
};
template <class M, class LHS, class A, class B, class C>
struct LE< COMP<M,LHS,A,B,C,void> > {
   typedef typename LEify<typename LE<LHS>::type>::type LHST;
   typedef typename LEify<typename LE<A>::type>::type AT;
   typedef exp::VertPair<LHST,AT> T;
   typedef typename LET_LEListify<T,B,C>::type TE;
   typedef typename comp_m<M>::template RT<TE>::type type;
};
template <class M, class LHS, class A, class B, class C, class D>
struct LE< COMP<M,LHS,A,B,C,D> > {
   typedef typename LEify<typename LE<LHS>::type>::type LHST;
   typedef typename LEify<typename LE<A>::type>::type AT;
   typedef exp::VertPair<LHST,AT> T;
   typedef typename LET_LEListify<T,B,C,D>::type TE;
   typedef typename comp_m<M>::template RT<TE>::type type;
};

} // end namespace lambda_impl

using lambda_impl::DOM;
using lambda_impl::GETS;
using lambda_impl::GUARD;
using lambda_impl::COMP_CAST;
using lambda_impl::COMP;
using lambda_impl::comp_m;   // a template type name

BOOST_FCPP_MAYBE_EXTERN lambda_impl::DoM_Lambdaoid do_m;
BOOST_FCPP_MAYBE_EXTERN lambda_impl::Guard_Lambdaoid guard;

template <class T>
lambda_impl::exp::CompCast<T> comp_cast() {
   return lambda_impl::exp::CompCast<T>();
}

//////////////////////////////////////////////////////////////////////

// fcomp_m<Monad>()[ comp ]  means  lambda()[ comp_m<Monad>()[ comp ] ]()
// FIX no good way to compute return type (see below)
template <class Monad>
struct fcomp_m {
   template <class K>
   typename RT<typename LE<LAM<
      // FIX cheating, need good user way to say (or have just be sugar)
      typename comp_m<Monad>::template RT<K>::type
   > >::type>::result_type
   operator[]( const K& comprehension ) {
      return lambda()[ comp_m<Monad>()[comprehension] ]();
   }
};

//////////////////////////////////////////////////////////////////////
// Monad concepts
//////////////////////////////////////////////////////////////////////
// Here we outline the C++ concepts necessary for monads to work in the
// framework.
/*
   concept Monad {
      // unit :: a -> m a
      // bind :: m a -> ( a -> m b ) -> m b
      typedef unit_type;           // a full functoid type
      static unit_type unit;
      typedef bind_type;           // a full functoid type
      static bind_type bind;
   }
   //////////////////////////////
   // Inferrable Monads let you use bind() instead of bind_m<Monad>()(), and
   // are also the only Monads that work with do-notation (do_m).
   concept InferrableMonad models Monad {
      // type constructor: pass an 'a', get back an 'M a'
      template <class A>  struct rep   { typedef type; };
      // type deconstructor: pass an 'M a', get back an 'a'
      template <class MA> struct unrep { typedef type; };
   }
   and also
   template <> struct monad_traits_specializer< RepType > {
      typedef Monad;   // the monad concept class
   };
   //////////////////////////////
   // Monad zeros enable comprehension guards to work
   concept MonadWithZero models Monad {
      // zero :: m a
      typedef zero_type;           // a value type
      static zero_type zero;
   }
   //////////////////////////////
   // Some monads have pluses
   concept MonadWithPlus models Monad {
      // plus :: m a -> m a -> m a
      typedef plus_type;           // a full functoid type
      static plus_type plus;
   }
*/
//////////////////////////////////////////////////////////////////////

struct list_m {
   template <class A> struct rep { typedef list<A> type; };
   template <class MA> struct unrep { typedef typename MA::value_type type; };

   struct XUnit {
      template <class A> struct sig : public fun_type<odd_list<A> > {};
      template <class A>                 
      typename sig<A>::result_type operator()( const A& x ) const {
         return cons(x,NIL);             
      }
   };
   typedef full1<XUnit> unit_type;
   static unit_type unit;

   struct XBind {
      template <class M, class K> struct sig : public fun_type<
         typename RT<concat_type,typename RT<map_type,K,M>
         ::result_type>::result_type> {};
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         return concat( map(k,m) );
      }
   };
   typedef full2<XBind> bind_type;
   static bind_type bind;

   typedef a_unique_type_for_nil zero_type;
   static zero_type zero;

   typedef cat_type plus_type;   // cat is monad plus for lists
   static plus_type plus;
   // FIX THIS maybe: map/join/bind_ could be implemented here as
   // well.  default implementations could be inherited.  Not clear if
   // it gains anything, since you can just call the "free function"
   // versions of them, which have defaults based on unit/bind.
};
BOOST_FCPP_MAYBE_DEFINE(list_m::unit_type list_m::unit;)
BOOST_FCPP_MAYBE_DEFINE(list_m::bind_type list_m::bind;)
BOOST_FCPP_MAYBE_DEFINE(list_m::zero_type list_m::zero;)
BOOST_FCPP_MAYBE_DEFINE(list_m::plus_type list_m::plus;)

template <class T> struct monad_traits_specializer<list<T> > {
   typedef list_m Monad;
};
template <class T> struct monad_traits_specializer<odd_list<T> > {
   typedef list_m Monad;
};

//////////////////////////////////////////////////////////////////////

struct maybe_m {
   template <class A>  struct rep   { typedef maybe<A> type; };
   template <class MA> struct unrep { typedef typename MA::value_type type; };

   typedef just_type unit_type;
   static unit_type unit;

   struct XBind {
      template <class M, class K> struct sig : public fun_type<
         typename RT<K,typename M::value_type>::result_type> {};
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         if( m.is_nothing() )
            return NOTHING;
         else
            return k( m.value() );
      }
   };
   typedef full2<XBind> bind_type;
   static bind_type bind;

   typedef a_unique_type_for_nothing zero_type;
   static zero_type zero;

   struct XPlus {
      template <class MA, class MA2> 
      struct sig : public fun_type<MA> {};
      template <class MA, class MA2>
      typename sig<MA,MA2>::result_type
      operator()( const MA& x, const MA2& y ) const {
         if( x.is_nothing() )
            return y;
         return x;
      }
   };
   typedef full2<XPlus> plus_type;
   static plus_type plus;
};
BOOST_FCPP_MAYBE_DEFINE(maybe_m::unit_type maybe_m::unit;)
BOOST_FCPP_MAYBE_DEFINE(maybe_m::bind_type maybe_m::bind;)
BOOST_FCPP_MAYBE_DEFINE(maybe_m::zero_type maybe_m::zero;)
BOOST_FCPP_MAYBE_DEFINE(maybe_m::plus_type maybe_m::plus;)

template <class T> struct monad_traits_specializer<maybe<T> > {
   typedef maybe_m Monad;
};

//////////////////////////////////////////////////////////////////////

struct identity_m {
   // M a = a

   typedef id_type unit_type;
   static unit_type unit;

   struct XBind {
      template <class M, class K> struct sig : public fun_type<
         typename RT<K,M>::result_type> {};
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         return k(m);
      }
   };
   typedef full2<XBind> bind_type;
   static bind_type bind;
};
BOOST_FCPP_MAYBE_DEFINE(identity_m::unit_type identity_m::unit;)
BOOST_FCPP_MAYBE_DEFINE(identity_m::bind_type identity_m::bind;)

//////////////////////////////////////////////////////////////////////

template <class State>
struct state_m {
   // M a =  State -> (a,State)

   struct XUnit {
      template <class A> struct sig : public fun_type<
         typename RT<make_pair_type,A>::result_type> {};
      template <class A>
      typename sig<A>::result_type
      operator()( const A& a ) const { return make_pair(a); }
   };
   typedef full1<XUnit> unit_type;
   static unit_type unit;

   struct XBind {
      template <class M, class K> struct sig : public fun_type<
         typename LE<LAM<LV<2>,LET<BIND<1,CALL<M,LV<2> > >,
                                       CALL<CALL<K,CALL<fst_type,LV<1> > >,
                                            CALL<snd_type,LV<1> > > 
                                      > > >::type > {};
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         // \s0 -> let (a,s1) = m s0
         //            (b,s2) = k a s1
         //        in  (b,s2)
         lambda_var<1> P;
         lambda_var<2> s0;
         return lambda(s0)[ let[ P == m[s0] ].in[
                            k[fst[P]][snd[P]] ] ];
      }
   };
   typedef full2<XBind> bind_type;
   static bind_type bind;

   static typename 
   LE< LAM<LV<1>,CALL<CALL<make_pair_type,empty_type>,State> > >::type
   assign( const State& sp ) {
      lambda_var<1> s;
      return lambda(s)[ make_pair[empty][sp] ];
   }

   static typename LE< LAM<LV<1>,CALL<make_pair_type,LV<1>,LV<1> > > >::type
   fetch() {
      lambda_var<1> s;
      return lambda(s)[ make_pair[s,s] ];
   }
};
template <class State> typename state_m<State>::unit_type state_m<State>::unit;
template <class State> typename state_m<State>::bind_type state_m<State>::bind;

//////////////////////////////////////////////////////////////////////
// lift_m, lift_m2, lift_m3   (as in Haskell)

namespace impl {
template <class Monad>
struct XLiftM {
   template <class F> struct sig : public fun_type<typename 
   LE<LAM<LV<1>,COMP<Monad,CALL<F,LV<2> >,GETS<2,LV<1> > > > >::type> {};
   template <class F>
   typename sig<F>::result_type operator()( const F& f ) const {
      lambda_var<1> X;
      lambda_var<2> X_;
      return lambda(X)[ comp_m<Monad>()[ f[X_] | X_ <= X ] ];
   }
};
}
template <class Monad> full1<impl::XLiftM<Monad> > lift_m()
{ return make_full1( impl::XLiftM<Monad>() ); }
template <class Monad> struct lift_m_type 
{ typedef full1<impl::XLiftM<Monad> > type; };

namespace impl {
template <class Monad>
struct XLiftM2 {
   template <class F> struct sig : public fun_type<typename 
      LE<LAM<LV<1>,LV<3>,COMP<Monad,CALL<F,LV<2>,LV<4> >,GETS<2,LV<1>
      >,GETS<4,LV<3> > > > >::type> {};
   template <class F>
   typename sig<F>::result_type operator()( const F& f ) const {
      lambda_var<1> X;
      lambda_var<2> X_;
      lambda_var<3> Y;
      lambda_var<4> Y_;
      return lambda(X,Y)[ comp_m<Monad>()[ f[X_,Y_] | X_ <= X, Y_ <= Y ] ];
   }
};
}
template <class Monad> full1<impl::XLiftM2<Monad> > lift_m2()
{ return make_full1( impl::XLiftM2<Monad>() ); }
template <class Monad> struct lift_m2_type
{ typedef full1<impl::XLiftM2<Monad> > type; };

namespace impl {
template <class Monad>
struct XLiftM3 {
   template <class F> struct sig : public fun_type<typename 
      LE<LAM<LV<1>,LV<3>,LV<5>,COMP<Monad,CALL<F,LV<2>,LV<4>,LV<6> 
      >,GETS<2,LV<1> >,GETS<4,LV<3> >,GETS<6,LV<5> > > > >::type> {};
   template <class F>
   typename sig<F>::result_type operator()( const F& f ) const {
      lambda_var<1> X;
      lambda_var<2> X_;
      lambda_var<3> Y;
      lambda_var<4> Y_;
      lambda_var<5> Z;
      lambda_var<6> Z_;
      return lambda(X,Y,Z)[ comp_m<Monad>()[ 
            f[X_,Y_,Z_] | X_ <= X, Y_ <= Y, Z_ <= Z ] ];
   }
};
}
template <class Monad> full1<impl::XLiftM3<Monad> > lift_m3()
{ return make_full1( impl::XLiftM3<Monad>() ); }
template <class Monad> struct lift_m3_type
{ typedef full1<impl::XLiftM3<Monad> > type; };

//////////////////////////////////////////////////////////////////////
// The by_need monad, for "lazifying" computations

template <class T>
class ByNeedImpl : boost::noncopyable {
   mutable RefCountType refC;

public:
   typedef 
      typename boost::type_with_alignment<boost::alignment_of<T>::value>::type
      xval_type;
private:
   union { mutable xval_type val; unsigned char xdummy[sizeof(T)]; };
   mutable fun0<T> fxn;
   mutable bool val_is_valid;  // FIX THIS: can optimize this away into fxn

   const T& value() const { 
      return *static_cast<const T*>(static_cast<const void*>(&val)); 
   }
   T& value() { 
      return *static_cast<T*>(static_cast<void*>(&val));
   }
   void init( const T& x ) const {
      new (static_cast<void*>(&val)) T(x);
   }

   static T dummyT() 
      { throw fcpp_exception("Used invalid fxn in ByNeedImpl"); }
   static fun0<T> dummy() { 
      static fun0<T> f( ptr_to_fun(&ByNeedImpl::dummyT) ); 
      return f;
   }
public:
   typedef T value_type;
   ByNeedImpl( const T& x ) 
     : refC(0), fxn( dummy() ), val_is_valid(true)  { init(x); }
   ByNeedImpl( fun0<T> f )  
     : refC(0), fxn( f       ), val_is_valid(false) {}
   const T& force() const {
      if( val_is_valid )
         return value();
      else {
         init( fxn() );
         val_is_valid = true;
         return value();
      }
   }
   template <class X>
   friend void intrusive_ptr_add_ref( const ByNeedImpl<X>* p );
   template <class X>
   friend void intrusive_ptr_release( const ByNeedImpl<X>* p );
};
template <class T> 
void intrusive_ptr_add_ref( const ByNeedImpl<T>* p ) {
   ++ (p->refC);
}  
template <class T> 
void intrusive_ptr_release( const ByNeedImpl<T>* p ) {
   if( !--(p->refC) ) delete p;
}  

template <class T>
class by_need {
   boost::intrusive_ptr<ByNeedImpl<T> > ref;
public:
   typedef T value_type;
   by_need( const T& x ) : ref( new ByNeedImpl<T>(x) ) {}
   by_need( fun0<T> f ) : ref( new ByNeedImpl<T>(f) ) {}
   // default copy constructor and assignment operator are fine
   const T& force() const {
      return ref->force();
   }
};

namespace impl {
struct XBForce {
   template <class BT> struct sig : fun_type<typename BT::value_type> {};
   template <class T>
   const T& operator()( const by_need<T>& bt ) const {
      return bt.force();
   }
};
}
typedef full1<impl::XBForce> b_force_type;
BOOST_FCPP_MAYBE_EXTERN b_force_type b_force;

namespace impl {
struct XBDelay {
   template <class T> struct sig : fun_type<by_need<T> > {};
   template <class T>
   by_need<T> operator()( const T& x ) const {
      return by_need<T>( x );
   }
};
}
typedef full1<impl::XBDelay> b_delay_type;
BOOST_FCPP_MAYBE_EXTERN b_delay_type b_delay;

// b_lift lazily lifts a function's results into the by_need monad:
//    b_lift :: (a -> b) -> a -> by_need b
//    b_lift(f)(args) == by_need<T>( lambda()[ f[args] ] )
namespace impl{
template <class F>
class XBLifter0 : public c_fun_type<by_need<typename RT<F>::result_type> > {
   F f;
public:
   XBLifter0( const F& ff ) : f(ff) {}
   by_need<typename RT<F>::result_type>
   operator()() const {
      return by_need<typename RT<F>::result_type>( lambda()[ f[_*_] ] );
   }
};
template <class F>
class XBLifter1 {
   F f;
public:
   XBLifter1( const F& ff ) : f(ff) {}
   template <class X> struct sig 
      : public fun_type<by_need<typename RT<F,X>::result_type> > {};
   template <class X>
   typename sig<X>::result_type
   operator()( const X& x ) const {
      return typename sig<X>::result_type( lambda()[ f[x] ] );
   }
};
template <class F>
class XBLifter2 {
   F f;
public:
   XBLifter2( const F& ff ) : f(ff) {}
   template <class X, class Y> struct sig 
      : public fun_type<by_need<typename RT<F,X,Y>::result_type> > {};
   template <class X, class Y>
   typename sig<X,Y>::result_type
   operator()( const X& x, const Y& y ) const {
      return typename sig<X,Y>::result_type( lambda()[ f[x][y] ] );
   }
};
template <class F>
class XBLifter3 {
   F f;
public:
   XBLifter3( const F& ff ) : f(ff) {}
   template <class X, class Y, class Z> struct sig 
      : public fun_type<by_need<typename RT<F,X,Y,Z>::result_type> > {};
   template <class X, class Y, class Z>
   typename sig<X,Y,Z>::result_type
   operator()( const X& x, const Y& y, const Z& z ) const {
      return typename sig<X,Y,Z>::result_type( lambda()[ f[x][y][z] ] );
   }
};
class XBLift {
   template <int i, class F> struct Helper;
   template <class F>
   struct Helper<0,F> {
      typedef full0<XBLifter0<F> > Result;
      static Result go( const F& f )
      { return make_full0( XBLifter0<F>(f) ); }
   };
   template <class F>
   struct Helper<1,F> {
      typedef full1<XBLifter1<F> > Result;
      static Result go( const F& f )
      { return make_full1( XBLifter1<F>(f) ); }
   };
   template <class F>
   struct Helper<2,F> {
      typedef full2<XBLifter2<F> > Result;
      static Result go( const F& f )
      { return make_full2( XBLifter2<F>(f) ); }
   };
   template <class F>
   struct Helper<3,F> {
      typedef full3<XBLifter3<F> > Result;
      static Result go( const F& f )
      { return make_full3( XBLifter3<F>(f) ); }
   };
public:
   template <class F> struct sig : public fun_type<
      typename Helper<functoid_traits<F>::max_args,F>::Result> {};
   template <class F>
   typename sig<F>::result_type operator()( const F& f ) const {
      return Helper<functoid_traits<F>::max_args,F>::go( f );
   }
};
}
typedef full1<impl::XBLift> b_lift_type;
BOOST_FCPP_MAYBE_EXTERN b_lift_type b_lift;

struct by_need_m {
   template <class A> struct rep { typedef by_need<A> type; };
   template <class MA> struct unrep { typedef typename MA::value_type type; };

   struct XUnit {
      template <class A> struct sig : public fun_type<by_need<A> > {};
      template <class A>
      typename sig<A>::result_type operator()( const A& x ) const {
         return by_need<A>( x );
      }
   };
   typedef full1<XUnit> unit_type;
   static unit_type unit;

   struct XBind {
      template <class M, class K> struct sig : public fun_type<
         typename RT<K,typename M::value_type>::result_type > {};
      template <class M, class K>
      typename sig<M,K>::result_type 
      operator()( const M& m, const K& k ) const {
         typedef typename sig<M,K>::result_type Res;
         return Res( lambda()[ b_force[ k[ b_force[ m ] ] ] ] );
         //return k( b_force( m ) );
      }
   };
   typedef full2<XBind> bind_type;
   static bind_type bind;
};
BOOST_FCPP_MAYBE_DEFINE(by_need_m::unit_type by_need_m::unit;)
BOOST_FCPP_MAYBE_DEFINE(by_need_m::bind_type by_need_m::bind;)
   
template <class T> struct monad_traits_specializer<by_need<T> > {
   typedef by_need_m Monad;
};

//////////////////////////////////////////////////////////////////////
// strict_list

struct strict_list_m {
   template <class A> struct rep { typedef strict_list<A> type; };
   template <class MA> struct unrep { typedef typename MA::value_type type; };

   struct XUnit {
      template <class A> struct sig : public fun_type<strict_list<A> > {};
      template <class A>                 
      typename sig<A>::result_type operator()( const A& x ) const {
         return cons(x,strict_list<A>());             
      }
   };
   typedef full1<XUnit> unit_type;
   static unit_type unit;

   struct XBind {
      template <class M, class K> struct sig : public fun_type<
         typename RT<concat_type,typename RT<map_type,K,M>
         ::result_type>::result_type> {};
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         return concat( map(k,m) );
      }
   };
   typedef full2<XBind> bind_type;
   static bind_type bind;

   typedef a_unique_type_for_nil zero_type;
   static zero_type zero;

   typedef cat_type plus_type;   // cat is monad plus for lists
   static plus_type plus;
};
BOOST_FCPP_MAYBE_DEFINE(strict_list_m::unit_type strict_list_m::unit;)
BOOST_FCPP_MAYBE_DEFINE(strict_list_m::bind_type strict_list_m::bind;)
BOOST_FCPP_MAYBE_DEFINE(strict_list_m::zero_type strict_list_m::zero;)
BOOST_FCPP_MAYBE_DEFINE(strict_list_m::plus_type strict_list_m::plus;)

} // end namespace fcpp
} // end namespace boost

#endif
