// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

/*
Note that "LE" means "Lambda Expression".

FIX list of things I know I can do better...
 - being more careful about LEs and 'wrapped' LEs (LambdaExp<LE>), so
   that I don't end up unnecessarily wrapping getses, etc.  Indeed,
   design so that comprehension commas make sense...
 - a way to ask for a fresh variable

syntax summary:
   lambda(X,Y)[ LE ] //(a,b)
   let[ X==f[a], Y==g[X,b] ].in[ LExy ] //()     
   letrec[ x==f[x,y], y==g[x,y] ].in[ LExy ] //()
   comp_m<M>()[ LE | LE, x<=LE, guard[LE] ] //()
   do_m[ LE, x<=LE, LE ] //()
   if0,if1,if2

Everything centers around LEs; whether LEs have free vars can be key to
whether they're callable; explicit lambdas are a special case of
callable LEs where certain bindings on params happen.
*/

#ifndef BOOST_FCPP_LAMBDA_HPP
#define BOOST_FCPP_LAMBDA_HPP

#include "full.hpp"

#ifdef BOOST_FCPP_ENABLE_LAMBDA
namespace boost {
namespace fcpp {
namespace lambda_impl {

//////////////////////////////////////////////////////////////////////
// We occasionally need a dummy type.  Sometimes this is used to help 
// partially specialize templates.  Other times it is used to ensure we
// get compile-time lazy evaluation--we turn a type computation into a
// "thunk" which takes a meaningless template argument.
struct DUMMY;

//////////////////////////////////////////////////////////////////////
// Literate error messages look nicer when emitted as compiler
// diagnostics when they're not deeply nested inside classes, so we move
// them all out to the 'top' level of this namespace here.
//////////////////////////////////////////////////////////////////////
template <class Actual, class Given, bool b>
struct TheActualTypeOfTheLambdaExpressionIsNotConvertibleToItsGivenType;
template <class Actual, class Given>
struct TheActualTypeOfTheLambdaExpressionIsNotConvertibleToItsGivenType
<Actual,Given,true> { typedef int Error; };

template <class FreeVars>
struct YouCannotInvokeALambdaContainingFreeVars {};

template <class Dummy, bool b>
struct TheTypeSpecYouHaveGivenIsIncompatibleWithItsLE;
template <class Dummy>
struct TheTypeSpecYouHaveGivenIsIncompatibleWithItsLE<Dummy,true>
{ typedef int Error; };

template <class EE, bool b> struct IfExpressionMustHaveTypeConvertibleToBool;
template <class EE> 
struct IfExpressionMustHaveTypeConvertibleToBool<EE,true> 
{ typedef EE type; };

template <class TT, class FF, bool b> struct
ActualTypeOfFalseBranchMustBeImplicitlyConvertibleToTypeOfTrueBranch;
template <class TT, class FF> struct
ActualTypeOfFalseBranchMustBeImplicitlyConvertibleToTypeOfTrueBranch
<TT,FF,true> { typedef TT Error; };

template <class FF, class TT, bool b> struct
ActualTypeOfTrueBranchMustBeImplicitlyConvertibleToTypeOfFalseBranch;
template <class FF, class TT> struct
ActualTypeOfTrueBranchMustBeImplicitlyConvertibleToTypeOfFalseBranch
<FF,TT,true> { typedef FF Error; };

template <class TT, class FF> struct TrueAndFalseBranchOfIfMustHaveSameType;
template <class TT> struct TrueAndFalseBranchOfIfMustHaveSameType<TT,TT>
{ typedef TT type; };

template <bool b> 
struct YouCannotPassTheSameLambdaVarTo_lambda_MoreThanOnce;
template <> 
struct YouCannotPassTheSameLambdaVarTo_lambda_MoreThanOnce<false>
{ static inline void go() {} };

//////////////////////////////////////////////////////////////////////
// Useful helpers
//////////////////////////////////////////////////////////////////////

// some quick forward decls
namespace exp { 
   template <int i> class lambda_var; 
   template <class T> class Value; 
}
using exp::lambda_var;

template <class T>
struct ThunkifyType {
   struct Result {
      template <class Dummy> struct Go {
         typedef T type;
      };
   };
};

//////////////////////////////////////////////////////////////////////
// Environment stuff
//////////////////////////////////////////////////////////////////////

// I seem to recall that
//   ET - Environment Thunk
//   BE - Binding Environment
//   TE - type Environment
// Yes.  See pre_lambda.hpp for a little more explanation.

struct NIL_ET {
   template <class Dummy> struct Go {
      typedef NIL TE;
      typedef NIL BE;
   };
};

template <class OldET, class LVList>
struct EraseLVsFromET {
   struct Result {
      template <class Dummy> struct Go {
         typedef typename OldET::template Go<Dummy>::TE TE1;
         typedef typename OldET::template Go<Dummy>::BE BE1;

         struct BPred {
            template <class BEP> struct Go;
            template <int i, class LEa> struct Go<BEPair<i,LEa> > { 
               static const bool value = 
                  !(Contains<LVList,lambda_var<i> >::value);
            };
         };
         typedef Filter<BPred,BE1> BF;
         typedef typename BF::Result BE;
         static inline BE go( const BE1& x ) { return BF::go(x); }
   
         struct TPred {
            template <class TEP> struct Go;
            template <int i, class T> struct Go<TEPair<i,T> > { 
               static const bool value = 
                  !(Contains<LVList,lambda_var<i> >::value);
            };
         };
         typedef Filter<TPred,TE1> TF;
         typedef typename TF::Result TE;
      };
   };
};

template <class ET1, int i, class X>
struct ETUpdateX {
   struct Result {
      template <class Dummy> struct Go {
         typedef CONS<lambda_var<i>,NIL> BoundVars;
         typedef typename ET1::template Go<Dummy>::BE BE1;
         typedef typename EraseLVsFromET<ET1,BoundVars>::Result ET2;
         typedef typename ET2::template Go<Dummy>::TE TE2;
         typedef typename ET2::template Go<Dummy>::BE BE2;
         typedef typename exp::Value<X> LEX; typedef BEPair<i,LEX> BEX;
         typedef CONS<BEX,BE2> BE;
         typedef typename ThunkifyType<X>::Result XTT;
         typedef TEPair<i,XTT> TEX;
         typedef CONS<TEX,TE2> TE;
         static inline BE go( const BE1& be, const X& x ) {
            return BE( BEX(LEX(x)), ET2::template Go<Dummy>::go(be) );
         }
      };
   };
};

template <class ET1, int i, class X, int j, class Y>
struct ETUpdateXY {
   struct Result {
      template <class Dummy> struct Go {
         typedef CONS<lambda_var<i>,CONS<lambda_var<j>,NIL> > BoundVars;
         typedef typename ET1::template Go<Dummy>::BE BE1;
         typedef typename EraseLVsFromET<ET1,BoundVars>::Result ET2;
         typedef typename ET2::template Go<Dummy>::TE TE2;
         typedef typename ET2::template Go<Dummy>::BE BE2;
         typedef typename exp::Value<X> LEX; typedef BEPair<i,LEX> BEX;
         typedef typename exp::Value<Y> LEY; typedef BEPair<j,LEY> BEY;
         typedef CONS<BEY,BE2> YBE;
         typedef CONS<BEX,YBE> BE;
         typedef typename ThunkifyType<X>::Result XTT;
         typedef TEPair<i,XTT> TEX;
         typedef typename ThunkifyType<Y>::Result YTT;
         typedef TEPair<j,YTT> TEY;
         typedef CONS<TEX,CONS<TEY,TE2> > TE;
         static inline BE go( const BE1& be, const X& x, const Y& y ) {
            return BE( BEX(LEX(x)), YBE( BEY(LEY(y)), 
                       ET2::template Go<Dummy>::go(be) ) );
         }
      };
   };
};

template <class ET1, int i, class X, int j, class Y, int k, class Z>
struct ETUpdateXYZ {
   struct Result {
      template <class Dummy> struct Go {
         typedef CONS<lambda_var<i>,CONS<lambda_var<j>,
                    CONS<lambda_var<k>,NIL> > > BoundVars;
         typedef typename ET1::template Go<Dummy>::BE BE1;
         typedef typename EraseLVsFromET<ET1,BoundVars>::Result ET2;
         typedef typename ET2::template Go<Dummy>::TE TE2;
         typedef typename ET2::template Go<Dummy>::BE BE2;
         typedef typename exp::Value<X> LEX; typedef BEPair<i,LEX> BEX;
         typedef typename exp::Value<Y> LEY; typedef BEPair<j,LEY> BEY;
         typedef typename exp::Value<Z> LEZ; typedef BEPair<k,LEZ> BEZ;
         typedef CONS<BEZ,BE2> ZBE;
         typedef CONS<BEY,ZBE> YBE;
         typedef CONS<BEX,YBE> BE;
         typedef typename ThunkifyType<X>::Result XTT;
         typedef TEPair<i,XTT> TEX;
         typedef typename ThunkifyType<Y>::Result YTT;
         typedef TEPair<j,YTT> TEY;
         typedef typename ThunkifyType<Z>::Result ZTT;
         typedef TEPair<k,ZTT> TEZ;
         typedef CONS<TEX,CONS<TEY,CONS<TEZ,TE2> > > TE;
         static inline BE go( const BE1& be, const X& x, const Y& y,
                              const Z& z ) {
            return BE( BEX(LEX(x)), YBE( BEY(LEY(y)), ZBE( BEZ(LEZ(z)),
                       ET2::template Go<Dummy>::go(be) ) ) );
         }
      };
   };
};

//////////////////////////////////////////////////////////////////////
// The lambda expression types, all defined right together here
// in one big batch
//////////////////////////////////////////////////////////////////////

namespace exp {

template <class X>
class LambdaExp : public LEBase {
   X x;
public:
   LambdaExp( const X& xx ) : x(xx) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<X>::go(); 
#endif
   }

   typedef typename X::FreeVars FreeVars;
   template <class EnvThunk> struct MyType {
      typedef typename X::template MyType<EnvThunk>::TypeThunk TypeThunk;
   };
   template <class EnvThunk> struct RecheckType {
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      typedef typename X::template RecheckType<EnvThunk>::Ok Ok;
#else
      typedef int Ok;
#endif
   };

   template <class EnvThunk>
   typename MyType<EnvThunk>::TypeThunk::template Go<DUMMY>::type 
   eval( const typename EnvThunk::template Go<DUMMY>::BE& be ) const {
      return x.template eval<EnvThunk>( be );
   }

   typedef LambdaExp This;
   template <class A> typename BracketCallable<This,A>::Result
   operator[]( const A& a ) const 
   { return BracketCallable<This,A>::go( *this, a ); }
};

template <class T>
class Value : public LEBase {
   T data;
public:
   Value( const T& x ) : data(x) {}

   typedef NIL FreeVars;
   template <class EnvThunk> struct MyType { 
      struct TypeThunk {
         template <class Dummy> struct Go {
            typedef T type; 
         };
      };
   };
   template <class EnvThunk> struct RecheckType { typedef int Ok; };
   template <class EnvThunk>
   typename MyType<EnvThunk>::TypeThunk::template Go<DUMMY>::type 
   eval( const typename EnvThunk::template Go<DUMMY>::BE& ) const
   { return data; }

   typedef Value This;
   template <class A> typename BracketCallable<This,A>::Result
   operator[]( const A& a ) const 
   { return BracketCallable<This,A>::go( *this, a ); }
};

template <int i>
class lambda_var : public LEBase {
   // 'ii' is used to get around what appears to be a g++ bug...
public:
   template <int ii, class TE> struct Lookup;
   template <int ii, class T, class Rest> 
   struct Lookup<ii,CONS<TEPair<ii,T>,Rest> >
   { typedef typename T::template Go<DUMMY>::type type; };
   template <int ii, class TEP, class Rest> 
   struct Lookup<ii,CONS<TEP,Rest> >
   { typedef typename Lookup<ii,Rest>::type type; };

private:
   template <int ii, class Result, class EnvThunk, class BE, class WBE> 
   struct Find;
   template <int ii, class R, class ET, class LEa, class Rest, class WBE>
   struct Find<ii,R,ET,CONS<BEPair<ii,LEa>,Rest>,WBE> { 
      static inline R go( const CONS<BEPair<ii,LEa>,Rest>& be,
                          const WBE& wbe ) 
      { return be.head.value.template eval<ET>(wbe); }
   };
   template <int ii, class R, class ET, class BEP, class Rest, class WBE> 
   struct Find<ii,R,ET,CONS<BEP,Rest>,WBE> { 
      static inline R go( const CONS<BEP,Rest>& be, const WBE& wbe ) {
         return Find<ii,R,ET,Rest,WBE>::go( be.tail, wbe );
      }
   };
public:
   typedef CONS<lambda_var,NIL> FreeVars;
   template <class EnvThunk> struct MyType { 
      struct TypeThunk {
         template <class Dummy> struct Go {
            typedef typename EnvThunk::template Go<Dummy>::TE TE;
            typedef typename Lookup<i,TE>::type type;
         };
      };
   };
   template <class EnvThunk> struct RecheckType { typedef int Ok; };
   template <class EnvThunk>
   typename MyType<EnvThunk>::TypeThunk::template Go<DUMMY>::type 
   eval( const typename EnvThunk::template Go<DUMMY>::BE& be ) const {
      typedef 
         typename MyType<EnvThunk>::TypeThunk::template Go<DUMMY>::type Result;
      typedef typename EnvThunk::template Go<DUMMY>::BE BE;
      return Find<i,Result,EnvThunk,BE,BE>::go(be,be);
   }

   typedef lambda_var This;
   template <class A> typename BracketCallable<This,A>::Result
   operator[]( const A& a ) const 
   { return BracketCallable<This,A>::go( *this, a ); }
};

template <class Fun, class Args>
class Call : public LEBase {
   Fun fun;  Args args;
public:  
   template <class ET, class F, class A> struct TypeHelper;
   template <class ET, class F> 
   struct TypeHelper<ET,F,NIL> {
      typedef typename RT<F>::result_type type;
   };
   template <class ET, class F, class A1> 
   struct TypeHelper<ET,F,CONS<A1,NIL> > {
      typedef typename A1::template MyType<ET>::TypeThunk XTT;
      typedef typename XTT::template Go<DUMMY>::type X;
      typedef typename RT<F,X>::result_type type;
   };
   template <class ET, class F, class A1, class A2> 
   struct TypeHelper<ET,F,CONS<A2,CONS<A1,NIL> > > {
      typedef typename A1::template MyType<ET>::TypeThunk XTT;
      typedef typename XTT::template Go<DUMMY>::type X;
      typedef typename A2::template MyType<ET>::TypeThunk YTT;
      typedef typename YTT::template Go<DUMMY>::type Y;
      typedef typename RT<F,X,Y>::result_type type;
   };
   template <class ET, class F, class A1, class A2, class A3> 
   struct TypeHelper<ET,F,CONS<A3,CONS<A2,CONS<A1,NIL> > > > {
      typedef typename A1::template MyType<ET>::TypeThunk XTT;
      typedef typename XTT::template Go<DUMMY>::type X;
      typedef typename A2::template MyType<ET>::TypeThunk YTT;
      typedef typename YTT::template Go<DUMMY>::type Y;
      typedef typename A3::template MyType<ET>::TypeThunk ZTT;
      typedef typename ZTT::template Go<DUMMY>::type Z;
      typedef typename RT<F,X,Y,Z>::result_type type;
   };

   template <class ET, class LEL> struct CheckHelper;
   template <class ET> 
   struct CheckHelper<ET,NIL> { typedef int Result; };
   template <class ET, class H, class T> 
   struct CheckHelper<ET,CONS<H,T> > { 
      typedef typename AlwaysFirst<typename H::template RecheckType<ET>::Ok,
         typename CheckHelper<ET,T>::Result>::type Result; 
   };

   template <class Result, class ET, class BE, class Args_> struct EvalHelp;
   template <class R, class ET, class BE> 
   struct EvalHelp<R,ET,BE,NIL> {
      static inline R go( const Fun& f, const NIL&, const BE& be ) {
         return (f.template eval<ET>(be))();
      }
   };
   template <class R, class ET, class BE, class A1> 
   struct EvalHelp<R,ET,BE,CONS<A1,NIL> > {
      static inline R go( const Fun& f, const CONS<A1,NIL>& a, const BE& be ) {
         return (f.template eval<ET>(be))(a.head.template eval<ET>(be));
      }
   };
   template <class R, class ET, class BE, class A1, class A2> 
   struct EvalHelp<R,ET,BE,CONS<A2,CONS<A1,NIL> > > {
      static inline R go( const Fun& f, const CONS<A2,CONS<A1,NIL> >& a, 
                          const BE& be ) {
         return (f.template eval<ET>(be))(
            a.tail.head.template eval<ET>(be),
            a.head.template eval<ET>(be));
      }
   };
   template <class R, class ET, class BE, class A1, class A2, class A3> 
   struct EvalHelp<R,ET,BE,CONS<A3,CONS<A2,CONS<A1,NIL> > > > {
      static inline R go( const Fun& f, 
                          const CONS<A3,CONS<A2,CONS<A1,NIL> > >& a, 
                          const BE& be ) {
         return (f.template eval<ET>(be))(
            a.tail.tail.head.template eval<ET>(be),
            a.tail.head.template eval<ET>(be),
            a.head.template eval<ET>(be));
      }
   };
   
   struct FoldrOp {
      template <class LEa, class FVList> struct Go {
         typedef typename AppendList<typename LEa::FreeVars,FVList>::Result 
            Result;
      };
   };
   typedef typename foldr_type<FoldrOp,NIL,CONS<Fun,Args> >::Result
      AccumulatedFreeVars;

   Call( const Fun& f, const Args& a ) : fun(f), args(a) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<Fun>::go(); 
      EnsureLEList<Args>::go(); 
#endif
   }

   typedef typename RemoveDuplicates<AccumulatedFreeVars>::Result FreeVars;
   template <class EnvThunk> struct MyType { 
      struct TypeThunk {
         template <class Dummy> struct Go {
            typedef typename Fun::template MyType<EnvThunk>::TypeThunk FTT;
            typedef typename FTT::template Go<Dummy>::type F;
            typedef typename TypeHelper<EnvThunk,F,Args>::type type; 
         };
      };
   };
   template <class EnvThunk> struct RecheckType { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      typedef typename CheckHelper<EnvThunk,CONS<Fun,Args> >::Result Ok; 
#else
      typedef int Ok;
#endif
   };
   template <class ET>
   typename MyType<ET>::TypeThunk::template Go<DUMMY>::type 
   eval( const typename ET::template Go<DUMMY>::BE& be ) const {
      typedef typename ET::template Go<DUMMY>::BE BE;
      typedef typename MyType<ET>::TypeThunk::template Go<DUMMY>::type Result;
      return EvalHelp<Result,ET,BE,Args>::go(fun,args,be);
   }

   typedef Call This;
   template <class A> typename BracketCallable<This,A>::Result
   operator[]( const A& a ) const 
   { return BracketCallable<This,A>::go( *this, a ); }
};

// This is a weird special class only BindingEnvExp below uses...
template <class RealET, class LEa, class FV, class BE1, class B>
class AddEnvLE : public LEBase {
   LEa exp;
   BE1 be1;
   B b;
public:
   AddEnvLE( const LEa& e, const BE1& x, const B& y ) : exp(e), be1(x), b(y) {}

   typedef FV FreeVars;

   template <class EnvThunk> struct MyType { 
      struct TypeThunk {
         template <class Dummy> struct Go {
            typedef typename LEa::template MyType<RealET>::TypeThunk LETT;
            typedef typename LETT::template Go<Dummy>::type type; 
         };
      };
   };
   template <class EnvThunk> struct RecheckType { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      typedef typename LEa::template RecheckType<RealET>::Ok Ok; 
#else
      typedef int Ok;
#endif
   };
   template <class EnvThunk>
   typename MyType<RealET>::TypeThunk::template Go<DUMMY>::type 
   eval( const typename EnvThunk::template Go<DUMMY>::BE& ) const { 
      return exp.template eval<RealET>( RealET::template Go<DUMMY>::go(be1,b) );
   }

   typedef AddEnvLE This;
   template <class A> typename BracketCallable<This,A>::Result
   operator[]( const A& a ) const 
   { return BracketCallable<This,A>::go( *this, a ); }
};

template <class B, class LEx>
class BindingEnvExp : public LEBase {    // letrec semantics
   B binders;
   LEx exp;

   // We use a dummy template parameter for convenience; without it,
   // we'd run into the rule that says you can't fully specialize a
   // template at non-namespace scope.
   template <class BinderList, class Dummy> struct AccumFree;
   template <class D> struct AccumFree<NIL,D> { typedef NIL Result; };
   template <int i, class LEa, class Rest, class D> 
   struct AccumFree<CONS<Binder<i,LEa>,Rest>,D>
   { typedef typename AppendList<typename LEa::FreeVars,
        typename AccumFree<Rest,D>::Result>::Result Result; };
   
   // See comment above.
   template <class BinderList, class Dummy> struct AccumBound;
   template <class D> struct AccumBound<NIL,D> { typedef NIL Result; };
   template <int i, class LEa, class Rest, class D> 
   struct AccumBound<CONS<Binder<i,LEa>,Rest>,D>
   { typedef CONS<lambda_var<i>,typename AccumBound<Rest,D>::Result> Result; };
   
public:
   typedef typename AccumBound<B,DUMMY>::Result NewlyBoundVars;
   typedef typename AccumFree<B,DUMMY>::Result FreeVarsInBinders;

   // In order to compute the environment for "exp", we must follow
   // these steps:
   //  - Call the outer env E1
   //  - foreach lv in lhs of binders, erase lv-entries from E1 => E2
   //  - foreach binder in binders, add <binder.lhs,AddEnvLE<E3>(binder.rhs)> 
   //    BE/TE pair to E2 => E3
   // Now exp should be evaled in E3.

   template <class Inner, class ET3>
   struct UnusualTTHelper {
      template <class DumDum> struct Go {
         typedef typename Inner::template MyType<ET3>::TypeThunk ITT;
         typedef typename ITT::template Go<DumDum>::type type;
      };
   };

   // Note the trick; we take the very result we are computing (ET3) as
   // a template parameter!  Ha ha!  Darn I'm clever.  :)
   template <class ET2, class ET3, class BinderList, class NBV, 
             class BE1, class BB> struct Env3FromEnv2;
   template <class ET2, class ET3, class NBV, class BE1, class BB> 
   struct Env3FromEnv2<ET2,ET3,NIL,NBV,BE1,BB> {
      struct Result {
         template <class Dummy> struct Go {
            typedef typename ET2::template Go<Dummy>::TE TE;
            typedef typename ET2::template Go<Dummy>::BE BE;
            static inline BE go( const BE& x, const NIL&, const BE1&,
                                 const BB& ) { return x; }
         };
      };
   };
   template <class ET2, class ET3, int i, class LEa, class Rest, class NBV, 
             class BE1, class BB> 
   struct Env3FromEnv2<ET2,ET3,CONS<Binder<i,LEa>,Rest>,NBV,BE1,BB> {
      struct Result {
         template <class Dummy> struct Go {
            typedef typename ET2::template Go<Dummy>::BE BE2;
            typedef typename 
               Env3FromEnv2<ET2,ET3,Rest,NBV,BE1,BB>::Result Recurse;
            typedef typename 
               ListDifference<typename LEa::FreeVars,NBV>::Result LEFV;
            // Inner will capture the letrec environment (by value)
            typedef AddEnvLE<ET3,LEa,LEFV,BE1,BB> Inner;
            typedef BEPair<i,Inner> BEP;
            typedef CONS<BEP,typename Recurse::template Go<Dummy>::BE> BE;
            static inline BE go( const BE2& be, 
                                 const CONS<Binder<i,LEa>,Rest>& binders,
                                 const BE1& be1, const BB& b ) { 
               return BE( BEP(Inner(binders.head.exp,be1,b)),
                  Recurse::template Go<Dummy>::go(be,binders.tail,be1,b) );
            }
            typedef UnusualTTHelper<Inner,ET3> TT;
            typedef TEPair<i,TT> TEP;
            typedef CONS<TEP,typename Recurse::template Go<Dummy>::TE> TE;
         };
      };
   };

   template <class OrigET>
   struct MakeNewET {
      struct Result {
         template <class Dummy> struct Go {
            typedef typename OrigET::template Go<Dummy>::BE BE1;
            typedef typename EraseLVsFromET<OrigET,NewlyBoundVars>::Result E2;
            // Here is the trick to tie the recursive knot:
            typedef typename 
               Env3FromEnv2<E2,Result,B,NewlyBoundVars,BE1,B>::Result E3;
            typedef typename OrigET::template Go<Dummy>::BE BE1;
            typedef typename E3::template Go<Dummy>::BE BE;
            typedef typename E3::template Go<Dummy>::TE TE;
            static inline BE go( const BE1& be, const B& binders ) { 
               return E3::template Go<Dummy>::go( 
                  E2::template Go<Dummy>::go(be), binders, be, binders );
            }
         };
      };
   };
      
private:
   template <class NewET, class BinderList> struct CheckHelp;
   template <class ET> struct CheckHelp<ET,NIL> { typedef int Result; };
   template <class ET, int i, class LEa, class Rest> 
   struct CheckHelp<ET,CONS<Binder<i,LEa>,Rest> > { 
      typedef typename LEa::template RecheckType<ET>::Ok ThisOne;
      typedef typename AlwaysFirst<ThisOne,
         typename CheckHelp<ET,Rest>::Result>::type Result;
   };
public:
   BindingEnvExp( const B& g, const LEx& e ) : binders(g), exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureBinderList<B>::go(); 
      EnsureLE<LEx>::go(); 
#endif
   }

   typedef typename RemoveDuplicates<typename ListDifference<
      typename AppendList<FreeVarsInBinders,typename LEx::FreeVars>::Result,
      NewlyBoundVars>::Result>::Result FreeVars;

   template <class ET> struct MyType { 
      typedef typename LEx::template MyType<typename
         MakeNewET<ET>::Result>::TypeThunk TypeThunk;
   };

   template <class ET> struct RecheckType { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      typedef typename MakeNewET<ET>::Result NET;
      typedef typename AlwaysFirst<
         typename LEx::template RecheckType<NET>::Ok,
         typename CheckHelp<NET,B>::Result>::type Ok; 
#else
      typedef int Ok;
#endif
   };

   template <class EnvThunk>
   typename MyType<EnvThunk>::TypeThunk::template Go<DUMMY>::type 
   eval( const typename EnvThunk::template Go<DUMMY>::BE& be ) const {
      typedef typename MakeNewET<EnvThunk>::Result NET;
      return exp.template eval<NET>( NET::template Go<DUMMY>::go(be,binders) );
   }

   typedef BindingEnvExp This;
   template <class A> typename BracketCallable<This,A>::Result
   operator[]( const A& a ) const 
   { return BracketCallable<This,A>::go( *this, a ); }
};

template <class LEx, class TBBV>  // To-Be-Bound Vars, in correct order
class LambdaExpWithFreeVars : public LEBase {
   // This class's name isn't great; nearly every kind of lambda
   // expression has free variables. Specifically, this class is about
   // such LEs which are created like so:
   //    lambda(X)[ f[X,Y] ]
   // LEs which, when evaluated, will result in a functoid which
   // captures a particular outer binding environment.

   LEx exp;
public:
   template <class LEa, class ET>
   class Lambda0WithFreeVars : public c_fun_type<typename LEa::template 
            MyType<ET>::TypeThunk::template Go<DUMMY>::type> {
      typedef typename ET::template Go<DUMMY>::BE BE;
      LEa exp;
      BE env;
   public:
      Lambda0WithFreeVars( const LEa& e, const BE& be ) : exp(e), env(be) {}
       
      typename AlwaysFirst<typename LEa::template MyType<ET>::TypeThunk
         ::template Go<DUMMY>::type, typename LEa::template 
         RecheckType<ET>::Ok>::type
      operator()() const {
         return exp.template eval<ET>( env );
      }
   };

   template <class LEa, class ET, int i>
   class Lambda1WithFreeVars {
      typedef typename ET::template Go<DUMMY>::BE BE;
      LEa exp;
      BE env;
      template <class X> struct NewET {
         typedef typename ETUpdateX<ET,i,X>::Result Result;
      };
   public:
      Lambda1WithFreeVars( const LEa& e, const BE& be ) : exp(e), env(be) {}
      template <class X> struct sig : public fun_type<
         typename LEa::template MyType<typename NewET<X>::Result>
         ::TypeThunk::template Go<DUMMY>::type> {};
      template <class X>
      typename sig<X>::result_type
      operator()( const X& x ) const {
         return exp.template eval<typename NewET<X>::Result>
            ( NewET<X>::Result::template Go<DUMMY>::go( env, x ) );
      }
   };

   template <class LEa, class ET, int i, int j>
   class Lambda2WithFreeVars {
      typedef typename ET::template Go<DUMMY>::BE BE;
      LEa exp;
      BE env;
      template <class X, class Y> struct NewET {
         typedef typename ETUpdateXY<ET,i,X,j,Y>::Result Result;
      };
   public:
      Lambda2WithFreeVars( const LEa& e, const BE& be ) : exp(e), env(be) {}
      template <class X, class Y> struct sig : public fun_type<
         typename LEa::template MyType<typename NewET<X,Y>::Result>
         ::TypeThunk::template Go<DUMMY>::type> {};
      template <class X, class Y>
      typename sig<X,Y>::result_type
      operator()( const X& x, const Y& y ) const {
         return exp.template eval<typename NewET<X,Y>::Result>
            ( NewET<X,Y>::Result::template Go<DUMMY>::go( env, x, y ) );
      }
   };

   template <class LEa, class ET, int i, int j, int k>
   class Lambda3WithFreeVars {
      typedef typename ET::template Go<DUMMY>::BE BE;
      LEa exp;
      BE env;
      template <class X, class Y, class Z> struct NewET {
         typedef typename ETUpdateXYZ<ET,i,X,j,Y,k,Z>::Result Result;
      };
   public:
      Lambda3WithFreeVars( const LEa& e, const BE& be ) : exp(e), env(be) {}
      template <class X, class Y, class Z> struct sig : public fun_type<
         typename LEa::template MyType<typename NewET<X,Y,Z>::Result>
         ::TypeThunk::template Go<DUMMY>::type> {};
      template <class X, class Y, class Z>
      typename sig<X,Y,Z>::result_type
      operator()( const X& x, const Y& y, const Z& z ) const {
         return exp.template eval<typename NewET<X,Y,Z>::Result>
            ( NewET<X,Y,Z>::Result::template Go<DUMMY>::go( env, x, y, z ) );
      }
   };

   template <class LEa, class ET, class TBBVars> 
   struct NumBoundVarsHelp;
   template <class LEa, class ET> 
   struct NumBoundVarsHelp<LEa,ET,NIL> {
      typedef Lambda0WithFreeVars<LEa,ET> Lam;
      typedef full0<Lam> Full;
   };
   template <class LEa, class ET, int i> 
   struct NumBoundVarsHelp<LEa,ET,CONS<lambda_var<i>,NIL> > {
      typedef Lambda1WithFreeVars<LEa,ET,i> Lam;
      typedef full1<Lam> Full;
   };
   template <class LEa, class ET, int i, int j> 
   struct NumBoundVarsHelp<LEa,ET,CONS<lambda_var<i>,CONS<lambda_var<j>,
            NIL> > > {
      typedef Lambda2WithFreeVars<LEa,ET,i,j> Lam;
      typedef full2<Lam> Full;
   };
   template <class LEa, class ET, int i, int j, int k> 
   struct NumBoundVarsHelp<LEa,ET,CONS<lambda_var<i>,CONS<lambda_var<j>,
            CONS<lambda_var<k>,NIL> > > > {
      typedef Lambda3WithFreeVars<LEa,ET,i,j,k> Lam;
      typedef full3<Lam> Full;
   };

   LambdaExpWithFreeVars( const LEx& e ) : exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<LEx>::go(); 
#endif
   }

   typedef typename RemoveDuplicates<typename ListDifference<typename 
      LEx::FreeVars,TBBV>::Result>::Result FreeVars;

   template <class ET> struct MyType { 
      typedef typename NumBoundVarsHelp<LEx,ET,TBBV>::Full Full;
      typedef typename ThunkifyType<Full>::Result TypeThunk; 
   };

   template <class ET> struct RecheckType { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      typedef typename LEx::template RecheckType<ET>::Ok Ok; 
#else
      typedef int Ok;
#endif
   };

   template <class EnvThunk>
   typename MyType<EnvThunk>::TypeThunk::template Go<DUMMY>::type 
   eval( const typename EnvThunk::template Go<DUMMY>::BE& be ) const {
      typedef NumBoundVarsHelp<LEx,EnvThunk,TBBV> NBVH; 
      typedef typename NBVH::Lam Lam;
      typedef typename NBVH::Full Full;
      return Full( Lam(exp,be) );
   }

   YouCannotInvokeALambdaContainingFreeVars<FreeVars>
   operator()() const { return 0; }
   template <class X> 
   YouCannotInvokeALambdaContainingFreeVars<FreeVars>
   operator()(const X&) const { return 0; }
   template <class X, class Y> 
   YouCannotInvokeALambdaContainingFreeVars<FreeVars>
   operator()(const X&, const Y&) const { return 0; }
   template <class X, class Y, class Z> 
   YouCannotInvokeALambdaContainingFreeVars<FreeVars>
   operator()(const X&, const Y&, const Z&) const { return 0; }
 
   template <class AA=void, class BB=void, class CC=void> struct sig {
      typedef typename YouCannotInvokeALambdaContainingFreeVars<FreeVars>
         ::AndThusYouShouldNotBeTryingToUseItsSigEither result_type;
   };

   typedef LambdaExpWithFreeVars This;
   template <class A> typename BracketCallable<This,A>::Result
   operator[]( const A& a ) const 
   { return BracketCallable<This,A>::go( *this, a ); }
};

template <class E, class T, class F, IfKind deduction_method>
class IfLE : public LEBase {
   E e;
   T t;
   F f;
public:
   template <class ET, IfKind deduct_method> struct XType;
   template <class ET, IfKind deduct_method> struct XRecheckType;

   // Normal if type deduction
   template <class ET> struct XType<ET,IfNormal> { 
      typedef typename T::template MyType<ET>::TypeThunk TypeThunk;
   };
   template <class ET> struct XRecheckType<ET,IfNormal> { 
      typedef typename E::template MyType<ET>::TypeThunk ETT; 
      typedef typename T::template MyType<ET>::TypeThunk TTT; 
      typedef typename F::template MyType<ET>::TypeThunk FTT; 
      typedef typename ETT::template Go<DUMMY>::type EType;
      typedef typename TTT::template Go<DUMMY>::type TType;
      typedef typename FTT::template Go<DUMMY>::type FType;
      static const bool b = boost::is_convertible<EType,bool>::value;
      typedef typename AlwaysFirst<typename AlwaysFirst<typename AlwaysFirst<
         typename AlwaysFirst<typename
         TrueAndFalseBranchOfIfMustHaveSameType<TType,FType>::type, typename 
         IfExpressionMustHaveTypeConvertibleToBool<EType,b>::type>::type,
         typename E::template RecheckType<ET>::Ok>::type,
         typename T::template RecheckType<ET>::Ok>::type,
         typename F::template RecheckType<ET>::Ok>::type Ok; 
   };

   // type deduction based on true-branch
   template <class ET> struct XType<ET,IfTrue> { 
      typedef typename T::template MyType<ET>::TypeThunk TypeThunk;
   };
   template <class ET> struct XRecheckType<ET,IfTrue> { 
      typedef typename E::template MyType<ET>::TypeThunk ETT; 
      typedef typename T::template MyType<ET>::TypeThunk TTT; 
      typedef typename F::template MyType<ET>::TypeThunk FTT; 
      typedef typename ETT::template Go<DUMMY>::type EType;
      typedef typename TTT::template Go<DUMMY>::type TType;
      typedef typename FTT::template Go<DUMMY>::type FType;
      static const bool bx = boost::is_convertible<EType,bool>::value;
      typedef typename 
         IfExpressionMustHaveTypeConvertibleToBool<EType,bx>::type Foo;
      static const bool b = boost::is_convertible<FType,TType>::value;
      typedef typename
         ActualTypeOfFalseBranchMustBeImplicitlyConvertibleToTypeOfTrueBranch
         <FType,TType,b>::Error Tmp;
      typedef typename AlwaysFirst<typename AlwaysFirst<
         typename AlwaysFirst<typename AlwaysFirst<Tmp,Foo>::type,
         typename E::template RecheckType<ET>::Ok>::type,
         typename T::template RecheckType<ET>::Ok>::type,
         typename F::template RecheckType<ET>::Ok>::type Ok; 
   };

   // type deduction based on false-branch
   template <class ET> struct XType<ET,IfFalse> { 
      typedef typename F::template MyType<ET>::TypeThunk TypeThunk;
   };
   template <class ET> struct XRecheckType<ET,IfFalse> { 
      typedef typename E::template MyType<ET>::TypeThunk ETT; 
      typedef typename T::template MyType<ET>::TypeThunk TTT; 
      typedef typename F::template MyType<ET>::TypeThunk FTT; 
      typedef typename ETT::template Go<DUMMY>::type EType;
      typedef typename TTT::template Go<DUMMY>::type TType;
      typedef typename FTT::template Go<DUMMY>::type FType;
      static const bool bx = boost::is_convertible<EType,bool>::value;
      typedef typename 
         IfExpressionMustHaveTypeConvertibleToBool<EType,bx>::type Foo;
      static const bool b = boost::is_convertible<TType,FType>::value;
      typedef typename
         ActualTypeOfTrueBranchMustBeImplicitlyConvertibleToTypeOfFalseBranch
         <TType,FType,b>::Error Tmp;
      typedef typename AlwaysFirst<typename AlwaysFirst<
         typename AlwaysFirst<typename AlwaysFirst<Tmp,Foo>::type,
         typename E::template RecheckType<ET>::Ok>::type,
         typename T::template RecheckType<ET>::Ok>::type,
         typename F::template RecheckType<ET>::Ok>::type Ok; 
   };

   IfLE( const E& ee, const T& tt, const F& ff ) : e(ee), t(tt), f(ff) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<E>::go(); 
      EnsureLE<T>::go(); 
      EnsureLE<F>::go(); 
#endif
   }

   typedef typename RemoveDuplicates<typename AppendList<typename E::FreeVars, 
      typename AppendList<typename T::FreeVars, typename F::FreeVars>::Result
      >::Result>::Result FreeVars;

   template <class ET> struct MyType 
   { typedef typename XType<ET,deduction_method>::TypeThunk TypeThunk; };

   template <class ET> struct RecheckType { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      typedef typename XRecheckType<ET,deduction_method>::Ok Ok; 
#else
      typedef int Ok;
#endif
   };

   template <class ET>
   typename MyType<ET>::TypeThunk::template Go<DUMMY>::type 
   eval( const typename ET::template Go<DUMMY>::BE& be ) const {
      if( e.template eval<ET>(be) )
         return t.template eval<ET>(be);
      else
         return f.template eval<ET>(be);
   }

   typedef IfLE This;
   template <class A> typename BracketCallable<This,A>::Result
   operator[]( const A& a ) const 
   { return BracketCallable<This,A>::go( *this, a ); }
};

// operator, overloads
//
// Koenig lookup will only find these overloads if one of the arguments
// to comma is an LE (a type defined in this namespace).

// Either the LHS is already a CONS...
template <class H, class T, class RHS>
CONS<typename LEify<RHS>::type,CONS<H,T> >
operator,( const CONS<H,T>& lhs, const RHS& rhs ) {
   return CONS<typename LEify<RHS>::type,CONS<H,T> >
      ( LEify<RHS>::go(rhs), lhs );
}

// ... or it's not
template <class LHS, class RHS>
CONS<typename LEify<RHS>::type,typename LEListify<LHS>::type>
operator,( const LHS& lhs, const RHS& rhs ) {
   return CONS<typename LEify<RHS>::type,typename LEListify<LHS>::type>
      ( LEify<RHS>::go(rhs), LEListify<LHS>::go(lhs) );
}

// This is so we can call it explicitly
template <class LHS, class RHS>
CONS<typename LEify<RHS>::type,typename LEListify<LHS>::type>
explicit_comma( const LHS& lhs, const RHS& rhs ) {
   return CONS<typename LEify<RHS>::type,typename LEListify<LHS>::type>
      ( LEify<RHS>::go(rhs), LEListify<LHS>::go(lhs) );
}

} // end namespace exp

//////////////////////////////////////////////////////////////////////
// lambda() and the functoids that get made when all the vars are bound
// and we make it back out into "C++ space"
//////////////////////////////////////////////////////////////////////

template <class LE>
class Lambda0 : public c_fun_type<typename LE::template
                   MyType<NIL_ET>::TypeThunk::template Go<DUMMY>::type> {
   LE exp;
public:
   Lambda0( const LE& e ) : exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<LE>::go(); 
#endif
   }
   typename LE::template MyType<NIL_ET>::TypeThunk::template Go<DUMMY>::type
   operator()() const {
      return exp.template eval<NIL_ET>( NIL() );
   }
};
template <class LEx, int i>
class Lambda1 {
   LEx exp;
   template <class X> struct NewET 
   { typedef typename ETUpdateX<NIL_ET,i,X>::Result Result; };
public:
   Lambda1( const LEx& e ) : exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<LEx>::go(); 
#endif
   }
   template <class X> struct sig : public fun_type<typename LEx::template 
     MyType<typename NewET<X>::Result>::TypeThunk::template Go<DUMMY>::type> {};
   template <class X>
   typename sig<X>::result_type
   operator()( const X& x ) const {
      typedef typename NewET<X>::Result NET;
      return exp.template eval<NET>( NET::template Go<DUMMY>::go(NIL(),x) );
   }
};
template <class LEx, int i, int j>
class Lambda2 {
   LEx exp;
   template <class X, class Y> struct NewET 
   { typedef typename ETUpdateXY<NIL_ET,i,X,j,Y>::Result Result; };
public:
   Lambda2( const LEx& e ) : exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<LEx>::go(); 
#endif
   }
   template <class X, class Y> struct sig : public fun_type<
      typename LEx::template MyType<typename NewET<X,Y>::Result>
      ::TypeThunk::template Go<DUMMY>::type> {};
   template <class X, class Y>
   typename sig<X,Y>::result_type
   operator()( const X& x, const Y& y ) const {
      typedef typename NewET<X,Y>::Result NET;
      return exp.template eval<NET>( NET::template Go<DUMMY>::go(NIL(),x,y) );
   }
};
template <class LEx, int i, int j, int k>
class Lambda3 {
   LEx exp;
   template <class X, class Y, class Z> struct NewET 
   { typedef typename ETUpdateXYZ<NIL_ET,i,X,j,Y,k,Z>::Result Result; };
public:
   Lambda3( const LEx& e ) : exp(e) { 
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<LEx>::go(); 
#endif
   }
   template <class X, class Y, class Z> struct sig : public fun_type<
      typename LEx::template MyType<typename NewET<X,Y,Z>::Result>
      ::TypeThunk::template Go<DUMMY>::type> {};
   template <class X, class Y, class Z>
   typename sig<X,Y,Z>::result_type
   operator()( const X& x, const Y& y, const Z& z ) const {
      typedef typename NewET<X,Y,Z>::Result NET;
      return exp.template eval<NET>( NET::template Go<DUMMY>::go(NIL(),x,y,z) );
   }
};

// LambdaThingy is the temporary object that lambda() returns which 
// understands operator[] calls.
template <class TBBV>
struct LambdaThingy {
   typedef TBBV VarsThisLambdaBinds;

   template <class LEx, class FV, class TBBVars> struct Help2 { 
      typedef exp::LambdaExpWithFreeVars<LEx,VarsThisLambdaBinds> Result; 
      typedef Result Full;
   };
   template <class LEx> struct Help2<LEx,NIL,NIL> { 
      typedef Lambda0<LEx> Result; 
      typedef full0<Result> Full;
   };
   template <class LEx, int i> 
   struct Help2<LEx,NIL,CONS<lambda_var<i>,NIL> > { 
      typedef Lambda1<LEx,i> Result; 
      typedef full1<Result> Full;
   };
   template <class LEx, int i, int j> 
   struct Help2<LEx,NIL,CONS<lambda_var<i>,CONS<lambda_var<j>,NIL> > > { 
      typedef Lambda2<LEx,i,j> Result; 
      typedef full2<Result> Full;
   };
   template <class LEx, int i, int j, int k> 
   struct Help2<LEx,NIL,CONS<lambda_var<i>,CONS<lambda_var<j>,
                       CONS<lambda_var<k>,NIL> > > > { 
      typedef Lambda3<LEx,i,j,k> Result; 
      typedef full3<Result> Full;
   };

   template <class LEx> struct Helper {
      typedef typename ListDifference<typename LEx::FreeVars,
         VarsThisLambdaBinds>::Result FreeVars;
      typedef typename Help2<LEx,FreeVars,TBBV>::Result Result;
      typedef typename Help2<LEx,FreeVars,TBBV>::Full Full;
   };
public:
   template <class E> struct RT
   { typedef typename Helper<typename LEify<E>::type>::Full type; };

   template <class E>
   typename RT<E>::type
   operator[]( const E& e ) const {
      typedef typename Helper<typename LEify<E>::type>::Result Result;
      typedef typename Helper<typename LEify<E>::type>::Full Full;
      return Full( Result( LEify<E>::go(e) ) );
   }
};

inline LambdaThingy<NIL> lambda() { return LambdaThingy<NIL>(); }

template <int i> 
LambdaThingy<CONS<lambda_var<i>,NIL> > 
lambda( const lambda_var<i>& ) 
{ return LambdaThingy<CONS<lambda_var<i>,NIL> >(); }

template <int i, int j> 
LambdaThingy<CONS<lambda_var<i>,CONS<lambda_var<j>,NIL> > > 
lambda( const lambda_var<i>&, const lambda_var<j>& ) { 
   YouCannotPassTheSameLambdaVarTo_lambda_MoreThanOnce<(i==j)>::go();
   return LambdaThingy<CONS<lambda_var<i>,CONS<lambda_var<j>,NIL> > >(); 
}

template <int i, int j, int k> 
LambdaThingy<CONS<lambda_var<i>,CONS<lambda_var<j>,CONS<lambda_var<k>,NIL> > > > 
lambda( const lambda_var<i>&, const lambda_var<j>&, const lambda_var<k>& ) { 
   YouCannotPassTheSameLambdaVarTo_lambda_MoreThanOnce<
      (i==j || i==k || j==k) >::go();
   return LambdaThingy<CONS<lambda_var<i>,CONS<lambda_var<j>,
            CONS<lambda_var<k>,NIL> > > >(); 
}

//////////////////////////////////////////////////////////////////////
// lambda language constructs 
//////////////////////////////////////////////////////////////////////

template <IfKind k>
struct IfLambdaoid {
   template <class E, class T, class F>
   exp::IfLE<E,T,F,k>
   operator[]( const CONS<F,CONS<T,CONS<E,NIL> > >& x ) const {
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureLE<E>::go(); EnsureLE<T>::go(); EnsureLE<F>::go();
#endif
      return exp::IfLE<E,T,F,k>( x.tail.tail.head, x.tail.head, x.head );
   }
};

template <int i, class E>
Binder<i,typename LEify<E>::type>
operator==( exp::lambda_var<i>, const E& e ) {
   return Binder<i,typename LEify<E>::type>( LEify<E>::go(e) );
}

template <class LHS, int i, class LE>
CONS<Binder<i,LE>,typename BinderListify<LHS>::type>
operator,( const LHS& lhs, const Binder<i,LE>& b ) {
   return CONS<Binder<i,LE>,typename BinderListify<LHS>::type>
      ( b, BinderListify<LHS>::go(lhs) );
}

template <class H, class T, int i, class LE>
CONS<Binder<i,LE>,CONS<H,T> >
operator,( const CONS<H,T>& lhs, const Binder<i,LE>& b ) {
   return CONS<Binder<i,LE>,CONS<H,T> >( b, lhs );
}

//////////////////////////////////////////////////////////////////////
// LE stuff
//////////////////////////////////////////////////////////////////////

template <class F, class X=void, class Y=void, class Z=void> struct CALL;
template <int i> struct LV;
template <class E, class T, class F> struct IF0;
template <class E, class T, class F> struct IF1;
template <class E, class T, class F> struct IF2;
template <class A, class B=void, class C=void, class D=void> struct LAM;

template <class T> struct LE { typedef T type; };

template <class A, class B=void, class C=void, class D=void> 
struct LET_LEListify {
   typedef typename LEify<typename LE<D>::type>::type LE_D;
   typedef CONS<LE_D,typename LET_LEListify<A,B,C>::type> type;
};
template <class A, class B, class C>
struct LET_LEListify<A,B,C,void> {
   typedef typename LEify<typename LE<C>::type>::type LE_C;
   typedef CONS<LE_C,typename LET_LEListify<A,B>::type> type;
};
template <class A, class B>
struct LET_LEListify<A,B,void,void> {
   typedef typename LEify<typename LE<B>::type>::type LE_B;
   typedef CONS<LE_B,typename LET_LEListify<A>::type> type;
};
template <class A>
struct LET_LEListify<A,void,void,void> {
   typedef typename LEListify<typename LE<A>::type>::type type;
};

template <class F, class X, class Y, class Z> 
struct LE< CALL<F,X,Y,Z> > {
   typedef typename LEify<typename LE<F>::type>::type FF;
   typedef exp::Call<FF,typename LET_LEListify<X,Y,Z>::type> type;
}; 
template <class F, class X, class Y>
struct LE< CALL<F,X,Y,void> > {
   typedef typename LEify<typename LE<F>::type>::type FF;
   typedef exp::Call<FF,typename LET_LEListify<X,Y>::type> type;
}; 
template <class F, class X>
struct LE< CALL<F,X,void,void> > {
   typedef typename LEify<typename LE<F>::type>::type FF;
   typedef exp::Call<FF,typename LET_LEListify<X>::type> type;
}; 
template <class F>
struct LE< CALL<F,void,void,void> > {
   typedef typename LEify<typename LE<F>::type>::type FF;
   typedef exp::Call<FF,NIL> type;
}; 

template <int i> struct LE< LV<i> > { typedef lambda_var<i> type; };

template <class E, class T, class F>
struct LE< IF0<E,T,F> > {
   typedef typename LEify<typename LE<E>::type>::type EE;
   typedef typename LEify<typename LE<T>::type>::type TT;
   typedef typename LEify<typename LE<F>::type>::type FF;
   typedef exp::IfLE<EE,TT,FF,IfNormal> type;
};
template <class E, class T, class F>
struct LE< IF1<E,T,F> > {
   typedef typename LEify<typename LE<E>::type>::type EE;
   typedef typename LEify<typename LE<T>::type>::type TT;
   typedef typename LEify<typename LE<F>::type>::type FF;
   typedef exp::IfLE<EE,TT,FF,IfTrue> type;
};
template <class E, class T, class F>
struct LE< IF2<E,T,F> > {
   typedef typename LEify<typename LE<E>::type>::type EE;
   typedef typename LEify<typename LE<T>::type>::type TT;
   typedef typename LEify<typename LE<F>::type>::type FF;
   typedef exp::IfLE<EE,TT,FF,IfFalse> type;
};

template <int i, int j, int k, class E>
struct LE< LAM<LV<i>,LV<j>,LV<k>,E> > {
   typedef typename LE<E>::type EE;
   typedef typename LambdaThingy<CONS<lambda_var<i>,CONS<lambda_var<j>,
      CONS<lambda_var<k>,NIL> > > >::template RT<EE>::type type;
};
template <int i, int j, class E>
struct LE< LAM<LV<i>,LV<j>,E,void> > {
   typedef typename LE<E>::type EE;
   typedef typename LambdaThingy<CONS<lambda_var<i>,CONS<lambda_var<j>,
      NIL> > >::template RT<EE>::type type;
};
template <int i, class E>
struct LE< LAM<LV<i>,E,void,void> > {
   typedef typename LE<E>::type EE;
   typedef typename LambdaThingy<CONS<lambda_var<i>,
      NIL> >::template RT<EE>::type type;
};
template <class E>
struct LE< LAM<E,void,void,void> > {
   typedef typename LE<E>::type EE;
   typedef typename LambdaThingy<NIL>::template RT<EE>::type type;
};

//////////////////////////////////////////////////////////////////////
// more lambda language constructs
//////////////////////////////////////////////////////////////////////

struct LetLambdaoid {
   template <class BL>
   struct InLambdaoid {
      class Help {
         BL bl;
      public:
         template <class B, class LE> struct Lambdify;
         template <int i, class LEa, class LEb> 
         struct Lambdify<CONS<Binder<i,LEa>,NIL>,LEb> {
            typedef typename LE<CALL<LAM<LV<i>,LEb>,LEa> >::type R;
            static inline R go( const CONS<Binder<i,LEa>,NIL>& binders, 
                                const LEb& le ) {
               lambda_var<i> X;
               return lambda(X)[ le ][ binders.head.exp ];
            }
         };
         template <int i, class LEa, class Rest, class LEb> 
         struct Lambdify<CONS<Binder<i,LEa>,Rest>,LEb> {
            typedef typename LE<CALL<LAM<LV<i>,LEb>,LEa> >::type Inner;
            typedef typename Lambdify<Rest,Inner>::R R;
            static inline R go( const CONS<Binder<i,LEa>,Rest>& binders, 
                                const LEb& le ) {
               lambda_var<i> X;
               return Lambdify<Rest,Inner>::go( binders.tail,
                  lambda(X)[ le ][ binders.head.exp ] );
            }
         };

         Help( const BL& l ) : bl(l) {}
         template <class E> struct RT
         { typedef typename Lambdify<BL,typename LEify<E>::type>::R type; };
         template <class E>
         typename RT<E>::type
         operator[]( const E& e ) {
            return Lambdify<BL,typename LEify<E>::type>::go
               ( bl, LEify<E>::go(e) );
         }
      };
      Help in;
      InLambdaoid( const BL& l ) : in(l) {}
   };

   template <class BL> struct RT
   { typedef InLambdaoid<typename BinderListify<BL>::type> type; };
   template <class BL>
   typename RT<BL>::type
   operator[]( const BL& bl ) const {
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureBinderList<typename BinderListify<BL>::type>::go();
#endif
      return InLambdaoid<typename BinderListify<BL>::type>
         ( BinderListify<BL>::go(bl) );
   }
};

struct LetRecLambdaoid {
   template <class BL>
   struct InLambdaoid {
      class Help {
         BL bl;
      public:
         Help( const BL& l ) : bl(l) {}
         template <class E> struct RT 
         { typedef exp::BindingEnvExp<BL,typename LEify<E>::type> type; };
         template <class E>
         typename RT<E>::type
         operator[]( const E& e ) {
            return exp::BindingEnvExp<BL,typename LEify<E>::type>
               ( bl, LEify<E>::go(e) );
         }
      };
      Help in;
      InLambdaoid( const BL& l ) : in(l) {}
   };

   template <class BL> struct RT;
   template <class BL> friend struct RT;
   template <class BL> struct RT
   { typedef InLambdaoid<typename BinderListify<BL>::type> type; };
   template <class BL>
   typename RT<BL>::type
   operator[]( const BL& bl ) const {
#ifdef BOOST_FCPP_LAMBDA_DEBUG
      EnsureBinderList<typename BinderListify<BL>::type>::go();
#endif
      return InLambdaoid<typename BinderListify<BL>::type>
         ( BinderListify<BL>::go(bl) );
   }
};

//////////////////////////////////////////////////////////////////////
// more LE stuff
//////////////////////////////////////////////////////////////////////

template <int i, class LEx> struct BIND;
template <class A, class B, class C=void, class D=void> struct LET;
template <class A, class B, class C=void, class D=void> struct LETREC;

template <class A, class B=void, class C=void, class D=void> 
struct LET_BinderListify {
   typedef CONS<D,typename LET_BinderListify<A,B,C>::type> type;
};
template <class A, class B, class C>
struct LET_BinderListify<A,B,C,void> {
   typedef CONS<C,typename LET_BinderListify<A,B>::type> type;
};
template <class A, class B>
struct LET_BinderListify<A,B,void,void> {
   typedef CONS<B,typename LET_BinderListify<A>::type> type;
};
template <class A>
struct LET_BinderListify<A,void,void,void> {
   typedef typename BinderListify<A>::type type;
};

template <int i, class A, class LEa>
struct LE< LET<BIND<i,A>,LEa,void,void> > {
   typedef typename LE<LEa>::type LELE;
   typedef Binder<i,typename LEify<typename LE<A>::type>::type> AA;
   typedef typename LetLambdaoid::template RT<typename
      LET_BinderListify<AA>::type>::type::Help::template RT<LELE>::type type;
};
template <int i, class A, int j, class B, class LEa>
struct LE< LET<BIND<i,A>,BIND<j,B>,LEa,void> > {
   typedef typename LE<LEa>::type LELE;
   typedef Binder<i,typename LEify<typename LE<A>::type>::type> AA;
   typedef Binder<j,typename LEify<typename LE<B>::type>::type> BB;
   typedef typename LetLambdaoid::template RT<typename
      LET_BinderListify<AA,BB>::type>::type::Help::template RT<LELE>::type type;
};
template <int i, class A, int j, class B, int k, class C, class LEa>
struct LE< LET<BIND<i,A>,BIND<j,B>,BIND<k,C>,LEa> > {
   typedef typename LE<LEa>::type LELE;
   typedef Binder<i,typename LEify<typename LE<A>::type>::type> AA;
   typedef Binder<j,typename LEify<typename LE<B>::type>::type> BB;
   typedef Binder<k,typename LEify<typename LE<C>::type>::type> CC;
   typedef typename LetLambdaoid::template RT<typename
      LET_BinderListify<AA,BB,CC>::type>::type::Help::template 
      RT<LELE>::type type;
};

template <int i, class A, class LEa>
struct LE< LETREC<BIND<i,A>,LEa,void,void> > {
   typedef typename LE<LEa>::type LELE;
   typedef Binder<i,typename LEify<typename LE<A>::type>::type> AA;
   typedef typename LetRecLambdaoid::template RT<typename
      LET_BinderListify<AA>::type>::type::Help::template RT<LELE>::type type;
};
template <int i, class A, int j, class B, class LEa>
struct LE< LETREC<BIND<i,A>,BIND<j,B>,LEa,void> > {
   typedef typename LE<LEa>::type LELE;
   typedef Binder<i,typename LEify<typename LE<A>::type>::type> AA;
   typedef Binder<j,typename LEify<typename LE<B>::type>::type> BB;
   typedef typename LetRecLambdaoid::template RT<typename
      LET_BinderListify<AA,BB>::type>::type::Help::template RT<LELE>::type type;
};
template <int i, class A, int j, class B, int k, class C, class LEa>
struct LE< LETREC<BIND<i,A>,BIND<j,B>,BIND<k,C>,LEa> > {
   typedef typename LE<LEa>::type LELE;
   typedef Binder<i,typename LEify<typename LE<A>::type>::type> AA;
   typedef Binder<j,typename LEify<typename LE<B>::type>::type> BB;
   typedef Binder<k,typename LEify<typename LE<C>::type>::type> CC;
   typedef typename LetRecLambdaoid::template RT<typename
      LET_BinderListify<AA,BB,CC>::type>::type::Help::template 
      RT<LELE>::type type;
};

} // end namespace lambda_impl

BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN lambda_impl::IfLambdaoid<lambda_impl::IfNormal> if0;
BOOST_FCPP_MAYBE_EXTERN lambda_impl::IfLambdaoid<lambda_impl::IfTrue>   if1;
BOOST_FCPP_MAYBE_EXTERN lambda_impl::IfLambdaoid<lambda_impl::IfFalse>  if2;
BOOST_FCPP_MAYBE_EXTERN lambda_impl::LetRecLambdaoid letrec;
BOOST_FCPP_MAYBE_EXTERN lambda_impl::LetLambdaoid let;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

using lambda_impl::lambda_var;
using lambda_impl::lambda;  // all that work for _one_ exported function :)

using lambda_impl::LE;
using lambda_impl::CALL;
using lambda_impl::LV;
using lambda_impl::IF0;
using lambda_impl::IF1;
using lambda_impl::IF2;
using lambda_impl::LAM;
using lambda_impl::BIND;
using lambda_impl::LET;
using lambda_impl::LETREC;

} // end namespace fcpp
} // end namespace boost

#endif
#endif

