// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

/*
These files (monad_0,monad_2,monad_3) implement the monad examples taken 
from the Wadler paper "Monads for Functional Programming" found at

   http://cm.bell-labs.com/cm/cs/who/wadler/topics/monads.html#marktoberdorf

and the examples implement variations 0, 2, and 3 as described in sections 
2.5-2.9 of the paper.

These files supersede monad0/monad2/monad3, now that monads and lambda
are built into FC++.
*/

#include <iostream>
#include <string>
#define BOOST_FCPP_ENABLE_LAMBDA
#include "prelude.hpp"

using std::cout;
using std::endl;
using std::pair;
using std::string;
using namespace boost::fcpp;

//////////////////////////////////////////////////////////////////////
// useful for variation 3

#include <sstream>
template <class T>
string toString( const T& x ) {
   std::ostringstream oss;
   oss << x;
   return oss.str();
}

//////////////////////////////////////////////////////////////////////

class Term {
   int a_;   // also a
   boost::shared_ptr<Term> t_, u_;
   enum { Con, Div } type;
public:
   Term( int aa ) : a_(aa), type(Con) {}
   Term( boost::shared_ptr<Term> tt, boost::shared_ptr<Term> uu ) 
      : t_(tt), u_(uu), type(Div) {}
   bool isCon() const { return type==Con; }
   int a() const { if( !isCon() ) throw "oops"; return a_; }
   boost::shared_ptr<Term> t() const { if( isCon() ) throw "oops"; return t_; }
   boost::shared_ptr<Term> u() const { if( isCon() ) throw "oops"; return u_; }
   string asString() const {
      if( isCon() ) return "(Con " + toString(a()) + ")";
      else return "(Div " + t()->asString() + " " + u()->asString() + ")";
   }
};

boost::shared_ptr<Term> Con( int a ) 
{ return boost::shared_ptr<Term>( new Term(a) ); }

boost::shared_ptr<Term> 
Div( boost::shared_ptr<Term> t, boost::shared_ptr<Term> u ) 
{ return boost::shared_ptr<Term>( new Term(t,u) ); }

// useful for variation 3
string xline( boost::shared_ptr<Term> t, int v ) { 
   return t->asString() + " --> " + toString(v) + "\n"; 
}
fun2<boost::shared_ptr<Term>,int,string> line = ptr_to_fun(&xline);

//////////////////////////////////////////////////////////////////////

struct OutputM {
   struct XUnit {
      template <class A> struct sig : public fun_type<pair<string,A> > {};
      template <class A>
      typename sig<A>::result_type operator()( const A& a ) const {
         return make_pair( string(), a );
      }
   };
   typedef full1<XUnit> unit_type;
   static unit_type unit;

   struct XBind {
      template <class M, class K> struct sig : public fun_type<
         typename RT<K,typename M::second_type>::result_type> {};
      
      template <class M, class K>
      typename sig<M,K>::result_type
      operator()( const M& m, const K& k ) const {
         lambda_var<1> XA;
         lambda_var<2> YB;
         return lambda()[ let[ XA == m, YB == k[snd[XA]] ]
            .in[ make_pair[ fst[XA] %plus% fst[YB], snd[YB] ] ] ]();
      }
   };
   typedef full2<XBind> bind_type;
   static bind_type bind;
};
OutputM::unit_type OutputM::unit;
OutputM::bind_type OutputM::bind;

fun1<string,pair<string,empty_type> > out = make_pair(_,empty);

//////////////////////////////////////////////////////////////////////

typedef OutputM          M;
typedef pair<string,int> M_int;

//////////////////////////////////////////////////////////////////////

struct Eval : c_fun_type<boost::shared_ptr<Term>,M_int> {
   M_int operator()( boost::shared_ptr<Term> term ) const {
      if( term->isCon() ) 
         return out(line(term,term->a())) ^bind_m_<M>()^ M::unit( term->a() );
      else {
         lambda_var<1> A;
         lambda_var<2> B;
         boost::shared_ptr<Term> t = term->t(), u = term->u();
         return lambda()[ comp_m<M>()[ A %divides% B |
            A <= Eval()(t), B <= Eval()(u),
            out[line[term,A %divides% B]] ] ]();
      }
   }
};

//////////////////////////////////////////////////////////////////////

boost::shared_ptr<Term> answer() {
   return Div( Div( Con(1972), Con(2) ), Con(23) );
}

int main() {
   Eval e;

   M_int r = e( answer() );   
   cout << "(" << r.first << "," << r.second << ")" << endl;
}

