// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_PRELUDE_HPP
#define BOOST_FCPP_PRELUDE_HPP

//////////////////////////////////////////////////////////////////////
// Note that this header file includes all the other FC++ header files,
// so including this one (prelude.hpp) is sufficient to suck in the whole
// library. 
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// Here we define a bunch of functions from the Haskell Standard
// Prelude (HSP).  For documentation of their behaviors, see 
//    http://haskell.org/onlinereport/standard-prelude.html
//
// A number of the functions are not from HSP, but seemed natural/useful.
//////////////////////////////////////////////////////////////////////

#include "list.hpp"

namespace boost {
namespace fcpp {

namespace impl {
struct XId {
   template <class T> struct sig : public fun_type<T> {};

   template <class T>
   T operator()( const T& x ) const {
      return x;
   }
};
}
typedef full1<impl::XId> id_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN id_type id;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
template <class F, class G>
class XCompose0Helper : public c_fun_type<
typename F::template sig<typename RT<G>::result_type>::result_type> {
   F f;
   G g;
public:
   XCompose0Helper( const F& a, const G& b ) : f(a), g(b) {}

   typename F::template sig<typename RT<G>::result_type>::result_type
   operator()() const {
      return f( g() );
   }
};
template <class F, class G>
class XCompose1Helper {
   F f;
   G g;
public:
   XCompose1Helper( const F& a, const G& b ) : f(a), g(b) {}
   template <class X> struct sig : public fun_type<
      typename RT<F,typename RT<G,X>::result_type>::result_type> {};
   template <class X>
   typename sig<X>::result_type operator()( const X& x ) const {
      return f( g(x) );
   }
};
template <class F, class G>
class XCompose2Helper {
   F f;
   G g;
public:
   XCompose2Helper( const F& a, const G& b ) : f(a), g(b) {}
   template <class X, class Y> struct sig : public fun_type<
      typename RT<F,typename RT<G,X,Y>::result_type>::result_type> {};
   template <class X, class Y>
   typename sig<X,Y>::result_type operator()( const X& x, const Y& y ) const {
      return f( g(x,y) );
   }
};
template <class F, class G>
class XCompose3Helper {
   F f;
   G g;
public:
   XCompose3Helper( const F& a, const G& b ) : f(a), g(b) {}
   template <class X, class Y, class Z> struct sig : public fun_type<
      typename RT<F,typename RT<G,X,Y,Z>::result_type>::result_type> {};
   template <class X, class Y, class Z>
   typename sig<X,Y,Z>::result_type 
   operator()( const X& x, const Y& y, const Z& z ) const {
      return f( g(x,y,z) );
   }
};

// compose_type is Haskell's operator (.) 
// compose(f,g)(x,y,z) = f( g(x,y,z) )
class XCompose {
   template <int i, class F, class G> struct Helper;
   template <class F, class G> 
   struct Helper<0,F,G> {
      typedef full0<XCompose0Helper<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full0(XCompose0Helper<F,G>(f,g)); }
   };
   template <class F, class G> 
   struct Helper<1,F,G> {
      typedef full1<XCompose1Helper<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full1(XCompose1Helper<F,G>(f,g)); }
   };
   template <class F, class G> 
   struct Helper<2,F,G> {
      typedef full2<XCompose2Helper<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full2(XCompose2Helper<F,G>(f,g)); }
   };
   template <class F, class G> 
   struct Helper<3,F,G> {
      typedef full3<XCompose3Helper<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full3(XCompose3Helper<F,G>(f,g)); }
   };
public:
   template <class F, class G> struct sig : public 
   fun_type<typename Helper<functoid_traits<G>::max_args,F,G>::Result> {};

   template <class F, class G>
   typename sig<F,G>::result_type operator()( const F& f, const G& g ) const {
      return Helper<functoid_traits<G>::max_args,F,G>::go( f, g );
   }
};
}
typedef full2<impl::XCompose> compose_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN compose_type compose;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// Now we can create "of", so that
//    f ^of^ g
// means
//    compose( funify(f), funify(g) )
//////////////////////////////////////////////////////////////////////

namespace impl {
struct XOf {
   template <class F, class G> struct sig : public fun_type<
      typename RT<compose_type,typename RT<funify_type,F>::result_type,
      typename RT<funify_type,G>::result_type>::result_type> {};
   template <class F, class G>
   typename sig<F,G>::result_type
   operator()( const F& f, const G& g ) const {
      return compose( funify(f), funify(g) );
   }
};
}
typedef full2<impl::XOf> of_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN of_type of;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
template <class F, class G, class H>
class XXCompose2Helper {
   F f;
   G g;
   H h;
public:
   XXCompose2Helper( const F& a, const G& b, const H& c) : f(a), g(b), h(c) {}

   template <class T>
   struct sig : public fun_type<
      typename F::template sig<typename G::template sig<T>::result_type, 
                    typename H::template sig<T>::result_type>::result_type> {};

   template <class T>
   typename sig<T>::result_type operator()( const T& x ) const {
      return f( g(x), h(x) );
   }
};
   
//      compose2(f,g,h)(x) == f( g(x), h(x) )
// compose2 composes a two argument function with two one-argument
// functions (taking the same type). This is quite useful for the
// common case of binary operators.  Use lambda for more-complicated stuff.
struct XCompose2 {
   template <class F, class G, class H>
   struct sig : public fun_type<full1<XXCompose2Helper<F,G,H> > > {};

   template <class F, class G, class H>
   full1<XXCompose2Helper<F,G,H> > 
   operator()(const F& f, const G& g, const H& h) const {
      return make_full1( XXCompose2Helper<F,G,H>( f, g, h ) );
   }
};
}
typedef full3<impl::XCompose2> compose2_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN compose2_type compose2;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XUntil {
   template <class Pred, class Unary, class T>
   struct sig : public fun_type<T> {};

   template <class Pred, class Unary, class T>
   T operator()( const Pred& p, const Unary& op, T start ) const {
      while( !p(start) ) {
         T tmp( start );
         start.~T();
         new (&start) T( op(tmp) );
      }
      return start;
   }
};
}
typedef full3<impl::XUntil> until_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN until_type until;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XLast {
   template <class L>
   struct sig : public fun_type<typename L::value_type> {};

   template <class L>
   typename sig<L>::result_type 
   operator()( const L& ll ) const {
      typename L::delay_result_type l = delay(ll);
      while( !null( tail(l) ) )
         l = tail(l);
      return head(l);
   }
};
}
typedef full1<impl::XLast> last_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN last_type last;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XInit {
   template <class L>
   struct sig : public fun_type<typename L::force_result_type> {};

   template <class L>
   typename sig<L>::result_type
   operator()( const L& l, 
               reuser1<INV,VAR,XInit,typename L::delay_result_type>
               r = NIL ) const {
      if( null( tail( l ) ) )
         return NIL;
      else
         return cons( head(l), r( XInit(), tail(l) ) );
   }
};
}
typedef full1<impl::XInit> init_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN init_type init;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XLength {
   template <class L>
   struct sig : public fun_type<size_t> {};

   template <class L>
   size_t operator()( const L& ll ) const {
      typename L::delay_result_type l = delay(ll);
      size_t x = 0;
      while( !null(l) ) {
         l = tail(l);
         ++x;
      }
      return x;
   }
};
}
typedef full1<impl::XLength> length_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN length_type length;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
// at_type is Haskell's operator (!!)
struct XAt {
   template <class L, class N>
   struct sig : public fun_type<typename L::value_type> {};

   template <class L>
   typename L::value_type operator()( L l, size_t n ) const {
      while( n!=0 ) {
         l = tail(l);
         --n;
      }
      return head(l);
   }
};
}
typedef full2<impl::XAt> at_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN at_type at;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XFilter {
   template <class P, class L>
   struct sig : public fun_type<typename L::force_result_type> {};

   template <class P, class L>
   typename sig<P,L>::result_type 
   operator()( const P& p, const L& ll,
               reuser2<INV,INV,VAR,XFilter,P,typename L::delay_result_type> 
               r = NIL ) const {
      typename L::delay_result_type l = delay(ll);
      while(1) {
         if( null(l) )
            return NIL;
         else if( p(head(l)) )
            return cons( head(l), r( XFilter(), p, tail(l) ) );
         else
            r.iter( XFilter(), p, l = tail(l) );
      }
   }
};
}
typedef full2<impl::XFilter> filter_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN filter_type filter;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XConcat {
   template <class LLT>
   struct sig : public fun_type<
      typename LLT::value_type::template cons_rebind<
                    typename LLT::value_type::value_type>::type> {};

   template <class L>
   typename sig<L>::result_type
   operator()( const L& l, 
               reuser1<INV,VAR,XConcat,typename RT<tail_type,L>::result_type> 
               r = NIL ) const {
      if( null(l) )
         return NIL;
      else
         return cat( head(l), r(XConcat(),tail(l)) );
   }
};
}
typedef full1<impl::XConcat> concat_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN concat_type concat;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
// Note: this isn't lazy (even if 'op' is 'cons').
struct XFoldr {
   template <class Op, class E, class L>
   struct sig : public fun_type<E> {};

   template <class Op, class E, class L>
   E operator()( const Op& op, const E& e, const L& l ) const {
      if( null(l) )
         return e;
      else 
         return op( head(l), XFoldr()( op, e, tail(l) ) );
   }
};
}
typedef full3<impl::XFoldr> foldr_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN foldr_type foldr;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XFoldr1 {
   template <class Op, class L>
   struct sig : public fun_type<typename L::value_type> {};

   template <class Op, class L>
   typename L::value_type operator()( const Op& op, const L& l ) const {
      return foldr( op, head(l), tail(l) );
   }
};
}
typedef full2<impl::XFoldr1> foldr1_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN foldr1_type foldr1;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XFoldl {
   template <class Op, class E, class L>
   struct sig : public fun_type<E> {};

   template <class Op, class E, class L>
   E operator()( const Op& op, E e, const L& ll ) const {
      typename L::delay_result_type l = delay(ll);
      while( !null(l) ) {
         E tmp( e );
         e.~E();
         new (&e) E( op(tmp,head(l)) );
         l = tail(l);
      }
      return e;
   }
};
}
typedef full3<impl::XFoldl> foldl_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN foldl_type foldl;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XFoldl1 {
   template <class Op, class L>
   struct sig : public fun_type<typename L::value_type> {};

   template <class Op, class L>
   typename L::value_type operator()( const Op& op, const L& l ) const {
      return foldl( op, head(l), tail(l) );
   }
};
}
typedef full2<impl::XFoldl1> foldl1_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN foldl1_type foldl1;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XScanr {
   template <class Op, class E, class L>
   struct sig : public fun_type<typename L::template
      cons_rebind<E>::type> {};

   template <class Op, class E, class L>
   typename sig<Op,E,L>::result_type
   operator()( const Op& op, const E& e, const L& l ) const {
      typedef typename L::template cons_rebind<E>::type EL;
      if( null(l) ) {
         EL empty;
         return cons( e, empty );
      }
      else {
         EL temp = XScanr()( op, e, tail(l) );
         return cons( op( head(l), head(temp) ), temp );
      }
   }
};
}
typedef full3<impl::XScanr> scanr_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN scanr_type scanr;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XScanr1 {
   template <class Op, class L>
   struct sig : public fun_type<typename L::force_result_type> {};

   template <class Op, class L>
   typename sig<Op,L>::result_type
   operator()( const Op& op, const L& l ) const {
      if( null( tail(l) ) )
         return l.force();
      else {
         typename sig<Op,L>::result_type temp = XScanr1()( op, tail(l) );
         return cons( op( head(l), head(temp) ), temp );
      }
   }
};
}
typedef full2<impl::XScanr1> scanr1_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN scanr1_type scanr1;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XScanl {
   template <class Op, class E, class L>
   struct sig : public fun_type<typename L::template
      cons_rebind<E>::type> {};

   template <class Op, class E, class L>
   typename sig<Op,E,L>::result_type 
   operator()( const Op& op, const E& e, const L& l,
         reuser3<INV,INV,VAR,VAR,XScanl,Op,E,typename L::delay_result_type>
         r = NIL ) const {
      typedef typename L::force_result_type OL;
      if( null(l) ) {
         OL empty;
         return cons( e, empty );
      }
      else
         return cons( e, r( XScanl(), op, op(e,head(l)), tail(l) ) );
   }
};
}
typedef full3<impl::XScanl> scanl_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN scanl_type scanl;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XScanl1 {
   template <class Op, class L>
   struct sig : public fun_type<typename L::force_result_type> {};

   template <class Op, class L>
   typename sig<Op,L>::result_type
   operator()( const Op& op, const L& l ) const {
      return scanl( op, head(l), tail(l) );
   }
};
}
typedef full2<impl::XScanl1> scanl1_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN scanl1_type scanl1;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XIterate {
   // Note: this does always return an odd_list; iterate() takes no ListLike
   // parameter, and it requires that its result be lazy.
   template <class F, class T>
   struct sig : public fun_type<odd_list<T> > {};

   template <class F, class T>
   odd_list<T> operator()( const F& f, const T& x,
                          reuser2<INV,INV,VAR,XIterate,F,T> r = NIL ) const {
      return cons( x, r( XIterate(), f, f(x) ) );
   }
};
}
typedef full2<impl::XIterate> iterate_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN iterate_type iterate;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XRepeat {
   // See note for iterate()
   template <class T>
   struct sig : public fun_type<odd_list<T> > {};

   template <class T>
   odd_list<T> operator()( const T& x, 
                          reuser1<INV,INV,XRepeat,T> r = NIL ) const {
      return cons( x, r( XRepeat(), x ) );
   }
};
}
typedef full1<impl::XRepeat> repeat_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN repeat_type repeat;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XMap {
   template <class F, class L>
   struct sig : public fun_type<typename L::template cons_rebind<
      typename RT<F,typename L::value_type>::result_type>::type> {};

   template <class F, class L>
   typename sig<F,L>::result_type
   operator()( const F& f, const L& l, 
               reuser2<INV,INV,VAR,XMap,F,
                       typename RT<tail_type,L>::result_type>
               r = NIL ) const {
      if( null(l) )
         return NIL;
      else
         return cons( f(head(l)), r( XMap(), f, tail(l) ) );
   }
};
}
typedef full2<impl::XMap> map_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN map_type map;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XTake {
   template <class N,class L>
   struct sig : public fun_type<typename L::force_result_type> {};

   template <class L>
   typename sig<size_t,L>::result_type
   operator()( size_t n, const L& l,
               reuser2<INV,VAR,VAR,XTake,size_t,typename L::force_result_type> 
               r = NIL ) const {
      if( n==0 || null(l) )
         return NIL;
      else
         return cons( head(l), r( XTake(), n-1, tail(l) ) );
   }
};
}
typedef full2<impl::XTake> take_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN take_type take;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XDrop {
   template <class Dummy, class L>
   struct sig : public fun_type<typename L::delay_result_type> {};
   
   template <class L>
   typename sig<size_t,L>::result_type 
   operator()( size_t n, const L& ll ) const {
      typename L::delay_result_type l = delay(ll);
      while( n!=0 && !null(l) ) {
         --n;
         l = tail(l);
      }
      return l;
   }
};
}
typedef full2<impl::XDrop> drop_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN drop_type drop;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XTakeWhile {
   template <class P, class L>
   struct sig : public fun_type<typename L::force_result_type> {};

   template <class P, class L>
   typename sig<P,L>::result_type
   operator()( const P& p, const L& l,
               reuser2<INV,INV,VAR,XTakeWhile,P,typename L::delay_result_type>
               r = NIL ) const {
      if( null(l) || !p( head(l) ) )
         return NIL;
      else
         return cons( head(l), r( XTakeWhile(), p, tail(l) ) );
   }
};
}
typedef full2<impl::XTakeWhile> take_while_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN take_while_type take_while;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XDropWhile {
   template <class P, class L>
   struct sig : public fun_type<typename L::delay_result_type> {};

   template <class P, class L>
   typename sig<P,L>::result_type
   operator()( const P& p, const L& ll ) const {
      typename L::delay_result_type l = delay(ll);
      while( !null(l) && p( head(l) ) )
         l = tail(l);
      return l;
   }
};
}
typedef full2<impl::XDropWhile> drop_while_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN drop_while_type drop_while;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XReplicate {
   // Note: this function returns an odd_list simply because there is no
   // convenient way to do otherwise without modifying the
   // interface/behavior.
   template <class N, class T>
   struct sig : public fun_type<odd_list<T> > {};

   template <class T>
   odd_list<T> operator()( size_t n, const T& x ) const {
      return take( n, repeat(x) );
   }
};
}
typedef full2<impl::XReplicate> replicate_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN replicate_type replicate;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XCycle {
   template <class L>
   struct sig : public fun_type<typename L::force_result_type> {};

   template <class L>
   typename sig<L>::result_type
   operator()( const L& l, reuser1<INV,INV,XCycle,L> r = NIL ) const {
      ensure_lazy<L>();
      return cat( l, r( XCycle(), l ) );
   }
};
}
typedef full1<impl::XCycle> cycle_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN cycle_type cycle;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XSplitAt {
   template <class N, class L>
   struct sig : public fun_type<std::pair<typename
      L::delay_result_type,typename L::delay_result_type> > {};

   template <class L>
   typename sig<size_t,L>::result_type
   operator()( size_t n, const L& l ) const {
      typedef typename L::delay_result_type EL;
      if( n==0 || null(l) ) {
         EL empty;
         return std::make_pair( empty, delay(l) );
      }
      else {
         std::pair<EL,EL> temp = XSplitAt()( n-1, tail(l) );
         EL tl = cons( head(l), temp.first );
         return std::make_pair( tl, temp.second );
      }
   }
};
}
typedef full2<impl::XSplitAt> split_at_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN split_at_type split_at;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XSpan {
   template <class P, class L>
   struct sig : public fun_type<std::pair<typename
      L::delay_result_type,typename L::delay_result_type> > {};

   template <class P, class L>
   typename sig<P,L>::result_type
   operator()( const P& p, const L& l ) const {
      typedef typename L::delay_result_type EL;
      if( null(l) || !p(head(l)) ) {
         EL empty;
         return std::make_pair( empty, l );
      }
      else {
         std::pair<EL,EL> temp = XSpan()( p, tail(l) );
         EL tl = cons( head(l), temp.first );
         return std::make_pair( tl, temp.second );
      }
   }
};
}
typedef full2<impl::XSpan> span_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN span_type span;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XBreak {
   template <class P, class L>
   struct sig : public fun_type<std::pair<typename
      L::delay_result_type,typename L::delay_result_type> > {};

   template <class P, class L>
   typename sig<P,L>::result_type
   operator()( const P& p, const L& l ) const {
      return span( compose_type()( logical_not_type(), p ), l );
   }
};
}
typedef full2<impl::XBreak> break_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
   // C++ keyword, so add trailing underscore
BOOST_FCPP_MAYBE_EXTERN break_type break_; 
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
template <class Binary>
class XFlipHelper {
   Binary op;
public:
   XFlipHelper( const Binary& b ) : op(b) {}
   
   template <class Y, class X>
   struct sig : public fun_type<
      typename Binary::template sig<X,Y>::result_type > {};

   template <class Y, class X>
   typename Binary::template sig<X,Y>::result_type
   operator()( const Y& y, const X& x ) const {
      return op( x, y );
   }
};
struct XFlip {
   template <class Binary>
   struct sig : public fun_type<full2<XFlipHelper<Binary> > > {};

   template <class Binary>
   full2<XFlipHelper<Binary> > operator()( const Binary& op ) const {
      return make_full2( XFlipHelper<Binary>( op ) );
   }
};
}
typedef full1<impl::XFlip> flip_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN flip_type flip;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XReverse {
   template <class L>
   struct sig : public fun_type<typename L::delay_result_type> {};

   template <class L>
   typename sig<L>::result_type
   operator()( const L& l ) const {
      typedef typename L::force_result_type OL;
      OL empty;
      return thunk3( foldl, flip(cons), empty, l );
   }
};
}
typedef full1<impl::XReverse> reverse_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN reverse_type reverse;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// Not HSP but close
//////////////////////////////////////////////////////////////////////

// These next two are defined as _lazy_ versions of these operators when
// the args are lazy lists
namespace impl {
struct XAnd {
   template <class L> struct sig : public fun_type<bool> {};
   template <class L>
   bool operator()( const L& l ) const {
      return null(drop_while( equal(true), l ));
   }
};
}
typedef full1<impl::XAnd> and_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN and_type and_;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XOr {
   template <class L> struct sig : public fun_type<bool> {};
   template <class L>
   bool operator()( const L& l ) const {
      return !null(drop_while( equal(false), l ));
   }
};
}
typedef full1<impl::XOr> or_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN or_type or_;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// Back to HSP
//////////////////////////////////////////////////////////////////////

namespace impl {
struct XAll {
   template <class P, class L>
   struct sig : public fun_type<bool> {};

   template <class P, class L>
   bool operator()( const P& p, const L& l ) const {
      return and_( map( p, l ) );
   }
};
}
typedef full2<impl::XAll> all_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN all_type all;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XAny {
   template <class P, class L>
   struct sig : public fun_type<bool> {};

   template <class P, class L>
   bool operator()( const P& p, const L& l ) const {
      return or_( map( p, l ) );
   }
};
}
typedef full2<impl::XAny> any_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN any_type any;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XElem {
   template <class T, class L>
   struct sig : public fun_type<bool> {};

   template <class T, class L>
   bool operator()( const T& x, const L& l ) const {
      return any( equal(x), l );
   }
};
}
typedef full2<impl::XElem> elem_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN elem_type elem;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XNotElem {
   template <class T, class L>
   struct sig : public fun_type<bool> {};

   template <class T, class L>
   bool operator()( const T& x, const L& l ) const {
      return all( not_equal(x), l );
   }
};
}
typedef full2<impl::XNotElem> not_elem_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN not_elem_type not_elem;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XSum {
   template <class L>
   struct sig : public fun_type<typename L::value_type> {};

   template <class L>
   typename L::value_type operator()( const L& l ) const {
      return foldl( plus, 0, l );
   }
};
}
typedef full1<impl::XSum> sum_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN sum_type sum;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XProduct {
   template <class L>
   struct sig : public fun_type<typename L::value_type> {};

   template <class L>
   typename L::value_type operator()( const L& l ) const {
      return foldl( multiplies, 1, l );
   }
};
}
typedef full1<impl::XProduct> product_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN product_type product;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XMinimum {
   template <class L>
   struct sig : public fun_type<typename L::value_type> {};

   template <class L>
   typename L::value_type operator()( const L& l ) const {
      return foldl1( min, l );
   }
};
}
typedef full1<impl::XMinimum> minimum_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN minimum_type minimum;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

// FIX it might be cool to have maximum_with and max_with which take a
// comparator as an extra argument.  Same for mins.
namespace impl {
struct XMaximum {
   template <class L>
   struct sig : public fun_type<typename L::value_type> {};

   template <class L>
   typename L::value_type operator()( const L& l ) const {
      return foldl1( max, l );
   }
};
}
typedef full1<impl::XMaximum> maximum_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN maximum_type maximum;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XZipWith {
   template <class Z, class LA, class LB>
   struct sig : public fun_type<typename LA::template cons_rebind<
      typename RT<Z,typename LA::value_type,
                    typename LB::value_type>::result_type>::type > {};

   template <class Z, class LA, class LB>
   typename sig<Z,LA,LB>::result_type
   operator()( const Z& z, const LA& a, const LB& b,
               reuser3<INV,INV,VAR,VAR,XZipWith,Z,
                  typename LA::delay_result_type,
                  typename LB::delay_result_type> r = NIL ) const {
      if( null(a) || null(b) )
         return NIL;
      else
         return cons( z(head(a),head(b)),
            r( XZipWith(), z, tail(a), tail(b) ) );
   }
};
}
typedef full3<impl::XZipWith> zip_with_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN zip_with_type zip_with;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XZip {
   template <class LA, class LB>
   struct sig : public fun_type<typename RT<zip_with_type,
      make_pair_type,LA,LB>::result_type> {};

   template <class LA, class LB>
   typename sig<LA,LB>::result_type
   operator()( const LA& a, const LB& b ) const {
      return zip_with( make_pair_type(), a, b );
   }
};
}
typedef full2<impl::XZip> zip_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN zip_type zip;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XFst {
   template <class P>
   struct sig : public fun_type<typename P::first_type> {};

   template <class A, class B>
   A operator()( const std::pair<A,B>& p ) const {
      return p.first;
   }
};
}
typedef full1<impl::XFst> fst_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN fst_type fst;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XSnd {
   template <class P>
   struct sig : public fun_type<typename P::second_type> {};

   template <class A, class B>
   B operator()( const std::pair<A,B>& p ) const {
      return p.second;
   }
};
}
typedef full1<impl::XSnd> snd_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN snd_type snd;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XUnzip {
   template <class LPT>
   struct sig : public fun_type<std::pair<
      typename LPT::template cons_rebind<
         typename LPT::value_type::first_type>::delay_type,
      typename LPT::template cons_rebind<
         typename LPT::value_type::second_type>::delay_type> > {};

   template <class LPT>
   typename sig<LPT>::result_type
   operator()( const LPT& l ) const {
      typedef typename LPT::template cons_rebind<
         typename LPT::value_type::first_type>::delay_type F;
      typedef typename LPT::template cons_rebind<
         typename LPT::value_type::second_type>::delay_type S;
      return std::make_pair( F(thunk2(map,fst,l)), 
                             S(thunk2(map,snd,l))  );
   }
};
}
typedef full1<impl::XUnzip> unzip_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN unzip_type unzip;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XGcdPrime {
   template <class T, class U> struct sig;
   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( T x, T y ) const {
      while( y!=0 ) {
         T tmp( x%y );
         x = y;
         y = tmp;
      }
      return x;
   }
};
struct XGcd {
   template <class T, class U> struct sig;
   template <class T>
   struct sig<T,T> : public fun_type<T> {};

   template <class T>
   T operator()( const T& x, const T& y ) const {
      if( x==0 && y==0 )
         throw fcpp_exception("gcd_type error: x and y both 0");
      return XGcdPrime()( x<0?-x:x, y<0?-y:y );
   }
};
}
typedef full2<impl::XGcd> gcd_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN gcd_type gcd;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XOdd {
   template <class T>
   struct sig : public fun_type<bool> {};

   template <class T>
   bool operator()( const T& x ) const {
      return x%2==1;
   }
};
}
typedef full1<impl::XOdd> odd_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN odd_type odd;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XEven {
   template <class T>
   struct sig : public fun_type<bool> {};

   template <class T>
   bool operator()( const T& x ) const {
      return x%2==0;
   }
};
}
typedef full1<impl::XEven> even_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN even_type even;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// Not HSP but close
//////////////////////////////////////////////////////////////////////

namespace impl {
template <class T>
struct XEFH : public Fun0Impl< odd_list<T> > {
   mutable T x;
   XEFH( const T& xx ) : x(xx) {}
   odd_list<T> operator()() const {
      ++x;
      return cons( x-1, fun0<odd_list<T> >(1,this) );
   }
};
struct XEnumFrom {
   template <class T>
   struct sig : fun_type<list<T> > {};

   template <class T>
   list<T> operator()( const T& x ) const {
      return fun0<odd_list<T> >(1, new XEFH<T>(x) );
   }
};
}
typedef full1<impl::XEnumFrom> enum_from_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN enum_from_type enum_from;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
template <class T>
struct XEFTH : public Fun0Impl<odd_list<T> > {
   mutable T x;
   T y;
   XEFTH( const T& xx, const T& yy ) : x(xx), y(yy) {}
   odd_list<T> operator()() const {
      if( x > y )
         return NIL;
      ++x;
      return cons( x-1, fun0<odd_list<T> >( 1, this ) );
   }
};
struct XEnumFromTo {
   // Note that this returns a list, as there's no convenient
   // parameter to specify otherwise.
   template <class T, class U> struct sig;
   template <class T>
   struct sig<T,T> : fun_type<list<T> > {};

   template <class T>
   list<T> operator()( const T& x, const T& y ) const {
      return fun0<odd_list<T> >( 1, new XEFTH<T>(x,y) );
   }
};
}
typedef full2<impl::XEnumFromTo> enum_from_to_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN enum_from_to_type enum_from_to;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
// Not HSP
struct XListUntil {
   template <class P, class F, class T>
   struct sig : public fun_type<list<T> > {};

   template <class Pred, class Unary, class T>
   list<T> operator()( const Pred& p, const Unary& f, const T& x ) const {
      return take_while( compose_type()(logical_not,p), iterate(f,x) );
   }
};
}
typedef full3<impl::XListUntil> list_until_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN list_until_type list_until;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// The "maybe" type, from Haskell
//////////////////////////////////////////////////////////////////////

struct a_unique_type_for_nothing {};
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN a_unique_type_for_nothing NOTHING;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

template <class T>
class maybe {
   odd_list<T> rep;
public:
   typedef T value_type;

   maybe( a_unique_type_for_nothing ) {}
   maybe() {}                                    // the Nothing constructor
   maybe( const T& x ) : rep( cons(x,NIL) ) {}   // the just_type constructor

   bool is_nothing() const { return null(rep); }
   T value() const { return head(rep); }
};

namespace impl {
   struct XJust {
      template <class T> struct sig : public fun_type<maybe<T> > {};
   
      template <class T>
      typename sig<T>::result_type
      operator()( const T& x ) const {
         return maybe<T>( x );
      }
   };
}
typedef full1<impl::XJust> just_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN just_type just;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

// Haskell's "()" type/value
struct empty_type {};
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN empty_type empty;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

// Haskell has curry() and uncurry() for pairs; we call these h_curry()
// and h_uncurry().
namespace impl {
   struct XHCurry {
      template <class F, class X, class Y> struct sig : public fun_type<
         typename RT<F,std::pair<X,Y> >::result_type> {};
      template <class F, class X, class Y>
      typename sig<F,X,Y>::result_type
      operator()( const F& f, const X& x, const Y& y ) const {
         return f( make_pair(x,y) );
      }
   };
}
typedef full3<impl::XHCurry> h_curry_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN h_curry_type h_curry;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
   struct XHUncurry {
      template <class F, class P> struct sig : public fun_type<typename 
         RT<F,typename P::first_type,typename P::second_type>::result_type> {};
      template <class F, class P>
      typename sig<F,P>::result_type
      operator()( const F& f, const P& p ) const {
         return f( fst(p), snd(p) );
      }
   };
}
typedef full2<impl::XHUncurry> h_uncurry_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN h_uncurry_type h_uncurry;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// That's the end of the Haskell stuff; on to made-just-for-FC++
//////////////////////////////////////////////////////////////////////

namespace impl {
struct XBetween {
   template <class T, class U, class V> struct sig
      : public fun_type<bool> {};
   template <class T>
   bool operator()( const T& lower, const T& upper, const T& goal ) const {
      return less_equal(goal,upper) && greater_equal(goal,lower);
   }
};
}
// between(lower,upper)(x)  =  x>=lower && x<=upper
typedef full3<impl::XBetween> between_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN between_type between;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// Useful effect combinators
//////////////////////////////////////////////////////////////////////
// Here we define some combinators for statement sequencing:
//    before(f,g)(args) = { f(); return g(args); }
//    after(f,g)(args)  = { r = f(args); g(); return r; }
// That is, before() prepends a thunk onto a functoid, and after()
// appends the thunk onto the back of a functoid.  Finally, no_op
// results in a thunk that does nothing, and serves as the left/right
// identity element for before/after thusly:
//    f  =  before( no_op, f )  =  after( f, no_op )
// Note: the effect happens when all of the functoid's expected
// arguments finally arrive (which, thanks to currying, might not be at
// the "next call".  So if g() expects two arguments, note that
//    before( f, g )( x )
// will not call f() now, rather it waits for another argument.

namespace impl {
struct XNoOp : public c_fun_type<void> {
   void operator()() const {}
};
}
typedef full0<impl::XNoOp> no_op_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN no_op_type no_op;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
template <class F, class G>
class XBeforer0 : public c_fun_type<typename RT<G>::result_type> {
   F f;
   G g;
public:
   XBeforer0( const F& ff, const G& gg ) : f(ff), g(gg) {}
   typename RT<G>::result_type 
   operator()() const { f(); return g(); }
};
template <class F, class G>
class XBeforer1 { 
   F f;
   G g;
public:
   XBeforer1( const F& ff, const G& gg ) : f(ff), g(gg) {}
   template <class X> struct sig 
   : public fun_type<typename RT<G,X>::result_type> {};
   template <class X>
   typename sig<X>::result_type 
   operator()( const X& x ) const { f(); return g(x); }
};
template <class F, class G>
class XBeforer2 { 
   F f;
   G g;
public:
   XBeforer2( const F& ff, const G& gg ) : f(ff), g(gg) {}
   template <class X, class Y> struct sig 
   : public fun_type<typename RT<G,X,Y>::result_type> {};
   template <class X, class Y>
   typename sig<X,Y>::result_type 
   operator()( const X& x, const Y& y ) const { f(); return g(x,y); }
};
template <class F, class G>
class XBeforer3 { 
   F f;
   G g;
public:
   XBeforer3( const F& ff, const G& gg ) : f(ff), g(gg) {}
   template <class X, class Y, class Z> struct sig 
   : public fun_type<typename RT<G,X,Y,Z>::result_type> {};
   template <class X, class Y, class Z>
   typename sig<X,Y,Z>::result_type 
   operator()( const X& x, const Y& y, const Z& z ) const 
   { f(); return g(x,y,z); }
};

class XBefore {
   template <int i, class F, class G> struct Helper;
   template <class F, class G> 
   struct Helper<0,F,G> {
      typedef full0<XBeforer0<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full0( XBeforer0<F,G>(f,g) ); }
   };
   template <class F, class G> 
   struct Helper<1,F,G> {
      typedef full1<XBeforer1<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full1( XBeforer1<F,G>(f,g) ); }
   };
   template <class F, class G> 
   struct Helper<2,F,G> {
      typedef full2<XBeforer2<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full2( XBeforer2<F,G>(f,g) ); }
   };
   template <class F, class G> 
   struct Helper<3,F,G> {
      typedef full3<XBeforer3<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full3( XBeforer3<F,G>(f,g) ); }
   };
public:
   template <class F, class G> struct sig : public fun_type<
      typename Helper<functoid_traits<G>::max_args,F,G>::Result> {};
   template <class F, class G>
   typename sig<F,G>::result_type operator()( const F& f, const G& g ) const {
      return Helper<functoid_traits<G>::max_args,F,G>::go( f, g );
   }
};
}
typedef full2<impl::XBefore> before_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN before_type before;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
template <class F, class G>
class XAfterer0 : public c_fun_type<typename RT<F>::result_type> {
   F f;
   G g;
public:
   XAfterer0( const F& ff, const G& gg ) : f(ff), g(gg) {}
   typename RT<F>::result_type operator()() const 
   { typename RT<F>::result_type tmp = f(); g();
return tmp; }
};
template <class F, class G>
class XAfterer1 { 
   F f;
   G g;
public:
   XAfterer1( const F& ff, const G& gg ) : f(ff), g(gg) {}
   template <class X> struct sig 
   : public fun_type<typename RT<F,X>::result_type> {};
   template <class X>
   typename sig<X>::result_type 
   operator()( const X& x ) const 
   { typename sig<X>::result_type tmp = f(x); g(); return tmp; }
};
template <class F, class G>
class XAfterer2 { 
   F f;
   G g;
public:
   XAfterer2( const F& ff, const G& gg ) : f(ff), g(gg) {}
   template <class X, class Y> struct sig 
   : public fun_type<typename RT<F,X,Y>::result_type> {};
   template <class X, class Y>
   typename sig<X,Y>::result_type 
   operator()( const X& x, const Y& y ) const 
   { typename sig<X,Y>::result_type tmp = f(x,y); g(); return tmp; }
};
template <class F, class G>
class XAfterer3 { 
   F f;
   G g;
public:
   XAfterer3( const F& ff, const G& gg ) : f(ff), g(gg) {}
   template <class X, class Y, class Z> struct sig 
   : public fun_type<typename RT<F,X,Y,Z>::result_type> {};
   template <class X, class Y, class Z>
   typename sig<X,Y,Z>::result_type 
   operator()( const X& x, const Y& y, const Z& z ) const 
   { typename sig<X,Y,Z>::result_type tmp = f(x,y,z); g(); return tmp; }
};

class XAfter {
   template <int i, class F, class G> struct Helper;
   template <class F, class G> 
   struct Helper<0,F,G> {
      typedef full0<XAfterer0<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full0( XAfterer0<F,G>(f,g) ); }
   };
   template <class F, class G> 
   struct Helper<1,F,G> {
      typedef full1<XAfterer1<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full1( XAfterer1<F,G>(f,g) ); }
   };
   template <class F, class G> 
   struct Helper<2,F,G> {
      typedef full2<XAfterer2<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full2( XAfterer2<F,G>(f,g) ); }
   };
   template <class F, class G> 
   struct Helper<3,F,G> {
      typedef full3<XAfterer3<F,G> > Result;
      static Result go( const F& f, const G& g ) 
      { return make_full3( XAfterer3<F,G>(f,g) ); }
   };
public:
   template <class F, class G> struct sig : public fun_type<
      typename Helper<functoid_traits<F>::max_args,F,G>::Result> {};
   template <class F, class G>
   typename sig<F,G>::result_type operator()( const F& f, const G& g ) const {
      return Helper<functoid_traits<F>::max_args,F,G>::go( f, g );
   }
};
}
typedef full2<impl::XAfter> after_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN after_type after;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

// emptify(f)(yadda) == f(yadda), but throws away f's result (even if
//    it's void) and returns Empty instead.  This is useful because
//       length( map( emptify(effectFunctoid), someList ) )
//    is an easy way to do something with each element of someList.
namespace impl {
template <class F>
class XEmptifier0 : public c_fun_type<empty_type> {
   F f;
public:
   XEmptifier0( const F& ff ) : f(ff) {}
   empty_type operator()() const { f(); return empty; }
};
template <class F>
class XEmptifier1 { 
   F f;
public:
   XEmptifier1( const F& ff ) : f(ff) {}
   template <class X> struct sig 
   : public fun_type<empty_type> {};
   template <class X>
   typename sig<X>::result_type 
   operator()( const X& x ) const { f(x); return empty; }
};
template <class F>
class XEmptifier2 { 
   F f;
public:
   XEmptifier2( const F& ff ) : f(ff) {}
   template <class X, class Y> struct sig 
   : public fun_type<empty_type> {};
   template <class X, class Y>
   typename sig<X,Y>::result_type 
   operator()( const X& x, const Y& y ) const { f(x,y); return empty; }
};
template <class F>
class XEmptifier3 { 
   F f;
public:
   XEmptifier3( const F& ff ) : f(ff) {}
   template <class X, class Y, class Z> struct sig 
   : public fun_type<empty_type> {};
   template <class X, class Y, class Z>
   typename sig<X,Y,Z>::result_type 
   operator()( const X& x, const Y& y, const Z& z ) const 
   { f(x,y,z); return empty; }
};

class XEmptify {
   template <int i, class F> struct Helper;
   template <class F> struct Helper<0,F> {
      typedef full0<XEmptifier0<F> > Result;
      static Result go( const F& f ) 
      { return make_full0( XEmptifier0<F>(f) ); }
   };
   template <class F> struct Helper<1,F> {
      typedef full1<XEmptifier1<F> > Result;
      static Result go( const F& f ) 
      { return make_full1( XEmptifier1<F>(f) ); }
   };
   template <class F> struct Helper<2,F> {
      typedef full2<XEmptifier2<F> > Result;
      static Result go( const F& f ) 
      { return make_full2( XEmptifier2<F>(f) ); }
   };
   template <class F> struct Helper<3,F> {
      typedef full3<XEmptifier3<F> > Result;
      static Result go( const F& f ) 
      { return make_full3( XEmptifier3<F>(f) ); }
   };
public:
   template <class F> struct sig : public fun_type<
      typename Helper<functoid_traits<F>::max_args,F>::Result> {};
   template <class F>
   typename sig<F>::result_type operator()( const F& f ) const {
      return Helper<functoid_traits<F>::max_args,F>::go( f );
   }
};
}
typedef full1<impl::XEmptify> emptify_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN emptify_type emptify;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XThunkFuncToFunc {
   // tftf(f)(yadda) = f()(yadda)
   // f is a thunk that returns a functoid; this combinator 'hides' the thunk.
   // can be useful to break what would otherwise be infinite recursion.
   template <class F> class TFTFH0 
   : c_fun_type<typename RT<typename RT<F>::result_type>::result_type> {
      F f;
   public:
      TFTFH0( const F& ff ) : f(ff) {}
      template <class X>
      typename RT<typename RT<F>::result_type>::result_type
      operator()() const {
         return f()();
      }
   };
   template <class F> class TFTFH1 {
      F f;
   public:
      TFTFH1( const F& ff ) : f(ff) {}
      template <class X> struct sig : public fun_type<
         typename RT<typename RT<F>::result_type,X>::result_type   > {};
      template <class X>
      typename sig<X>::result_type
      operator()( const X& x ) const {
         return f()( x );
      }
   };
   template <class F> class TFTFH2 {
      F f;
   public:
      TFTFH2( const F& ff ) : f(ff) {}
      template <class X, class Y> struct sig : public fun_type<
         typename RT<typename RT<F>::result_type,X,Y>::result_type   > {};
      template <class X, class Y>
      typename sig<X,Y>::result_type
      operator()( const X& x, const Y& y ) const {
         return f()( x, y );
      }
   };
   template <class F> class TFTFH3 {
      F f;
   public:
      TFTFH3( const F& ff ) : f(ff) {}
      template <class X, class Y, class Z> struct sig : public fun_type<
         typename RT<typename RT<F>::result_type,X,Y,Z>::result_type   > {};
      template <class X, class Y, class Z>
      typename sig<X,Y,Z>::result_type
      operator()( const X& x, const Y& y, const Z& z ) const {
         return f()( x, y, z );
      }
   };
   template <int i, class F> struct Helper;
   template <class F> struct Helper<0,F> {
      typedef full0<TFTFH0<F> > Result;
      static Result go( const F& f )
      { return make_full0( TFTFH0<F>(f) ); }
   };
   template <class F> struct Helper<1,F> {
      typedef full1<TFTFH1<F> > Result;
      static Result go( const F& f )
      { return make_full1( TFTFH1<F>(f) ); }
   };
   template <class F> struct Helper<2,F> {
      typedef full2<TFTFH2<F> > Result;
      static Result go( const F& f )
      { return make_full2( TFTFH2<F>(f) ); }
   };
   template <class F> struct Helper<3,F> {
      typedef full3<TFTFH3<F> > Result;
      static Result go( const F& f )
      { return make_full3( TFTFH3<F>(f) ); }
   };

   template <class F> struct sig : public fun_type<typename Helper<
      functoid_traits<typename RT<F>::result_type>::max_args,F>::Result> {};
   template <class F>
   typename sig<F>::result_type
   operator()( const F& f ) const {
      return Helper<functoid_traits<typename RT<F>::result_type>
         ::max_args,F>::go(f);
   }
};
}
typedef full1<impl::XThunkFuncToFunc> thunk_func_to_func_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN thunk_func_to_func_type thunk_func_to_func;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE
   
//////////////////////////////////////////////////////////////////////
// split_args
//////////////////////////////////////////////////////////////////////
// Sometimes FC++ expressions result in functoids where, for example,
//    f(x)(y)
// is legal but
//    f(x,y)
// is not, owing to the fact that (in the example) f() is a one-argument
// functoid the returns another one-argument functoid, which is 
// different from a two-argument functoid.  (In Haskell, the two types
// are identical.)  split_args() wraps a functoid in a magical cloak which
// splits up its arguments, so that, for example,
//    split_args(f)(x,y,z) = f(x)(y)(z)
// It rarely arises that you need this, but when you do, you can't live
// without it.

namespace impl {
template <class F>
class XSplitArgsable : public smart_functoid3 {
   F f;
public:
   XSplitArgsable( const F& ff ) : f(ff) { 
      functoid_traits<F>::template ensure_accepts<1>::args();
   }

   template <class X, class Y=void, class Z=void, class Dummy=void>
   struct sig : public fun_type<
typename RT<typename RT<typename RT<F,X>::result_type,Y>::result_type,Z>
   ::result_type> {};

   template <class X, class Y, class Dummy>
   struct sig<X,Y,void,Dummy> : public fun_type<
            typename RT<typename RT<F,X>::result_type,Y>::result_type> {};

   template <class X, class Dummy>
   struct sig<X,void,void,Dummy> : public fun_type<
            typename RT<F,X>::result_type> {};

   template <class X, class Y, class Z>
   typename sig<X,Y,Z>::result_type
   operator()( const X& x, const Y& y, const Z& z ) const { return f(x)(y)(z); }

   template <class X, class Y>
   typename sig<X,Y>::result_type
   operator()( const X& x, const Y& y ) const { return f(x)(y); }

   template <class X>
   typename sig<X>::result_type
   operator()( const X& x ) const { return f(x); }
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef XSplitArgsable This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif

   template <class X> struct result;
   template <class Me, class X, class Y, class Z>
   struct result<Me(X,Y,Z)> :public Type<typename sig<X,Y,Z>::result_type> {};
   template <class Me, class X, class Y>
   struct result<Me(X,Y)> :public Type<typename sig<X,Y,void>::result_type> {};
   template <class Me, class X>
   struct result<Me(X)> :public Type<typename sig<X,void,void>::result_type> {};
};
struct XSplitArgs {
   template <class F>
   struct sig : fun_type<XSplitArgsable<F> > {};

   template <class F>
   typename sig<F>::result_type 
   operator()( const F& f ) const { return XSplitArgsable<F>(f); }
};
}
typedef full1<impl::XSplitArgs> split_args_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN split_args_type split_args;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// duplicate() and ignore()
//////////////////////////////////////////////////////////////////////
// duplicate() duplicates the first argument of a functoid, whereas
// ignore() ignores it:
//    duplicate(f)(x)    = f(x)(x)
//    ignore(f)(x)(args) = f(args)

namespace impl {
template <class F>
class XDuplicater {
   F f;
public:
   XDuplicater( const F& ff ) : f(ff) {}

   template <class X>
   struct sig : public fun_type<typename RT<typename RT<F,X>::result_type,
                                             X>::result_type> {};
   template <class X>
   typename sig<X>::result_type
   operator()( const X& x ) const {
      return f(x)(x);
   }
};
struct XDuplicate {
   template <class F>
   struct sig : public fun_type<full1<XDuplicater<F> > > {};

   template <class F>
   full1<XDuplicater<F> > operator()( const F& f ) const {
      return make_full1( XDuplicater<F>(f) );
   }
};
}
typedef full1<impl::XDuplicate> duplicate_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN duplicate_type duplicate;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
template <class F>
class XIgnorer1 {
   F f;
public:
   XIgnorer1( const F& ff ) : f(ff) {}

   template <class X> 
   struct sig : public fun_type<typename RT<F>::result_type> {};

   template <class X>
   typename sig<X>::result_type operator()( const X& ) const {
      return f();
   }
};
template <class F>
class XIgnorer2 {
   F f;
public:
   XIgnorer2( const F& ff ) : f(ff) {}

   template <class X, class Y> 
   struct sig : public fun_type<typename RT<F,Y>::result_type> {};

   template <class X, class Y>
   typename sig<X,Y>::result_type 
   operator()( const X&, const Y& y ) const {
      return f(y);
   }
};
template <class F>
class XIgnorer3 {
   F f;
public:
   XIgnorer3( const F& ff ) : f(ff) {}

   template <class X, class Y, class Z> 
   struct sig : public fun_type<
      typename RT<F,Y,Z>::result_type> {};

   template <class X, class Y, class Z>
   typename sig<X,Y,Z>::result_type 
   operator()( const X&, const Y& y, const Z& z ) const {
      return f(y,z);
   }
};
class XIgnore {
   template<int n, class F> struct Helper;
   template<class F> struct Helper<0,F> {
      typedef full1< XIgnorer1<F> > Result;
      static inline Result go( const F& f ) {
         return make_full1( XIgnorer1<F>(f) );
      }
   };
   template<class F> struct Helper<1,F> {
      typedef full2< XIgnorer2<F> > Result;
      static inline Result go( const F& f ) {
         return make_full2( XIgnorer2<F>(f) );
      }
   };
   template<class F> struct Helper<2,F> {
      typedef full3< XIgnorer3<F> > Result;
      static inline Result go( const F& f ) {
         return make_full3( XIgnorer3<F>(f) );
      }
   };
public:
   template <class F>
   struct sig : public fun_type<
      typename Helper<functoid_traits<F>::max_args,F>::Result> {};

   template <class F>
   typename sig<F>::result_type operator()( const F& f ) const {
      return Helper<functoid_traits<F>::max_args,F>::go(f);
   }
};
}
typedef full1<impl::XIgnore> ignore_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN ignore_type ignore;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// ConstructN
//////////////////////////////////////////////////////////////////////
// C++ constructors are not functions, and thus cannot easily be turned
// into functoids.  So we write these helpers.  For example,
//    construct2<Foo>()(x,y) = Foo(x,y)    // Foo is a type name
// Note also that construct1 also serves the role of an explicit
// converter; if Foos (or any type) can be converted into Bars, then we 
// could use a construct1 functoid to capture the conversion function:
//    construct1<Bar>()       // functoid that converts arg into a Bar
//    construct1<Bar>()(x) = Bar(x)
// Note also that these are template functions returning full functoids,
// and we have template structs which name the types of the functoids.

namespace impl {
template <class T>
struct XConstruct0 : public c_fun_type<T> {
   T operator()() const { return T(); }
};
}
template <class T> full0<impl::XConstruct0<T> > construct0() 
{ return make_full0( impl::XConstruct0<T>() ); }
template <class T> struct construct0_type
{ typedef full0<impl::XConstruct0<T> > type; };

namespace impl {
template <class T>
struct XConstruct1 {
   template <class X> struct sig : fun_type<T> {};
   template <class X>
   T operator()( const X& x ) const { return T(x); }
};
}
template <class T> full1<impl::XConstruct1<T> > construct1() 
{ return make_full1( impl::XConstruct1<T>() ); }
template <class T> struct construct1_type
{ typedef full1<impl::XConstruct1<T> > type; };
   
namespace impl {
template <class T>
struct XConstruct2 {
   template <class X, class Y> struct sig : fun_type<T> {};
   template <class X, class Y>
   T operator()( const X& x, const Y& y ) const { return T(x,y); }
};
}
template <class T> full2<impl::XConstruct2<T> > construct2() 
{ return make_full2( impl::XConstruct2<T>() ); }
template <class T> struct construct2_type
{ typedef full2<impl::XConstruct2<T> > type; };
   
namespace impl {
template <class T>
struct XConstruct3 {
   template <class X, class Y, class Z> struct sig : fun_type<T> {};
   template <class X, class Y, class Z>
   T operator()( const X& x, const Y& y, const Z& z ) const { return T(x,y,z); }
};
}
template <class T> full3<impl::XConstruct3<T> > construct3() 
{ return make_full3( impl::XConstruct3<T>() ); }
template <class T> struct construct3_type
{ typedef full3<impl::XConstruct3<T> > type; };
   
//////////////////////////////////////////////////////////////////////
// NewN works like ConstructN but "new"s it and returns the ptr
//////////////////////////////////////////////////////////////////////
namespace impl {
template <class T>
struct XNew0 : public c_fun_type<T*> {
   T* operator()() const { return new T(); }
};
}
template <class T> full0<impl::XNew0<T> > new0() 
{ return make_full0( impl::XNew0<T>() ); }
template <class T> struct new0_type
{ typedef full0<impl::XNew0<T> > type; };

namespace impl {
template <class T>
struct XNew1 {
   template <class X> struct sig : fun_type<T*> {};
   template <class X>
   T* operator()( const X& x ) const { return new T(x); }
};
}
template <class T> full1<impl::XNew1<T> > new1() 
{ return make_full1( impl::XNew1<T>() ); }
template <class T> struct new1_type
{ typedef full1<impl::XNew1<T> > type; };
   
namespace impl {
template <class T>
struct XNew2 {
   template <class X, class Y> struct sig : fun_type<T*> {};
   template <class X, class Y>
   T* operator()( const X& x, const Y& y ) const { return new T(x,y); }
};
}
template <class T> full2<impl::XNew2<T> > new2() 
{ return make_full2( impl::XNew2<T>() ); }
template <class T> struct new2_type
{ typedef full2<impl::XNew2<T> > type; };
   
namespace impl {
template <class T>
struct XNew3 {
   template <class X, class Y, class Z> struct sig : fun_type<T*> {};
   template <class X, class Y, class Z>
   T* operator()( const X& x, const Y& y, const Z& z ) const 
   { return new T(x,y,z); }
};
}
template <class T> full3<impl::XNew3<T> > new3() 
{ return make_full3( impl::XNew3<T>() ); }
template <class T> struct new3_type
{ typedef full3<impl::XNew3<T> > type; };

} // end namespace fcpp
} // end namespace boost

#ifdef BOOST_FCPP_ENABLE_LAMBDA
#include "monad.hpp"
#endif

#endif
