// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_LIST_HPP
#define BOOST_FCPP_LIST_HPP

///////////////////////////////////////////////////////////////////////////
// Here we implement (lazy) lists in the list class.  There are also a
// number of functions associated with lists:
//  - head, tail, cons, cat, null, and the (in)equality operators
// We also have odd_list and strict_list.
///////////////////////////////////////////////////////////////////////////

// Order-of-initialization debugging help
// Note that you only might need this with the BOOST_FCPP_1_3_LIST_IMPL version
#ifdef BOOST_FCPP_OOI_DEBUG
#include <iostream>
#include <typeinfo>
#endif

#include <exception>
#include <new>
#include <cstdlib>
#include <vector>

#include "reuse.hpp"

namespace boost {
namespace fcpp {

struct fcpp_exception : public std::exception {
   const char* s;
   fcpp_exception( const char* ss ) : s(ss) {}
   const char* what() const throw() { return s; }
};

struct ListLike {};   // This lets us use is_base_and_derived() to see
   // (at compile-time) what classes are user-defined lists.
template <class T> class strict_list;

namespace impl {
struct XCons; struct XHead; struct XTail; struct XNull; class XCat;

struct CacheEmpty {};

template <class T> class Cache;
template <class T> class odd_list;
template <class T> class list_iterator;
template <class T, class It>
struct ListItHelp2 : public c_fun_type<It,It,odd_list<T> > {
   odd_list<T> operator()( It begin, const It& end, 
         reuser2<INV,VAR,INV,ListItHelp2,It,It> r = NIL ) const;
};
template <class U,class F> struct cvt;
template <class T, class F, class R> struct ListHelp;
template <class T> Cache<T>* xempty_helper();
template <class T, class F, class L, bool b> struct ConsHelp2;

struct ListRaw {};

template <class T> 
class list : public ListLike {
   boost::intrusive_ptr<Cache<T> > rep; // never NIL, unless an empty odd_list

   template <class U> friend class Cache;
   template <class U> friend class odd_list;
   template <class TT, class F, class L, bool b> friend struct ConsHelp2;
   template <class U,class F> friend struct cvt;

   list( const boost::intrusive_ptr<Cache<T> >& p ) : rep(p) { }
   list( ListRaw, Cache<T>* p ) : rep(p) { }

   bool priv_isEmpty() const { 
      return rep->cache().second.rep == Cache<T>::XNIL(); 
   }
   T priv_head() const { 
#ifdef BOOST_FCPP_DEBUG
      if( priv_isEmpty() )
         throw fcpp_exception("Tried to take head() of empty list");
#endif
      return rep->cache().first(); 
   }
   list<T> priv_tail() const { 
#ifdef BOOST_FCPP_DEBUG
      if( priv_isEmpty() )
         throw fcpp_exception("Tried to take tail() of empty list");
#endif
      return rep->cache().second; 
   }
public:
   static const bool is_lazy = true;

   typedef T value_type;
   typedef list tail_result_type;
   typedef odd_list<T> force_result_type;
   typedef list delay_result_type;
   template <class UU> struct cons_rebind { 
      typedef odd_list<UU> type; 
      typedef list<UU> delay_type; 
   };

   list( a_unique_type_for_nil ) : rep( Cache<T>::XEMPTY() ) { }
   list() : rep( Cache<T>::XEMPTY() ) { }

   template <class F>  // works on both ()->odd_list and ()->list 
   list( const F& f )
   : rep( ListHelp<T,F,typename F::result_type>()(f) ) { }

   // Note:  this constructor is still part of list and thus still lazy;
   // the iterators may not get evaluated until much later.  This is a
   // feature, not a bug.  So if the iterators are going to be invalidated
   // before you finish using the list, then you'd better force evaluation 
   // of the entire list before the iterators go away.
   template <class It>
   list( const It& begin, const It& end )
   : rep( new Cache<T>( thunk2(ListItHelp2<T,It>(),begin,end) ) ) { }

   list( const odd_list<T>& e )
   : rep( (e.second.rep != Cache<T>::XNIL()) ? 
          new Cache<T>(e) : Cache<T>::XEMPTY() ) { }

#ifdef BOOST_FCPP_SAFE_LIST
   // Long lists create long recursions of destructors that blow the
   // stack.  So we have an iterative destructor.  It is quite tricky to
   // get right.  The danger is that, when "bypassing" a node to be
   // unlinked and destructed, that node's 'next' pointer is, in fact, a
   // list object, whose destructor will be called.  As a result, as you
   // bypass a node, you need to see if its refC is down to 1, and if
   // so, mutate its next pointer so that when its destructor is called,
   // it won't cause a recursive cascade.  
   ~list() {
      while( rep != Cache<T>::XNIL() && rep != Cache<T>::XBAD() ) {
         if( rep->refC == 1 ) {
            // This is a rotate(), but this sequence is actually faster
            // than rotate(), so we do it explicitly
            boost::intrusive_ptr<Cache<T> > tmp( rep );
            rep = rep->val.second.rep;
            tmp->val.second.rep = Cache<T>::XNIL();
         }
         else
            rep = Cache<T>::XNIL();
      }
   }
#endif

   operator bool() const { return !priv_isEmpty(); }
   const force_result_type& force() const { return rep->cache(); }
   const delay_result_type& delay() const { return *this; }
   // Note: force returns a reference; implicit conversion now returns a copy.
   operator odd_list<T>() const { return force(); }

   // VC++7.1 says line below makes "return l;" (when l is a list and
   // function returns an odd_list) illegal, and I think it's right.
   //operator const odd_list<T>&() const { return force(); }

   T head() const { return priv_head(); }
   tail_result_type tail() const { return priv_tail(); }

   // The following helps makes list almost an STL "container"
   typedef list_iterator<T> const_iterator;
   typedef const_iterator iterator;         // list is immutable
   iterator begin() const { return list_iterator<T>( *this ); }
   iterator end() const   { return list_iterator<T>(); }
};

struct OddListDummyY {};

template <class T> 
class odd_list : public ListLike {
public:
   typedef 
      typename boost::type_with_alignment<boost::alignment_of<T>::value>::type
      xfst_type;
private:
   union { xfst_type fst; unsigned char dummy[sizeof(T)]; };

   const T& first() const { 
      return *static_cast<const T*>(static_cast<const void*>(&fst)); 
   }
   T& first() { 
      return *static_cast<T*>(static_cast<void*>(&fst));
   }
   list<T>  second;   // If XNIL, then this odd_list is NIL

   template <class U> friend class list;
   template <class U> friend class Cache;

   odd_list( OddListDummyY )
   : second( Cache<T>::XBAD() ) { }

   void init( const T& x ) {
      new (static_cast<void*>(&fst)) T(x);
   } 

   bool fst_is_valid() const {
      if( second.rep != Cache<T>::XNIL() )
         if( second.rep != Cache<T>::XBAD() )
            return true;
      return false;
   }

   bool priv_isEmpty() const { return second.rep == Cache<T>::XNIL(); }
   T priv_head() const { 
#ifdef BOOST_FCPP_DEBUG
      if( priv_isEmpty() )
         throw fcpp_exception("Tried to take head() of empty odd_list");
#endif
      return first(); 
   }
   list<T> priv_tail() const { 
#ifdef BOOST_FCPP_DEBUG
      if( priv_isEmpty() )
         throw fcpp_exception("Tried to take tail() of empty odd_list");
#endif
      return second; 
   }

public:
   static const bool is_lazy = true;

   typedef T value_type;
   typedef list<T> tail_result_type;
   typedef odd_list<T> force_result_type;
   typedef list<T> delay_result_type;
   template <class UU> struct cons_rebind { 
      typedef odd_list<UU> type; 
      typedef list<UU> delay_type; 
   };

   odd_list() : second( Cache<T>::XNIL() ) { }
   odd_list( a_unique_type_for_nil ) : second( Cache<T>::XNIL() ) { }
   odd_list( const T& x, const list<T>& y ) : second(y) { init(x); }
   odd_list( const T& x, a_unique_type_for_nil ) : second(NIL) { init(x); }

   odd_list( const odd_list<T>& x ) : second(x.second) {
      if( fst_is_valid() ) {
         init( x.first() );
      }
   }

   template <class It>
   odd_list( It begin, const It& end ) 
   : second( begin==end ? Cache<T>::XNIL() :
             ( init(*begin++), list<T>( begin, end ) ) ) {}

   odd_list<T>& operator=( const odd_list<T>& x ) {
      if( this == &x ) return *this;  
      if( fst_is_valid() ) {
         if( x.fst_is_valid() )
            first() = x.first();
         else
            first().~T();
      }
      else {
         if( x.fst_is_valid() )
            init( x.first() );
      }
      second = x.second;
      return *this;
   }
      
   ~odd_list() {
      if( fst_is_valid() ) {
         first().~T(); 
      }
   }

   operator bool() const { return !priv_isEmpty(); }
   const force_result_type& force() const { return *this; }
   delay_result_type delay() const { return list<T>(*this); }

   T head() const { return priv_head(); }
   tail_result_type tail() const { return priv_tail(); }

   // The following helps makes odd_list almost an STL "container"
   typedef list_iterator<T> const_iterator;
   typedef const_iterator iterator;         // odd_list is immutable
   iterator begin() const { return list_iterator<T>( this->delay() ); }
   iterator end() const   { return list_iterator<T>(); }
};

// This converts ()->list<T> to ()->odd_list<T>.
// In other words, here is the 'extra work' done when using the
// unoptimized interface.
template <class U,class F>
struct cvt : public c_fun_type<odd_list<U> > {
   F f;
   cvt( const F& ff ) : f(ff) {}
   odd_list<U> operator()() const {
      list<U> l = f();
      return l.force();
   }
};

// I malloc a RefCountType to hold the refCount and init it to 1 to ensure the
// refCount will never get to 0, so the destructor-of-global-object
// order at the end of the program is a non-issue.  In other words, the
// memory allocated here is only reclaimed by the operating system.
template <class T> 
Cache<T>* xnil_helper() {
   void *p = std::malloc( sizeof(RefCountType) );
   *((RefCountType*)p) = 1;
   return static_cast<Cache<T>*>( p );
}
template <class T> 
Cache<T>* xnil_helper_nil() {
   Cache<T>* p = xnil_helper<T>();
#ifdef BOOST_FCPP_OOI_DEBUG
   std::cout << "making xnil:   " << typeid(T).name() 
             << " at address " << p << std::endl;
#endif
   return p;
}
template <class T> 
Cache<T>* xnil_helper_bad() {
   Cache<T>* p = xnil_helper<T>();
#ifdef BOOST_FCPP_OOI_DEBUG
   std::cout << "making xbad:   " << typeid(T).name() 
             << " at address " << p << std::endl;
#endif
   return p;
}

template <class T> 
Cache<T>* xempty_helper() {
#ifdef BOOST_FCPP_1_3_LIST_IMPL
   (void) Cache<T>::xnil;   // Make sure xnil exists before moving forward
#endif
#ifdef BOOST_FCPP_OOI_DEBUG
   std::cout << "about to make xempty: " << typeid(T).name() << std::endl;
#endif
   Cache<T>* p = new Cache<T>( CacheEmpty() );
#ifdef BOOST_FCPP_OOI_DEBUG
   std::cout << "making xempty: " << typeid(T).name() 
             << " at address " << p << std::endl;
#endif
   return p;
}

template <class T> 
class Cache : boost::noncopyable {
   mutable RefCountType refC;
   mutable fun0<odd_list<T> >   fxn;
   mutable odd_list<T>          val;
   // val.second.rep can be XBAD, XNIL, or a valid ptr
   //  - XBAD: val is invalid (fxn is valid)
   //  - XNIL: this is the empty list
   //  - anything else: val.first() is head, val.second is tail()

   // This functoid should never be called; it represents a
   // self-referent Cache, which should be impossible under the current
   // implementation.  Nonetheless, we need a 'dummy' function object to
   // represent invalid 'fxn's (val.second.rep!=XBAD), and this
   // implementation seems to be among the most reasonable.
   struct blackhole_helper : c_fun_type< odd_list<T> > {
      odd_list<T> operator()() const {
         throw fcpp_exception("You have entered a black hole.");
      }
   };
#ifdef BOOST_FCPP_1_3_LIST_IMPL
   static boost::intrusive_ptr<Cache<T> > xnil, xbad;
   static boost::intrusive_ptr<Cache<T> > xempty;
#endif

   // Don't get rid of these XFOO() functions; they impose no overhead,
   // and provide a useful place to add debugging code for tracking down
   // before-main()-order-of-initialization problems.
   static const boost::intrusive_ptr<Cache<T> >& XEMPTY() {
#ifndef BOOST_FCPP_1_3_LIST_IMPL
      static boost::intrusive_ptr<Cache<T> > xempty( xempty_helper<T>() );
#endif
#ifdef BOOST_FCPP_OOI_DEBUG
      static bool b = true;
      if(b) {
         std::cout << "access xempty: " << typeid(T).name() << std::endl;
         b = false;
      }
#endif
      return xempty;
   }
   static const boost::intrusive_ptr<Cache<T> >& XNIL() {    // this list is nil
#ifndef BOOST_FCPP_1_3_LIST_IMPL
      static boost::intrusive_ptr<Cache<T> > xnil( xnil_helper_nil<T>() );
#endif
#ifdef BOOST_FCPP_OOI_DEBUG
      static bool b = true;
      if(b) {
         std::cout << "access xnil:   " << typeid(T).name() << std::endl;
         b = false;
      }
#endif
      return xnil;
   }
   static const boost::intrusive_ptr<Cache<T> >& XBAD() {    // the pair is invalid; use fxn
#ifndef BOOST_FCPP_1_3_LIST_IMPL
      static boost::intrusive_ptr<Cache<T> > xbad( xnil_helper_bad<T>() );
#endif
#ifdef BOOST_FCPP_OOI_DEBUG
      static bool b = true;
      if(b) {
         std::cout << "access xbad:   " << typeid(T).name() << std::endl;
         b = false;
      }
#endif
      return xbad;
   }
   static fun0<odd_list<T> > the_blackhole;
   static fun0<odd_list<T> >& blackhole() {
#ifndef BOOST_FCPP_1_3_LIST_IMPL
      static fun0<odd_list<T> > the_blackhole( make_fun0( blackhole_helper() ) );
#endif
      return the_blackhole;
   }

   odd_list<T>& cache() const {
      if( val.second.rep == XBAD() ) {
         val = fxn();
         fxn = blackhole();
      }
      return val;
   }

   template <class U> friend class list;
   template <class U> friend class odd_list;
   template <class TT, class F, class L, bool b> friend struct ConsHelp2;
   template <class U,class F> friend struct cvt;
   template <class U, class F, class R> friend struct ListHelp;
   template <class U> friend Cache<U>* xempty_helper();

   Cache( CacheEmpty ) : refC(0), fxn(blackhole()), val() {}
   Cache( const odd_list<T>& x ) : refC(0), fxn(blackhole()), val(x) {}
   Cache( const T& x, const list<T>& l ) : refC(0),fxn(blackhole()),val(x,l) {}

   Cache( const fun0<odd_list<T> >& f )
   : refC(0), fxn(f), val( OddListDummyY() ) {}

   template <class F>
   Cache( const F& f )    // ()->odd_list
   : refC(0), fxn(make_fun0(f)), val( OddListDummyY() ) {}

   // This is for ()->list<T> to ()->odd_list<T>
   struct CvtFxn {};
   template <class F>
   Cache( CvtFxn, const F& f )    // ()->list
   : refC(0), fxn(make_fun0(cvt<T,F>(f))), val( OddListDummyY() ) {}

   template <class X>
   friend void intrusive_ptr_add_ref( const Cache<X>* p );
   template <class X>
   friend void intrusive_ptr_release( const Cache<X>* p );
};
template <class T>
void intrusive_ptr_add_ref( const Cache<T>* p ) {
   ++ (p->refC);
}
template <class T>
void intrusive_ptr_release( const Cache<T>* p ) {
   if( !--(p->refC) ) delete p;
}

#ifdef BOOST_FCPP_1_3_LIST_IMPL
template <class T>
fun0<odd_list<T> > Cache<T>::the_blackhole( make_fun0( blackhole_helper() ) );

template <class T> boost::intrusive_ptr<Cache<T> > 
Cache<T>::xnil( xnil_helper_nil<T>() );
template <class T> boost::intrusive_ptr<Cache<T> > 
Cache<T>::xbad( xnil_helper_bad<T>() );
template <class T> boost::intrusive_ptr<Cache<T> > 
Cache<T>::xempty( xempty_helper<T>() );
#endif

// Rest of list's stuff

template <class T, class F> struct ListHelp<T,F,list<T> > {
   boost::intrusive_ptr<Cache<T> > operator()( const F& f ) const {
      return boost::intrusive_ptr<Cache<T> >
         (new Cache<T>(Cache<T>::CvtFxn(),f));
   }
};
template <class T, class F> struct ListHelp<T,F,odd_list<T> > {
   boost::intrusive_ptr<Cache<T> > operator()( const F& f ) const {
      return boost::intrusive_ptr<Cache<T> >(new Cache<T>(f));
   }
};

template <class T>
class list_iterator 
: public std::iterator<std::input_iterator_tag,T,ptrdiff_t> {
   list<T> l;
   bool is_nil;
   void advance() {
      l = l.tail();
      if( !l )
         is_nil = true;
   }
   class Proxy {  // needed for operator->
      const T x;
      friend class list_iterator;
      Proxy( const T& xx ) : x(xx) {}
   public:
      const T* operator->() const { return &x; }
   };
public:
   list_iterator() : l(), is_nil(true) {}
   explicit list_iterator( const list<T>& ll ) : l(ll), is_nil(!ll) {}
   
   const T operator*() const { return l.head(); }
   const Proxy operator->() const { return Proxy(l.head()); }
   list_iterator<T>& operator++() {
      advance();
      return *this;
   }
   const list_iterator<T> operator++(int) {
      list_iterator<T> i( *this );
      advance();
      return i;
   }
   bool operator==( const list_iterator<T>& i ) const {
      return is_nil && i.is_nil;
   }
   bool operator!=( const list_iterator<T>& i ) const {
      return ! this->operator==(i);
   }
};
}

using impl::list;
using impl::odd_list;
using impl::list_iterator;

//////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////

/*
concept ListLike: Given a list representation type L

L<T> inherits ListLike and has
   // typedefs just show typical values
   typedef T value_type
   typedef L<T> force_result_type
   typedef L<T> delay_result_type
   typedef L<T> tail_result_type
   template <class UU> struct cons_rebind { 
      typedef L<UU> type;        // force type
      typedef L<UU> delay_type;  // delay type
   };

   L()
   L( a_unique_type_for_nil )
   template <class F> L(F)       // F :: ()->L

   constructor: force_result_type( T, L<T> )
   template <class F>
   constructor: force_result_type( T, F )  // F :: ()->L

   template <class It>
   L( It, It )

   // FIX THIS instead of operator bool(), does Boost have something better?
   operator bool() const
   force_result_type force() const
   delay_result_type delay() const
   T head() const
   tail_result_type tail() const

   static const bool is_lazy;   // true if can represent infinite lists

   typedef const_iterator;
   typedef const_iterator iterator;  // ListLikes are immutable
   iterator begin() const
   iterator end() const 
*/

template <class L, bool is_lazy> struct ensure_lazy_helper {};
template <class L> struct ensure_lazy_helper<L,true> {
   static void requires_lazy_list_to_prevent_infinite_recursion() {}
};
template <class L>
void ensure_lazy() {
   ensure_lazy_helper<L,L::is_lazy>::
      requires_lazy_list_to_prevent_infinite_recursion();
}

//////////////////////////////////////////////////////////////////////

template <class L, bool b>
struct EnsureListLikeHelp {
   static void trying_to_call_a_list_function_on_a_non_list() {}
};
template <class L> struct EnsureListLikeHelp<L,false> { };
template <class L>
void EnsureListLike() {
   EnsureListLikeHelp<L,boost::is_base_and_derived<ListLike,L>::value>::
   trying_to_call_a_list_function_on_a_non_list();
}

//////////////////////////////////////////////////////////////////////

namespace impl{
struct XHead {
   template <class L>
   struct sig : public fun_type<typename L::value_type> {};

   template <class L>
   typename sig<L>::result_type operator()( const L& l ) const {
      EnsureListLike<L>();
      return l.head();
   }
};
}
typedef full1<impl::XHead> head_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN head_type head;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XTail {
   template <class L>
   struct sig : public fun_type<typename L::tail_result_type> {};

   template <class L>
   typename sig<L>::result_type 
   operator()( const L& l ) const {
      EnsureListLike<L>();
      return l.tail();
   }
};
}
typedef full1<impl::XTail> tail_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN tail_type tail;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XNull {
   template <class L>
   struct sig : public fun_type<bool> {};

   template <class L>
   bool operator()( const L& l ) const {
      EnsureListLike<L>();
      return !l;
   }
};
}
typedef full1<impl::XNull> null_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN null_type null;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
template <class T, class F, class L> struct ConsHelp2<T,F,L,true> {
   typedef typename L::force_result_type type;
   static type go( const T& x, const F& f ) {
      return type( x, f );
   }
};
template <class T, class F> struct ConsHelp2<T,F,list<T>,true> {
   typedef list<T> L;
   typedef typename L::force_result_type type;
   static type go( const T& x, const F& f ) {
      return odd_list<T>(x, list<T>(
         boost::intrusive_ptr<Cache<T> >(new Cache<T>(Cache<T>::CvtFxn(),f))));
   }
};
template <class T, class F> struct ConsHelp2<T,F,odd_list<T>,true> {
   typedef odd_list<T> L;
   typedef typename L::force_result_type type;
   static type go( const T& x, const F& f ) {
      return odd_list<T>(x, list<T>( ListRaw(), new Cache<T>(f) ));
   }
};
template <class T, class F> struct ConsHelp2<T,F,a_unique_type_for_nil,false> {
   typedef odd_list<T> type;
   static type go( const T& x, const F& f ) {
      return odd_list<T>(x, list<T>( ListRaw(), new Cache<T>(f) ));
   }
};
template <class T, class L, bool b> struct ConsHelp1 {
   typedef typename L::force_result_type type;
   static type go( const T& x, const L& l ) {
      return type(x,l);
   }
};
template <class T> struct ConsHelp1<T,a_unique_type_for_nil,false> {
   typedef odd_list<T> type;
   static type go( const T& x, const a_unique_type_for_nil& n ) {
      return type(x,n);
   }
};
template <class T, class F> struct ConsHelp1<T,F,false> {
   // It's a function returning a list
   typedef typename F::result_type L;
   typedef ConsHelp2<T,F,L,boost::is_base_and_derived<ListLike,L>::value> help;
   typedef typename help::type type;
   static type go( const T& x, const F& f ) {
      return help::go(x,f);
   }
};
struct XCons {
   template <class T, class L> struct sig : public fun_type<
      typename ConsHelp1<T,L,
      boost::is_base_and_derived<ListLike,L>::value>::type> {};

   template <class T, class L>
   typename sig<T,L>::result_type operator()( const T& x, const L& l ) const {
      typedef typename sig<T,L>::result_type LL;
      typedef ConsHelp1<T,L,
         boost::is_base_and_derived<ListLike,L>::value> help;
      return help::go(x,l);
   }
};
}
typedef full2<impl::XCons> cons_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN cons_type cons;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl{
   template <class T, class It>
   odd_list<T> ListItHelp2<T,It>::operator()( It begin, const It& end, 
            reuser2<INV,VAR,INV,ListItHelp2,It,It> r ) const {
      if( begin == end )
         return NIL;
      else {
         T x( *begin );
         return cons( x, r( ListItHelp2<T,It>(), ++begin, end ) );
      }
   }
}

namespace impl {
class XCat {
   template <class L, class M, bool b, class R>
   struct Helper : public c_fun_type<L,M,R> {
      R operator()( const L& l, const M& m, 
             reuser2<INV,VAR,INV,Helper,
                     typename RT<tail_type,L>::result_type,M>
             r = NIL ) const {
         if( null(l) ) 
            return m().force();
         else
            return cons( head(l), r( *this, tail(l), m ) );
      }
   };
   template <class L, class M, class R>
   struct Helper<L,M,true,R> : public c_fun_type<L,M,R> {
      R operator()( const L& l, const M& m,
             reuser2<INV,VAR,INV,Helper,
                     typename RT<tail_type,L>::result_type,M>
             r = NIL ) const {
         if( null(l) ) 
            return m.force();
         else
            return cons( head(l), r( *this, tail(l), m ) );
      }
   };
   template <class L, class R>
   struct Helper<L,a_unique_type_for_nil,false,R> 
   : public c_fun_type<L,
                  a_unique_type_for_nil,odd_list<typename L::value_type> > {
      odd_list<typename L::value_type> 
      operator()( const L& l, const a_unique_type_for_nil& ) const {
         return l;
      }
   };
public:
   template <class L, class M> struct sig : public fun_type<
      typename RT<cons_type,typename L::value_type,M>::result_type> {};

   // Note: first arg must be a list, but second arg can be either a list 
   // or a function that returns a list.
   template <class L, class M>
   typename sig<L,M>::result_type operator()( const L& l, const M& m ) const {
      EnsureListLike<L>();
      return Helper<L,M,boost::is_base_and_derived<ListLike,M>::value,
         typename sig<L,M>::result_type>()(l,m);
   }
};
}
typedef full2<impl::XCat> cat_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN cat_type cat;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XDelay {
   template <class L>
   struct sig : public fun_type<typename L::delay_result_type> {};

   template <class L>
   typename sig<L>::result_type operator()( const L& l ) const {
      EnsureListLike<L>();
      return l.delay();
   }
};
}
typedef full1<impl::XDelay> delay_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN delay_type delay;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

namespace impl {
struct XForce {
   template <class L>
   struct sig : public fun_type<typename L::force_result_type> {};

   template <class L>
   typename sig<L>::result_type operator()( const L& l ) const {
      EnsureListLike<L>();
      return l.force();
   }
};
}
typedef full1<impl::XForce> force_type;
BOOST_FCPP_MAYBE_NAMESPACE_OPEN
BOOST_FCPP_MAYBE_EXTERN force_type force;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

//////////////////////////////////////////////////////////////////////
// op== and op<, overloaded for all combos of list, odd_list, and NIL
//////////////////////////////////////////////////////////////////////

// FIX THIS comparison operators can be implemented simpler with enable_if
template <class T>
bool operator==( const odd_list<T>& a, a_unique_type_for_nil ) {
   return null(a);
}
template <class T>
bool operator==( const list<T>& a, a_unique_type_for_nil ) {
   return null(a);
}
template <class T>
bool operator==( a_unique_type_for_nil, const odd_list<T>& a ) {
   return null(a);
}
template <class T>
bool operator==( a_unique_type_for_nil, const list<T>& a ) {
   return null(a);
}
template <class T>
bool operator==( const list<T>& a, const list<T>& b ) {
   if( null(a) && null(b) )
      return true;
   if( null(a) || null(b) )
      return false;
   return (head(a)==head(b)) && (tail(a)==tail(b));
}
template <class T>
bool operator==( const odd_list<T>& a, const odd_list<T>& b ) {
   if( null(a) && null(b) )
      return true;
   if( null(a) || null(b) )
      return false;
   return (head(a)==head(b)) && (tail(a)==tail(b));
}
template <class T>
bool operator==( const list<T>& a, const odd_list<T>& b ) {
   if( null(a) && null(b) )
      return true;
   if( null(a) || null(b) )
      return false;
   return (head(a)==head(b)) && (tail(a)==tail(b));
}
template <class T>
bool operator==( const odd_list<T>& a, const list<T>& b ) {
   if( null(a) && null(b) )
      return true;
   if( null(a) || null(b) )
      return false;
   return (head(a)==head(b)) && (tail(a)==tail(b));
}

template <class T>
bool operator<( const list<T>& a, const list<T>& b ) {
   if( null(a) && !null(b) )  return true;
   if( null(b) )              return false;
   if( head(b) < head(a) )    return false;
   if( head(a) < head(b) )    return true;
   return (tail(a) < tail(b));
}
template <class T>
bool operator<( const odd_list<T>& a, const list<T>& b ) {
   if( null(a) && !null(b) )  return true;
   if( null(b) )              return false;
   if( head(b) < head(a) )    return false;
   if( head(a) < head(b) )    return true;
   return (tail(a) < tail(b));
}
template <class T>
bool operator<( const list<T>& a, const odd_list<T>& b ) {
   if( null(a) && !null(b) )  return true;
   if( null(b) )              return false;
   if( head(b) < head(a) )    return false;
   if( head(a) < head(b) )    return true;
   return (tail(a) < tail(b));
}
template <class T>
bool operator<( const odd_list<T>& a, const odd_list<T>& b ) {
   if( null(a) && !null(b) )  return true;
   if( null(b) )              return false;
   if( head(b) < head(a) )    return false;
   if( head(a) < head(b) )    return true;
   return (tail(a) < tail(b));
}
template <class T>
bool operator<( const odd_list<T>&, a_unique_type_for_nil ) {
   return false;
}
template <class T>
bool operator<( const list<T>&, a_unique_type_for_nil ) {
   return false;
}
template <class T>
bool operator<( a_unique_type_for_nil, const odd_list<T>& b ) {
   return !null(b);
}
template <class T>
bool operator<( a_unique_type_for_nil, const list<T>& b ) {
   return !null(b);
}

//////////////////////////////////////////////////////////////////////
// Handy functions for making list literals
//////////////////////////////////////////////////////////////////////
// Yes, these aren't functoids, they're just template functions.  I'm
// lazy and created these mostly to make it easily to make little lists
// in the sample code snippets that appear in papers.

struct UseList {
   template <class T> struct List { typedef list<T> type; };
};
struct UseOddList {
   template <class T> struct List { typedef odd_list<T> type; };
};
struct UseStrictList {
   template <class T> struct List { typedef strict_list<T> type; };
};

template <class Kind = UseList>
struct list_with {
   template <class T>
   typename Kind::template List<T>::type
   operator()( const T& a ) const {
      typename Kind::template List<T>::type l;
      l = cons( a, l );
      return l;
   }
   
   template <class T>
   typename Kind::template List<T>::type
   operator()( const T& a, const T& b ) const {
      typename Kind::template List<T>::type l;
      l = cons( b, l );
      l = cons( a, l );
      return l;
   }
   
   template <class T>
   typename Kind::template List<T>::type
   operator()( const T& a, const T& b, const T& c ) const {
      typename Kind::template List<T>::type l;
      l = cons( c, l );
      l = cons( b, l );
      l = cons( a, l );
      return l;
   }
   
   template <class T>
   typename Kind::template List<T>::type
   operator()( const T& a, const T& b, const T& c, const T& d ) const {
      typename Kind::template List<T>::type l;
      l = cons( d, l );
      l = cons( c, l );
      l = cons( b, l );
      l = cons( a, l );
      return l;
   }
   
   template <class T>
   typename Kind::template List<T>::type
   operator()( const T& a, const T& b, const T& c, const T& d, 
               const T& e ) const {
      typename Kind::template List<T>::type l;
      l = cons( e, l );
      l = cons( d, l );
      l = cons( c, l );
      l = cons( b, l );
      l = cons( a, l );
      return l;
   }
};

//////////////////////////////////////////////////////////////////////
// Strict lists
//////////////////////////////////////////////////////////////////////

namespace impl {
template <class T>
struct strict_cons : public boost::noncopyable {
   mutable RefCountType refC;
   T head;
   typedef boost::intrusive_ptr<strict_cons> tail_type;
   tail_type tail;
   strict_cons( const T& h, const tail_type& t ) : refC(0), head(h), tail(t) {}
};
template <class T>
void intrusive_ptr_add_ref( const strict_cons<T>* p ) {
   ++ (p->refC);
}
template <class T>
void intrusive_ptr_release( const strict_cons<T>* p ) {
   if( !--(p->refC) ) delete p;
}

template <class T>
class strict_list_iterator 
: public std::iterator<std::input_iterator_tag,T,ptrdiff_t> {
   typedef boost::intrusive_ptr<strict_cons<T> > rep_type;
   rep_type l;
   bool is_nil;
   void advance() {
      l = l->tail;
      if( !l )
         is_nil = true;
   }
   class Proxy {  // needed for operator->
      const T x;
      friend class strict_list_iterator;
      Proxy( const T& xx ) : x(xx) {}
   public:
      const T* operator->() const { return &x; }
   };
public:
   strict_list_iterator() : l(), is_nil(true) {}
   explicit strict_list_iterator( const rep_type& ll ) : l(ll), is_nil(!ll) {}
   
   const T operator*() const { return l->head; }
   const Proxy operator->() const { return Proxy(l->head); }
   strict_list_iterator<T>& operator++() {
      advance();
      return *this;
   }
   const strict_list_iterator<T> operator++(int) {
      strict_list_iterator<T> i( *this );
      advance();
      return i;
   }
   bool operator==( const strict_list_iterator<T>& i ) const {
      return is_nil && i.is_nil;
   }
   bool operator!=( const strict_list_iterator<T>& i ) const {
      return ! this->operator==(i);
   }
};
}

template <class T>
class strict_list : public ListLike {
   typedef boost::intrusive_ptr<impl::strict_cons<T> > rep_type;
   rep_type rep;
   struct Make {};

   template <class Iter>
   static rep_type help( Iter a, const Iter& b ) {
      rep_type r;
      while( a != b ) {
         T x( *a );
         r = rep_type( new impl::strict_cons<T>( x, r ) );
         ++a;
      }
      return r;
   }
public:
   static const bool is_lazy = false;

   typedef T value_type;
   typedef strict_list force_result_type;
   typedef strict_list delay_result_type;
   typedef strict_list tail_result_type;
   template <class UU> struct cons_rebind { 
      typedef strict_list<UU> type; 
      typedef strict_list<UU> delay_type; 
   };

   strict_list( Make, const rep_type& r ) : rep(r) {}

   strict_list() : rep() {}

   strict_list( a_unique_type_for_nil ) : rep() {}

   template <class F>
   strict_list( const F& f ) : rep( f().rep ) {
      functoid_traits<F>::template ensure_accepts<0>::args();
   }

   strict_list( const T& x, const strict_list& y ) 
      : rep( new impl::strict_cons<T>(x,y.rep) ) {}

   template <class F>
   strict_list( const T& x, const F& f ) 
      : rep( new impl::strict_cons<T>(x,f().rep) ) {}
   
   operator bool() const { return rep; }
   force_result_type force() const { return *this; }
   delay_result_type delay() const { return *this; }
   T head() const { 
#ifdef BOOST_FCPP_DEBUG
      if( !*this )
         throw fcpp_exception("Tried to take head() of empty strict_list");
#endif
      return rep->head; 
   }
   tail_result_type tail() const { 
#ifdef BOOST_FCPP_DEBUG
      if( !*this )
         throw fcpp_exception("Tried to take tail() of empty strict_list");
#endif
      return strict_list(Make(),rep->tail); 
   }

   template <class Iter>
   strict_list( const Iter& a, const Iter& b ) : rep( rep_type() ) {
      // How ironic.  We need to reverse the iterator range in order to
      // non-recursively build this!
      std::vector<T> tmp(a,b);
      rep = help( tmp.rbegin(), tmp.rend() );
   }

   // Since the strict_cons destructor can't call the strict_list
   // destructor, the "simple" iterative destructor is correct and
   // efficient.  Hurray.
   ~strict_list() { while(rep && (rep->refC == 1)) rep = rep->tail; }

   // The following helps makes strict_list almost an STL "container"
   typedef impl::strict_list_iterator<T> const_iterator;
   typedef const_iterator iterator;         // strict_list is immutable
   iterator begin() const { return impl::strict_list_iterator<T>( rep ); }
   iterator end() const   { return impl::strict_list_iterator<T>(); }
};

template <class T>
bool operator==( const strict_list<T>& a, a_unique_type_for_nil ) {
   return null(a);
}
template <class T>
bool operator==( a_unique_type_for_nil, const strict_list<T>& a ) {
   return null(a);
}
template <class T>
bool operator==( const strict_list<T>& a, const strict_list<T>& b ) {
   if( null(a) && null(b) )
      return true;
   if( null(a) || null(b) )
      return false;
   return (head(a)==head(b)) && (tail(a)==tail(b));
}

template <class T>
bool operator<( const strict_list<T>& a, const strict_list<T>& b ) {
   if( null(a) && !null(b) )  return true;
   if( null(b) )              return false;
   if( head(b) < head(a) )    return false;
   if( head(a) < head(b) )    return true;
   return (tail(a) < tail(b));
}
template <class T>
bool operator<( const strict_list<T>&, a_unique_type_for_nil ) {
   return false;
}
template <class T>
bool operator<( a_unique_type_for_nil, const strict_list<T>& b ) {
   return !null(b);
}

} // namespace fcpp
} // namespace boost

#endif
