// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_FUNCTION_HPP
#define BOOST_FCPP_FUNCTION_HPP

//////////////////////////////////////////////////////////////////////////
// Here is where funN and FunNImpl classes are declared.  Also here are
//  - convertN    for implicit conversions (subtype polymorphism)
//  - explicit_convertN   like convertN, but uses casts (non-implicit)
// Note that makeFunN has been deprecated and removed.
//////////////////////////////////////////////////////////////////////////

#include "ref_count.hpp"
#include "operator.hpp"

namespace boost {
namespace fcpp {

//////////////////////////////////////////////////////////////////////
// Ok, this file has been a mess, so I'm trying to clean it up.  The
// file is divided into 4 sections, for fun0, fun1, fun2, and fun3.
// The sections are all pretty redundant, except that
//  - fun2 implements its own currying (it was written before curryables were)
//  - fun3 uses curryable3 to implement its currying
// As a result, I'm removing some of the redundant comments from all the
// sections except fun0.  So basically, use the comments in the fun0
// section as a reference for the corresponding structures in the other
// funN classes.
//////////////////////////////////////////////////////////////////////

template <class Result>
class Fun0Impl;

template <class Rd, class Rs>   // result of dest, result of src
Fun0Impl<Rd>* convert0( const boost::intrusive_ptr<const Fun0Impl<Rs> >& f );

template <class Rd, class DF>
struct Fun0Constructor;

template <class Result>
class fun0 
: public c_fun_type<Result>, public smart_functoid0 {
   typedef boost::intrusive_ptr<const Fun0Impl<Result> > RefImpl;

   RefImpl ref;
   template <class T> friend class fun0; 
   template <class Rd, class Rs>
   friend fun0<Rd> explicit_convert0( const fun0<Rs>& f );

   template <class Rd, class DF>
   friend struct Fun0Constructor;
public:
   typedef const Fun0Impl<Result>* Impl;
   // int is dummy arg to differentiate from the template constructor
   fun0( int, Impl i ) : ref(i) {}

   Result operator()() const { return ref->operator()(); }

   template <class DF>   // direct functoid (or subtype polymorphism)
   fun0( const DF& f ) : ref( Fun0Constructor<Result,DF>::make(f) ) {}

   fun0( const fun0& x ) : ref(x.ref) {}
   fun0& operator=( const fun0& x ) { ref = x.ref; return *this; }
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef fun0 This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif
};

template <class Result>
class Fun0Impl : public c_fun_type<Result> {
   mutable RefCountType refC;
public:
   Fun0Impl() : refC(0) {}
   virtual Result operator()() const =0;
   virtual ~Fun0Impl() {}

   template <class X>
   friend void intrusive_ptr_add_ref( const Fun0Impl<X>* p );
   template <class X>
   friend void intrusive_ptr_release( const Fun0Impl<X>* p );
};
template <class T> 
void intrusive_ptr_add_ref( const Fun0Impl<T>* p ) {
   ++ (p->refC);
}  
template <class T> 
void intrusive_ptr_release( const Fun0Impl<T>* p ) {
   if( !--(p->refC) ) delete p;
}  

template <class Rd, class Rs>
class Fun0Converter : public Fun0Impl<Rd> {
   typedef boost::intrusive_ptr<const Fun0Impl<Rs> > MyFun;
   MyFun f;
public:
   Fun0Converter( const MyFun& g ) : f(g) {}
   Rd operator()() const {
      return f->operator()();
   }
};

template <class Rd, class Rs>
Fun0Impl<Rd>* convert0( const boost::intrusive_ptr<const Fun0Impl<Rs> >& f ) {
   return new Fun0Converter<Rd,Rs>( f );
}

template <class Rd, class Rs>
class Fun0ExplicitConverter : public Fun0Impl<Rd> {
   typedef boost::intrusive_ptr<const Fun0Impl<Rs> > MyFun;
   MyFun f;
public:
   Fun0ExplicitConverter( const MyFun& g ) : f(g) {}
   Rd operator()() const {
      return static_cast<Rd>( f->operator()() );
   }
};

template <class Rd, class Rs>
fun0<Rd> explicit_convert0( const fun0<Rs>& f ) {
   return fun0<Rd>( 1, new Fun0ExplicitConverter<Rd,Rs>( f.ref ) );
}

template <class Gen>
class Gen0 : public Fun0Impl<typename RT<Gen>::result_type> {
   Gen g;
public:
   Gen0( Gen x ) : g(x) {}
   typename RT<Gen>::result_type operator()() const { return g(); }
};
   
template <class Gen>
fun0<typename RT<Gen>::result_type> make_fun0( const Gen& g ) {
   return fun0<typename RT<Gen>::result_type>(1,new Gen0<Gen>(g));
}

template <class Nullary>
Gen0<Nullary>* makeFun0Ref( const Nullary& g ) {
   return new Gen0<Nullary>(g);
}

// Note: conversion-from-direct-functoid and subtype-polymorphism are
// really two different kinds of functionality.  However, if we try to
// create two separate constructors in the fun0 class, we end up with
// ambiguity (C++ can't tell which one to call).  As a result, there is
// one constructor that handles both cases by forwarding the work to
// this class, which uses partial specialization to distinguish between
// the two cases.
template <class Rd, class DF>
struct Fun0Constructor {
   static Fun0Impl<Rd>* make( const DF& df ) {
      return makeFun0Ref( monomorphize0<Rd>(df) );
   }
};
template <class Rd, class Rs>
struct Fun0Constructor<Rd,fun0<Rs> > {
   static Fun0Impl<Rd>* make( const fun0<Rs>& f ) {
      return convert0<Rd>(f.ref);
   }
};

//////////////////////////////////////////////////////////////////////

template <class Arg1, class Result>
class Fun1Impl;

template <class A1d, class Rd, class A1s, class Rs> 
Fun1Impl<A1d,Rd>* 
convert1( const boost::intrusive_ptr<const Fun1Impl<A1s,Rs> >& f );

template <class Ad, class Rd, class DF>
struct Fun1Constructor;

template <class Arg1, class Result> 
class fun1 : public c_fun_type<Arg1,Result>, public smart_functoid1 {
   typedef boost::intrusive_ptr<const Fun1Impl<Arg1,Result> > RefImpl;
   RefImpl ref;
   template <class A, class B> friend class fun1;
   
   template <class Ad, class Rd, class DF>
   friend struct Fun1Constructor;

   template <class A1d, class Rd, class A1s, class Rs> 
   friend fun1<A1d,Rd> explicit_convert1( const fun1<A1s,Rs>& f );
public:
   typedef Fun1Impl<Arg1,Result>* Impl;

   fun1( int, Impl i ) : ref(i) {}
   Result operator()( const Arg1& x ) const { return ref->operator()(x); }

   template <class DF>
   fun1( const DF& df ) : ref( Fun1Constructor<Arg1,Result,DF>::make(df) ) {}

   fun1( const fun1& x ) : ref(x.ref) {}
   fun1& operator=( const fun1& x ) { ref = x.ref; return *this; }
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef fun1 This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif
   // result_of
   template <class X> struct result;
   template <class Me, class X>
   struct result<Me(X)> : public Type<typename RT<fun1,X>::result_type> {};
};

template <class Arg1, class Result>
class Fun1Impl : public c_fun_type<Arg1,Result> {
   mutable RefCountType refC;
public:
   Fun1Impl() : refC(0) {}
   virtual Result operator()( const Arg1& ) const =0;
   virtual ~Fun1Impl() {}

   template <class X, class Y>
   friend void intrusive_ptr_add_ref( const Fun1Impl<X,Y>* p );
   template <class X, class Y>
   friend void intrusive_ptr_release( const Fun1Impl<X,Y>* p );
};
template <class T, class U> 
void intrusive_ptr_add_ref( const Fun1Impl<T,U>* p ) {
   ++ (p->refC);
}  
template <class T, class U> 
void intrusive_ptr_release( const Fun1Impl<T,U>* p ) {
   if( !--(p->refC) ) delete p;
}  

template <class A1d, class Rd, class A1s, class Rs> 
class Fun1Converter : public Fun1Impl<A1d,Rd> {
   typedef boost::intrusive_ptr<const Fun1Impl<A1s,Rs> > MyFun;
   MyFun f;
public:
   Fun1Converter( const MyFun& g ) : f(g) {}
   Rd operator()( const A1d& x ) const {
      return f->operator()( x );
   }
};

template <class A1d, class Rd, class A1s, class Rs> 
Fun1Impl<A1d,Rd>* 
convert1( const boost::intrusive_ptr<const Fun1Impl<A1s,Rs> >& f ) {
   return new Fun1Converter<A1d,Rd,A1s,Rs>( f );
}

template <class A1d, class Rd, class A1s, class Rs> 
class Fun1ExplicitConverter : public Fun1Impl<A1d,Rd> {
   typedef boost::intrusive_ptr<const Fun1Impl<A1s,Rs> > MyFun;
   MyFun f;
public:
   Fun1ExplicitConverter( const MyFun& g ) : f(g) {}
   Rd operator()( const A1d& x ) const {
      return static_cast<Rd>( f->operator()(  static_cast<A1s>(x)  ) );
   }
};

template <class A1d, class Rd, class A1s, class Rs> 
fun1<A1d,Rd> explicit_convert1( const fun1<A1s,Rs>& f ) {
   return fun1<A1d,Rd>( 1, new Fun1ExplicitConverter<A1d,Rd,A1s,Rs>(f.ref) );
}


template <class Gen>
class Gen1 : public Fun1Impl<
      typename monomorphic_traits<Gen>::first_argument_type,
      typename monomorphic_traits<Gen>::result_type> {
   Gen g;
public:
   Gen1( Gen x ) : g(x) {}
   typename monomorphic_traits<Gen>::result_type 
   operator()( const typename 
               monomorphic_traits<Gen>::first_argument_type& x ) const {
      return g(x); 
   }
};
   
template <class Unary>
Gen1<Unary>* makeFun1Ref( const Unary& g ) {
   return new Gen1<Unary>(g);
}

template <class Ad, class Rd, class DF>
struct Fun1Constructor {
   static Fun1Impl<Ad,Rd>* make( const DF& df ) {
      return makeFun1Ref( monomorphize1<Ad,Rd>(df) );
   }
};
template <class Ad, class Rd, class As, class Rs>
struct Fun1Constructor<Ad,Rd,fun1<As,Rs> > {
   static Fun1Impl<Ad,Rd>* make( const fun1<As,Rs>& f ) {
      return convert1<Ad,Rd>(f.ref);
   }
};

//////////////////////////////////////////////////////////////////////

template <class Arg1, class Arg2, class Result>
class Fun2Impl;

template <class A1d, class A2d, class Rd, class A1s, class A2s, class Rs>
Fun2Impl<A1d,A2d,Rd>* 
convert2( const boost::intrusive_ptr<const Fun2Impl<A1s,A2s,Rs> >& f );

template <class A1d, class A2d, class Rd, class DF>
struct Fun2Constructor;

// Note that this class has two signatures: it can be used either as
// a two argument function or as a single argument function (currying).
template <class Arg1, class Arg2, class Result>
class fun2 : public smart_functoid2 {
   typedef boost::intrusive_ptr<const Fun2Impl<Arg1, Arg2, Result> > RefImpl;
   RefImpl ref;
   template <class A1, class A2, class R> friend class fun2;
   template <class A1d, class A2d, class Rd, class A1s, class A2s, class Rs>
   friend fun2<A1d,A2d,Rd> explicit_convert2( const fun2<A1s,A2s,Rs>& f );

   template <class A1d, class A2d, class Rd, class DF>
   friend struct Fun2Constructor;

public:
   // Signature for normal use of the functoid (2 args)
   template <class P1, class P2 = void>
   struct sig : public fun_type<Result> {};

   // Signature for using this function with automatic currying (1-arg)
   template <class P1> struct sig<P1,void> : public fun_type<
      typename RT<bind1of2_type,fun2,P1>::result_type> {};

   // Signatures for using this function with underscore currying (1-arg)
   template <class P2> struct sig<auto_curry_type,P2> : public 
      fun_type<typename RT<bind2of2_type,fun2,P2>::result_type > {};
   template <class P1> struct sig<P1,auto_curry_type> : public 
      fun_type<typename RT<bind1of2_type,fun2,P1>::result_type > {};

   typedef Fun2Impl<Arg1,Arg2,Result>* Impl;
   fun2( int, Impl i ) : ref(i) {}
   
   template <class DF>
   fun2( const DF& df ) 
   : ref( Fun2Constructor<Arg1,Arg2,Result,DF>::make(df) ) {}

   fun2( const fun2& x ) : ref(x.ref) {}
   fun2& operator=( const fun2& x ) { ref = x.ref; return *this; }

   // normal call
   Result operator()( const Arg1& x, const Arg2& y ) const { 
      return ref->operator()(x,y); 
   }
  
   // inheritable underscore currying!
   typename sig<auto_curry_type,Arg2>::result_type
   operator()( const auto_curry_type&, const Arg2& y ) const { 
      return bind2of2(*this, y);
   }
   typename sig<Arg1,auto_curry_type>::result_type
   operator()( const Arg1& x, const auto_curry_type& ) const { 
      return bind1of2(*this, x);
   }
  
   // inheritable automatic currying!
   typename sig<Arg1>::result_type operator()( const Arg1& x) const {
      return bind1of2(*this, x);
   }
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef fun2 This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif
   // result_of
   template <class F> struct result;
   template <class Me, class X, class Y>
   struct result<Me(X,Y)> :public Type<typename sig<X,Y>::result_type> {};
   template <class Me, class X>
   struct result<Me(X)> :public Type<typename sig<X,void>::result_type> {};
};

template <class Arg1, class Arg2, class Result>
class Fun2Impl : public c_fun_type<Arg1,Arg2,Result> {
   mutable RefCountType refC;
public:
   Fun2Impl() : refC(0) {}
   virtual Result operator()( const Arg1&, const Arg2& ) const =0;
   virtual ~Fun2Impl() {}

   template <class X, class Y, class Z>
   friend void intrusive_ptr_add_ref( const Fun2Impl<X,Y,Z>* p );
   template <class X, class Y, class Z>
   friend void intrusive_ptr_release( const Fun2Impl<X,Y,Z>* p );
};
template <class T, class U, class V> 
void intrusive_ptr_add_ref( const Fun2Impl<T,U,V>* p ) {
   ++ (p->refC);
}  
template <class T, class U, class V> 
void intrusive_ptr_release( const Fun2Impl<T,U,V>* p ) {
   if( !--(p->refC) ) delete p;
}  

template <class A1d, class A2d, class Rd, class A1s, class A2s, class Rs>
class Fun2Converter : public Fun2Impl<A1d,A2d,Rd> {
   typedef boost::intrusive_ptr<const Fun2Impl<A1s,A2s,Rs> > MyFun;
   MyFun f;
public:
   Fun2Converter( const MyFun& g ) : f(g) {}
   Rd operator()( const A1d& x, const A2d& y ) const {
      return f->operator()( x, y );
   }
};

template <class A1d, class A2d, class Rd, class A1s, class A2s, class Rs>
Fun2Impl<A1d,A2d,Rd>* 
convert2( const boost::intrusive_ptr<const Fun2Impl<A1s,A2s,Rs> >& f ) {
   return new Fun2Converter<A1d,A2d,Rd,A1s,A2s,Rs>( f );
}

template <class A1d, class A2d, class Rd, class A1s, class A2s, class Rs>
class Fun2ExplicitConverter : public Fun2Impl<A1d,A2d,Rd> {
   typedef boost::intrusive_ptr<const Fun2Impl<A1s,A2s,Rs> > MyFun;
   MyFun f;
public:
   Fun2ExplicitConverter( const MyFun& g ) : f(g) {}
   Rd operator()( const A1d& x, const A2d& y ) const {
      return static_cast<Rd>( f->operator()( static_cast<A1s>(x), 
                                             static_cast<A2s>(y) ) );
   }
};

template <class A1d, class A2d, class Rd, class A1s, class A2s, class Rs>
fun2<A1d,A2d,Rd> explicit_convert2( const fun2<A1s,A2s,Rs>& f ) {
   return fun2<A1d,A2d,Rd>( 1, 
      new Fun2ExplicitConverter<A1d,A2d,Rd,A1s,A2s,Rs>(f.ref) );
}

template <class Gen>
class Gen2 : public Fun2Impl<
      typename monomorphic_traits<Gen>::first_argument_type,
      typename monomorphic_traits<Gen>::second_argument_type,
      typename monomorphic_traits<Gen>::result_type>
{
   Gen g;
public:
   Gen2( Gen x ) : g(x) {}
   typename monomorphic_traits<Gen>::result_type 
   operator()( 
      const typename monomorphic_traits<Gen>::first_argument_type& x,
      const typename monomorphic_traits<Gen>::second_argument_type& y ) const {
      return g(x,y); 
   }
};
   
template <class Binary>
Gen2<Binary>* makeFun2Ref( const Binary& g ) {
   return new Gen2<Binary>(g);
}

template <class A1d, class A2d, class Rd, class DF>
struct Fun2Constructor {
   static Fun2Impl<A1d,A2d,Rd>* make( const DF& df ) {
      return makeFun2Ref( monomorphize2<A1d,A2d,Rd>(df) );
   }
};
template <class A1d, class A2d, class Rd, class A1s, class A2s, class Rs>
struct Fun2Constructor<A1d,A2d,Rd,fun2<A1s,A2s,Rs> > {
   static Fun2Impl<A1d,A2d,Rd>* make( const fun2<A1s,A2s,Rs>& f ) {
      return convert2<A1d,A2d,Rd>(f.ref);
   }
};

//////////////////////////////////////////////////////////////////////

template <class Arg1, class Arg2, class Arg3, class Result>
class Fun3Impl;

template <class A1d, class A2d, class A3d, class Rd, 
          class A1s, class A2s, class A3s, class Rs> 
Fun3Impl<A1d,A2d,A3d,Rd>* 
convert3( const boost::intrusive_ptr<const Fun3Impl<A1s,A2s,A3s,Rs> >& f );

template <class A1d, class A2d, class A3d, class Rd, class DF>
struct Fun3Constructor;

// The "Guts" class helps us implement currying; fun3 floats gently atop
// Fun3Guts and adds currying.  
template <class Arg1, class Arg2, class Arg3, class Result>
class Fun3Guts : public c_fun_type<Arg1,Arg2,Arg3,Result> {
   typedef boost::intrusive_ptr<const Fun3Impl<Arg1,Arg2,Arg3,Result> > RefImpl;
   RefImpl ref;
   template <class A, class B, class C, class D> friend class Fun3Guts;
   template <class A, class B, class C, class D> friend class fun3;
   
   template <class A1d, class A2d, class A3d, class Rd, class DF>
   friend struct Fun3Constructor;

   template <class A1d, class A2d, class A3d, class Rd, 
             class A1s, class A2s, class A3s, class Rs> 
   friend fun3<A1d,A2d,A3d,Rd> 
   explicit_convert3( const fun3<A1s,A2s,A3s,Rs>& f );
public:
   typedef Fun3Impl<Arg1,Arg2,Arg3,Result>* Impl;

   Fun3Guts( int, Impl i ) : ref(i) {}
   Result operator()( const Arg1& x, const Arg2& y, const Arg3& z ) const 
      { return ref->operator()(x,y,z); }

   template <class DF>
   Fun3Guts( const DF& df )
   : ref( Fun3Constructor<Arg1,Arg2,Arg3,Result,DF>::make(df) ) {}

   Fun3Guts( const Fun3Guts& x ) : ref(x.ref) {}
   Fun3Guts& operator=( const Fun3Guts& x ) { ref = x.ref; return *this; }
};

template <class Arg1, class Arg2, class Arg3, class Result>
class fun3 : public smart_functoid3 {
   template <class A1d, class A2d, class A3d, class Rd, class DF>
   friend struct Fun3Constructor;

   template <class A1d, class A2d, class A3d, class Rd, 
             class A1s, class A2s, class A3s, class Rs> 
   friend fun3<A1d,A2d,A3d,Rd> 
   explicit_convert3( const fun3<A1s,A2s,A3s,Rs>& f );

   Fun3Guts<Arg1,Arg2,Arg3,Result> rep;
public:
   typedef Fun3Impl<Arg1,Arg2,Arg3,Result>* Impl;

   fun3( int, Impl i ) : rep(1,i) {}

   template <class DF> 
   fun3( const DF& df ) : rep(df) {}

   fun3( const fun3& x ) : rep(x.rep) {}
   fun3& operator=( const fun3& x ) { rep = x.rep; return *this; }
   
   typedef full3<Fun3Guts<Arg1,Arg2,Arg3,Result> > SigHelp;
   template <class A, class B=void, class C=void>
   struct sig : public SigHelp::template sig<A,B,C> {};
 
   template <class A, class B, class C>
   typename sig<A,B,C>::result_type
   operator()( const A& x, const B& y, const C& z ) const 
      { return make_curryable3(rep)(x,y,z); }

   template <class A, class B>
   typename sig<A,B>::result_type
   operator()( const A& x, const B& y ) const 
      { return thunk3(rep,x,y); }

   template <class A>
   typename sig<A>::result_type
   operator()( const A& x ) const 
      { return thunk3(rep,x); }
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef fun3 This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif
   // result_of
   template <class F> struct result;
   template <class Me, class X, class Y, class Z>
   struct result<Me(X,Y,Z)> :public Type<typename sig<X,Y,Z>::result_type> {};
   template <class Me, class X, class Y>
   struct result<Me(X,Y)> :public Type<typename sig<X,Y,void>::result_type> {};
   template <class Me, class X>
   struct result<Me(X)> :public Type<typename sig<X,void,void>::result_type> {};
};

template <class Arg1, class Arg2, class Arg3, class Result>
class Fun3Impl : public c_fun_type<Arg1,Arg2,Arg3,Result> {
   mutable RefCountType refC;
public:
   Fun3Impl() : refC(0) {}

   virtual Result operator()( const Arg1&, const Arg2&, const Arg3& ) const =0;
   virtual ~Fun3Impl() {}

   template <class X, class Y, class Z, class W>
   friend void intrusive_ptr_add_ref( const Fun3Impl<X,Y,Z,W>* p );
   template <class X, class Y, class Z, class W>
   friend void intrusive_ptr_release( const Fun3Impl<X,Y,Z,W>* p );
};
template <class T, class U, class V, class W> 
void intrusive_ptr_add_ref( const Fun3Impl<T,U,V,W>* p ) {
   ++ (p->refC);
}  
template <class T, class U, class V, class W> 
void intrusive_ptr_release( const Fun3Impl<T,U,V,W>* p ) {
   if( !--(p->refC) ) delete p;
}  

template <class A1d, class A2d, class A3d, class Rd, 
          class A1s, class A2s, class A3s, class Rs> 
class Fun3Converter : public Fun3Impl<A1d,A2d,A3d,Rd> {
   typedef boost::intrusive_ptr<const Fun3Impl<A1s,A2s,A3s,Rs> > MyFun;
   MyFun f;
public:
   Fun3Converter( const MyFun& g ) : f(g) {}
   Rd operator()( const A1d& x, const A2d& y, const A3d& z ) const {
      return f->operator()( x, y, z );
   }
};

template <class A1d, class A2d, class A3d, class Rd, 
          class A1s, class A2s, class A3s, class Rs> 
Fun3Impl<A1d,A2d,A3d,Rd>* 
convert3( const boost::intrusive_ptr<const Fun3Impl<A1s,A2s,A3s,Rs> >& f ) {
   return new Fun3Converter<A1d,A2d,A3d,Rd,A1s,A2s,A3s,Rs>( f );
}

template <class A1d, class A2d, class A3d, class Rd, 
          class A1s, class A2s, class A3s, class Rs> 
class Fun3ExplicitConverter : public Fun3Impl<A1d,A2d,A3d,Rd> {
   typedef boost::intrusive_ptr<const Fun3Impl<A1s,A2s,A3s,Rs> > MyFun;
   MyFun f;
public:
   Fun3ExplicitConverter( const MyFun& g ) : f(g) {}
   Rd operator()( const A1d& x, const A2d& y, const A3d& z ) const {
      return static_cast<Rd>( f->operator()(  static_cast<A1s>(x),
         static_cast<A2s>(y), static_cast<A3s>(z)  ) );
   }
};

template <class A1d, class A2d, class A3d, class Rd, 
          class A1s, class A2s, class A3s, class Rs> 
fun3<A1d,A2d,A3d,Rd> explicit_convert3( const fun3<A1s,A2s,A3s,Rs>& f ) {
   return fun3<A1d,A2d,A3d,Rd>( 1, 
      new Fun3ExplicitConverter<A1d,A2d,A3d,Rd,A1s,A2s,A3s,Rs>(f.rep.ref) );
}

template <class Gen>
class Gen3 : public Fun3Impl<
   typename monomorphic_traits<Gen>::first_argument_type,
   typename monomorphic_traits<Gen>::second_argument_type,
   typename monomorphic_traits<Gen>::third_argument_type,
   typename monomorphic_traits<Gen>::result_type> {
   Gen g;
public:
   Gen3( Gen x ) : g(x) {}
   typename monomorphic_traits<Gen>::result_type operator()( 
      const typename monomorphic_traits<Gen>::first_argument_type& x,
      const typename monomorphic_traits<Gen>::second_argument_type& y,
      const typename monomorphic_traits<Gen>::third_argument_type& z ) const { 
      return g(x,y,z); 
   }
};

template <class Ternary>
Gen3<Ternary>* makeFun3Ref( const Ternary& g ) {
   return new Gen3<Ternary>(g);
}

template <class A1d, class A2d, class A3d, class Rd, class DF>
struct Fun3Constructor {
   static Fun3Impl<A1d,A2d,A3d,Rd>* make( const DF& df ) {
      return makeFun3Ref( monomorphize3<A1d,A2d,A3d,Rd>(df) );
   }
};
template <class A1d, class A2d, class A3d, class Rd, 
          class A1s, class A2s, class A3s, class Rs> 
struct Fun3Constructor<A1d,A2d,A3d,Rd,fun3<A1s,A2s,A3s,Rs> > {
   static Fun3Impl<A1d,A2d,A3d,Rd>* make( const fun3<A1s,A2s,A3s,Rs>& f ) {
      return convert3<A1d,A2d,A3d,Rd>(f.rep.ref);
   }
};

} // end namespace fcpp
} // end namespace boost

#endif
