// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_FULL_HPP
#define BOOST_FCPP_FULL_HPP

#include "smart.hpp"
#include "curry.hpp"
#include "pre_lambda.hpp"

namespace boost {
namespace fcpp {

template <class F>
struct monomorphic_traits : public F {};
template <class F>
struct monomorphic_traits<full0<F> > {
   typedef typename F::result_type result_type;
};
template <class F>
struct monomorphic_traits<full1<F> > {
   typedef typename F::first_argument_type first_argument_type;
   typedef typename F::argument_type argument_type;
   typedef typename F::result_type result_type;
};
template <class F>
struct monomorphic_traits<full2<F> > {
   typedef typename F::first_argument_type first_argument_type;
   typedef typename F::second_argument_type second_argument_type;
   typedef typename F::result_type result_type;
};
template <class F>
struct monomorphic_traits<full3<F> > {
   typedef typename F::first_argument_type first_argument_type;
   typedef typename F::second_argument_type second_argument_type;
   typedef typename F::third_argument_type third_argument_type;
   typedef typename F::result_type result_type;
};

//////////////////////////////////////////////////////////////////////
// Full functoids
//////////////////////////////////////////////////////////////////////
// Just as curryable2/curryable3 serve as wrappers which decorate
// functoids to give them 'curryability', fullN are wrappers to give
// functoids _all_ the 'features' that functoids can have.  The current
// "extra feature set" for functoids is
//   - curryability     (ability to call with fewer args (curry.h))
//   - lambda-awareness (operator[] for use inside lambda)
//   - smartness        (inherited typedefs to answer various
//                       introspective questions (smart.h))
//   - infix syntax     (call using x ^f^ y (operator.h))
// The fullN classes just combine all of the features into one
// uber-wrapper which does it all.

// Don't forget that any features added here need to be added to 
// the indirect functoids in function.h, too.
// And also to the SplitArgsable class in prelude.h.

template <class F>
class full0 : public smart_functoid0, 
public c_fun_type<typename RT<F>::result_type> {
   F f;
public:
   full0() : f() {}
   full0( const F& ff ) : f(ff) {}
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef full0 This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif
   inline typename RT<F>::result_type operator()() const {
      return f();
   }
   // Already works with result_of due to result_type member typedef
};

template <class F>
class full1 : public smart_functoid1 {
   F f;
public:
   full1() : f() {}
   full1( const F& ff ) : f(ff) {}
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef full1 This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif
   template <class T> struct sig 
      : public fun_type<typename RT<F,T>::result_type> {};
   template <class T>
   inline typename sig<T>::result_type operator()( const T& x ) const {
      return f(x);
   }
   // result_of
   template <class X> struct result;
   template <class Me, class X>
   struct result<Me(X)> : public Type<typename sig<X>::result_type> {};
};

template <class F>
class full2 : public smart_functoid2 {
   F f;
public:
   full2() : f() {}
   full2( const F& ff ) : f(ff) {}
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef full2 This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif
/////////////  copied from curryable2; added impl:: to binders  //////
   template <class X, class Y=void>
   struct sig
   : public fun_type<typename RT<F,X,Y>::result_type> {};

   template <class X>
   struct sig<X,void> : public fun_type<full1<impl::binder1of2<F,X> > > {};

   template <class Y>
   struct sig<auto_curry_type,Y>
   : public fun_type<full1<impl::binder2of2<F,Y> > > {};

   template <class X>
   struct sig<X,auto_curry_type>
   : public fun_type<full1<impl::binder1of2<F,X> > > {};

   template <class X>
   typename sig<X>::result_type operator()( const X& x ) const {
      return make_full1( impl::binder1of2<F,X>(f,x) );
   }
   template <class X, class Y>
   inline typename sig<X,Y>::result_type
   operator()( const X& x, const Y& y ) const {
      // need partial specialization, so defer to a class helper
return impl::Curryable2Helper<typename sig<X,Y>::result_type,F,X,Y>::go(f,x,y);
   }
//////////////////////////////////////////////////////////////////////
   // result_of
   template <class X> struct result;
   template <class Me, class X, class Y>
   struct result<Me(X,Y)> : public Type<typename sig<X,Y>::result_type> {};
   template <class Me, class X>
   struct result<Me(X)> : public Type<typename sig<X,void>::result_type> {};
};

template <class F>
class full3 : public smart_functoid3 {
   F f;
public:
   full3() : f() {}
   full3( const F& ff ) : f(ff) {}
#ifdef BOOST_FCPP_ENABLE_LAMBDA
   typedef full3 This;
   template <class A> typename lambda_impl::BracketCallable<This,A>::Result
   operator[]( const A& a ) const
   { return lambda_impl::BracketCallable<This,A>::go( *this, a ); }
#endif
/////////////  copied from curryable3; added impl:: to all binders  //
   template <class X, class Y=void, class Z=void>
   struct sig
   : public fun_type<typename RT<F,X,Y,Z>::result_type> {};

   template <class X,class Y> struct sig<X,Y,void>
   : public fun_type<full1<impl::binder1and2of3<F,X,Y> > > {};

   template <class X> struct sig<X,auto_curry_type,void>
   : public fun_type<full2<impl::binder1of3<F,X> > > {};

   template <class Y> struct sig<auto_curry_type,Y,void>
   : public fun_type<full2<impl::binder2of3<F,Y> > > {};

   template <class X> struct sig<X,void,void>
   : public fun_type<full2<impl::binder1of3<F,X> > > {};

   template <class X> struct sig<X,auto_curry_type,auto_curry_type>
   : public fun_type<full2<impl::binder1of3<F,X> > > {};

   template <class Y> struct sig<auto_curry_type,Y,auto_curry_type>
   : public fun_type<full2<impl::binder2of3<F,Y> > > {};

   template <class Z> struct sig<auto_curry_type,auto_curry_type,Z>
   : public fun_type<full2<impl::binder3of3<F,Z> > > {};

   template <class X,class Z> struct sig<X,auto_curry_type,Z> : public 
      fun_type<full1<impl::binder1and3of3<F,X,Z> > > {};

   template <class Y,class Z> struct sig<auto_curry_type,Y,Z> : public 
      fun_type<full1<impl::binder2and3of3<F,Y,Z> > > {};

   template <class X,class Y> struct sig<X,Y,auto_curry_type> : public 
      fun_type<full1<impl::binder1and2of3<F,X,Y> > > {};

   template <class X,class Y>
   typename sig<X,Y>::result_type operator()( const X& x, const Y& y ) const {
      // need partial specialization, so defer to a class helper
return impl::Curryable3Helper2<typename sig<X,Y>::result_type,F,X,Y>::go(f,x,y);
   }
   template <class X>
   typename sig<X>::result_type operator()( const X& x ) const {
      return make_full2(impl::binder1of3<F,X>(f,x));
   }

   template <class X, class Y, class Z>
   inline typename sig<X,Y,Z>::result_type
   operator()( const X& x, const Y& y, const Z& z ) const {
      // need partial specialization, so defer to a class helper
      return impl::Curryable3Helper<typename sig<X,Y,Z>::result_type,F,X,Y,Z>
      ::go(f,x,y,z);
   }
//////////////////////////////////////////////////////////////////////
   // result_of
   template <class X> struct result;
   template <class Me, class X, class Y, class Z>
   struct result<Me(X,Y,Z)> :public Type<typename sig<X,Y,Z>::result_type> {};
   template <class Me, class X, class Y>
   struct result<Me(X,Y)> :public Type<typename sig<X,Y,void>::result_type> {};
   template <class Me, class X>
   struct result<Me(X)> :public Type<typename sig<X,void,void>::result_type> {};
};

template <class F> full0<F> make_full0( const F& f ) { return full0<F>(f); }
template <class F> full1<F> make_full1( const F& f ) { return full1<F>(f); }
template <class F> full2<F> make_full2( const F& f ) { return full2<F>(f); }
template <class F> full3<F> make_full3( const F& f ) { return full3<F>(f); }

//////////////////////////////////////////////////////////////////////
// Definitions of stuff heretofore put-off...
//////////////////////////////////////////////////////////////////////
// from curry.h:
typedef full1<impl::Xconst_x_type> const_x_type;
typedef full1<impl::Xkonst_type>   konst_type;
typedef full2<impl::bind1of1_type> bind1of1_type;

typedef full2<impl::bind1of2_type> bind1of2_type;
typedef full2<impl::bind2of2_type> bind2of2_type;
typedef full3<impl::bind1and2of2_type> bind1and2of2_type;

// The only 4-arg function in the entire library; it isn't a functoid.
typedef impl::bind1and2and3of3_type bind1and2and3of3_type;

typedef full3<impl::bind1and2of3_type> bind1and2of3_type;
typedef full3<impl::bind2and3of3_type> bind2and3of3_type;
typedef full3<impl::bind1and3of3_type> bind1and3of3_type;
typedef full2<impl::bind1of3_type> bind1of3_type;
typedef full2<impl::bind2of3_type> bind2of3_type;
typedef full2<impl::bind3of3_type> bind3of3_type;

BOOST_FCPP_MAYBE_NAMESPACE_OPEN
   // C++ keyword, so add trailing underscore
BOOST_FCPP_MAYBE_EXTERN const_x_type const_;   
BOOST_FCPP_MAYBE_EXTERN konst_type konst;   
BOOST_FCPP_MAYBE_EXTERN bind1of1_type bind1of1;

BOOST_FCPP_MAYBE_EXTERN bind1of2_type bind1of2;
BOOST_FCPP_MAYBE_EXTERN bind2of2_type bind2of2;
BOOST_FCPP_MAYBE_EXTERN bind1and2of2_type bind1and2of2;

BOOST_FCPP_MAYBE_EXTERN bind1and2and3of3_type bind1and2and3of3;
BOOST_FCPP_MAYBE_EXTERN bind1and2of3_type bind1and2of3;
BOOST_FCPP_MAYBE_EXTERN bind2and3of3_type bind2and3of3;
BOOST_FCPP_MAYBE_EXTERN bind1and3of3_type bind1and3of3;
BOOST_FCPP_MAYBE_EXTERN bind1of3_type bind1of3;
BOOST_FCPP_MAYBE_EXTERN bind2of3_type bind2of3;
BOOST_FCPP_MAYBE_EXTERN bind3of3_type bind3of3;
BOOST_FCPP_MAYBE_NAMESPACE_CLOSE

} // end namespace fcpp
} // end namespace boost

#endif
