// Copyright Brian McNamara and Yannis Smaragdakis 2000-2003.
// Use, modification and distribution is subject to the
// Boost Software License, Version 1.0.  (See accompanying file
// LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_FCPP_CURRY_HPP
#define BOOST_FCPP_CURRY_HPP

#include "signature.hpp"

namespace boost {
namespace fcpp {

//////////////////////////////////////////////////////////////////////
// This file implements currying for functoids.  Included here are
//  - bindMofN    for currying the Mth of N arguments
//  - const_      for turning a value into a nullary constant function
//  - konst       for turning a value into a unary constant function
//  - thunkN      curries all N args of a functoid into a thunk
// as well as the infrastructure for the now-deprecated
//  - curryableN  way to curry with underscores (e.g. f(_,y,_); )
// which has been subsumed by the fullN classes.
//
// For more info, see
//    http://www.cc.gatech.edu/~yannis/fc++/currying.html
// which is now somewhat out-of-date, and
//    http://www.cc.gatech.edu/~yannis/fc++/New1.5/full.html
// which helps bring you up to date.
//////////////////////////////////////////////////////////////////////

// Important to implementation of CurryableN classes
struct auto_curry_type {};
BOOST_FCPP_MAYBE_EXTERN auto_curry_type _;   

// Forward declarations; curryability and fullness are now somewhat
// inextricably intertwined...
template <class F> class full0;
template <class F> class full1;
template <class F> class full2;
template <class F> class full3;
template <class F> full0<F> make_full0( const F& f );
template <class F> full1<F> make_full1( const F& f );
template <class F> full2<F> make_full2( const F& f );
template <class F> full3<F> make_full3( const F& f );

namespace impl {

//////////////////////////////////////////////////////////////////////
// const_ and konst
//////////////////////////////////////////////////////////////////////

template <class T>
struct ConstHelper : public c_fun_type<T> {
   const T x;
public:
   ConstHelper( const T& a ) : x(a) {}
   T operator()() const { return x; }
};
struct Xconst_x_type {
   template <class T>
   struct sig : public fun_type<full0<ConstHelper<T> > > {};

   template <class T>
   full0<ConstHelper<T> > operator()( const T& x ) const {
      return make_full0( ConstHelper<T>(x) );
   }
};

// It is so common to say ignore(const_(x)) that we make a composite
// function out of it, and call it konst.  That is
//    konst(x)(y) = x
template <class X>
struct XkonstHelp {
   X x;
   XkonstHelp( const X& xx ) : x(xx) {}
   template <class A> struct sig : public fun_type<X> {};
   template <class A>
   X operator()( const A& ) const {
      return x;
   }
};
struct Xkonst_type {
   template <class X> struct sig : public fun_type<
      full1<XkonstHelp<X> > > {};
   template <class X>
   typename sig<X>::result_type operator()( const X& x ) const {
      return make_full1( XkonstHelp<X>( x ) );
   }
};

//////////////////////////////////////////////////////////////////////
// Binders (through "...of2")
//////////////////////////////////////////////////////////////////////

template <class Unary, class Arg>
class binder1of1 
: public c_fun_type<typename RT<Unary,Arg>::result_type> {
   const Unary f;
   const Arg a;
public:
   binder1of1( const Unary& x, const Arg& y ) : f(x), a(y) {}
   typename RT<Unary,Arg>::result_type operator()() const { return f(a); }
};

struct bind1of1_type {
   template <class Unary, class Arg> struct sig 
   : public fun_type<full0<binder1of1<Unary,Arg> > > {};
   
   template <class Unary, class Arg>
   inline full0<binder1of1<Unary,Arg> > 
   operator()( const Unary& f, const Arg& a ) const {
      return make_full0( binder1of1<Unary,Arg>(f,a) );
   }
};

template <class Binary, class Arg1>
class binder1of2 {
   const Binary f;
   const Arg1 x;
public:
   binder1of2( const Binary& a, const Arg1& b ) : f(a), x(b) {}

   template <class Arg2> struct sig 
   : public fun_type<typename Binary::template sig<Arg1,Arg2>::result_type> {};

   template <class Arg2>
   typename Binary::template sig<Arg1,Arg2>::result_type
   operator()( const Arg2& y ) const {
      return f(x,y);
   }
};

struct bind1of2_type {
   template <class Binary, class Arg1> struct sig 
   : public fun_type<full1<binder1of2<Binary,Arg1> > > {};

   template <class Binary, class Arg1>
   full1<binder1of2<Binary,Arg1> >
   operator()( const Binary& f, const Arg1& x ) const {
      return make_full1( binder1of2<Binary,Arg1>(f,x) );
   }
};

template <class Binary, class Arg2>
class binder2of2 {
   const Binary f;
   const Arg2 y;
public:
   binder2of2( const Binary& a, const Arg2& b ) : f(a), y(b) {}

   template <class Arg1> struct sig 
   : public fun_type<typename Binary::template sig<Arg1,Arg2>::result_type> {};

   template <class Arg1>
   typename Binary::template sig<Arg1,Arg2>::result_type
   operator()( const Arg1& x ) const {
      return f(x,y);
   }
};

struct bind2of2_type {
   template <class Binary, class Arg2> struct sig 
   : public fun_type<full1<binder2of2<Binary,Arg2> > > {};

   template <class Binary, class Arg2>
   full1<binder2of2<Binary,Arg2> >
   operator()( const Binary& f, const Arg2& y ) const {
      return make_full1( binder2of2<Binary,Arg2>(f,y) );
   }
};

template <class Binary, class Arg1, class Arg2>
class binder1and2of2 
: public c_fun_type<typename RT<Binary,Arg1,Arg2>::result_type > {
   const Binary f;
   const Arg1 a1;
   const Arg2 a2;
public:
   binder1and2of2( const Binary& x, const Arg1& y, const Arg2& z ) 
   : f(x), a1(y), a2(z) {}
   typename RT<Binary,Arg1,Arg2>::result_type  
   operator()() const { return f(a1,a2); }
};

struct bind1and2of2_type {
   template <class Binary, class Arg1, class Arg2>
   struct sig 
   : public fun_type< 
      full0<binder1and2of2<Binary,Arg1,Arg2> > > {};

   template <class Binary, class Arg1, class Arg2>
   full0<binder1and2of2<Binary,Arg1,Arg2> >
   operator()( const Binary& f, const Arg1& a1, const Arg2& a2 ) const {
      return make_full0( binder1and2of2<Binary,Arg1,Arg2>(f,a1,a2) );
   }
};

//////////////////////////////////////////////////////////////////////
// Now that bindNof2 are defined, we can define curryable2, which then
// some of the later binders can use.
//////////////////////////////////////////////////////////////////////

template <class R, class F, class X, class Y>
struct Curryable2Helper {
   static inline R go( const F& f, const X& x, const Y& y ) {
      return f(x,y); 
   }
};

template <class R, class F, class Y>
struct Curryable2Helper<R,F,auto_curry_type,Y> {
   static R go( const F& f, const auto_curry_type& , const Y& y ) {
      return make_full1( binder2of2<F,Y>(f,y) );
   }
};

template <class R, class F, class X>
struct Curryable2Helper<R,F,X,auto_curry_type> {
   static R go( const F& f, const X& x, const auto_curry_type& ) {
      return make_full1( binder1of2<F,X>(f,x) );
   }
};

// Note: curryable2 is gone.  It was deprecated and has been removed.

//////////////////////////////////////////////////////////////////////
// With curryable2 out of the way, we can go back to the 3-arg binders.
//////////////////////////////////////////////////////////////////////

template <class Ternary, class A1, class A2, class A3>
class binder1and2and3of3
: public c_fun_type<typename RT<Ternary,A1,A2,A3>::result_type> {
   const Ternary f;
   const A1 a1;
   const A2 a2;
   const A3 a3;
public:
   binder1and2and3of3( const Ternary& w, const A1& x, const A2& y, const A3& z )
   : f(w), a1(x), a2(y), a3(z) {}
   typename RT<Ternary,A1,A2,A3>::result_type 
   operator()() const { return f(a1,a2,a3); }
};

struct bind1and2and3of3_type {
   // No sig, as there are no 4-arg functoids
   template <class Ternary, class A1, class A2, class A3>
   full0<binder1and2and3of3<Ternary,A1,A2,A3> >
   operator()( const Ternary& f, const A1& a1, 
               const A2& a2, const A3& a3 ) const {
      return make_full0( binder1and2and3of3<Ternary,A1,A2,A3>(f,a1,a2,a3) );
   }
};

template <class Ternary, class Arg1, class Arg2>
class binder1and2of3 {
   const Ternary f;
   const Arg1 a1;
   const Arg2 a2;
public:
   template <class Arg3>
   struct sig 
   : public fun_type<
               typename Ternary::template sig<Arg1,Arg2,Arg3>::result_type> {};

   binder1and2of3(const Ternary& w, const Arg1& x, const Arg2& y) : 
     f(w), a1(x), a2(y) {}
   template <class Arg3>
   typename sig<Arg3>::result_type 
   operator()(const Arg3& z) const { return f(a1,a2,z); }
};

struct bind1and2of3_type {
   template <class Ternary, class A1, class A2>
   struct sig 
   : public fun_type<full1<binder1and2of3<Ternary,A1,A2> > > {};

   template <class Ternary, class A1, class A2>
   full1<binder1and2of3<Ternary,A1,A2> >
   operator()( const Ternary& f, const A1& a1, const A2& a2 ) const {
      return make_full1( binder1and2of3<Ternary,A1,A2>(f,a1,a2) );
   }
};

template <class Ternary, class Arg2, class Arg3>
class binder2and3of3 {
   const Ternary f;
   const Arg2 a2;
   const Arg3 a3;
public:
   template <class Arg1>
   struct sig 
   : public fun_type<
                    typename RT<Ternary,Arg1,Arg2,Arg3>::result_type> {};

   binder2and3of3(const Ternary& w, const Arg2& y, const Arg3& z) : 
     f(w), a2(y), a3(z) {}
   template <class Arg1>
   typename sig<Arg1>::result_type 
   operator()(const Arg1& x) const { return f(x,a2,a3); }
};

struct bind2and3of3_type {
   template <class Ternary, class A2, class A3>
   struct sig 
   : public fun_type<full1<binder2and3of3<Ternary,A2,A3> > > {};

   template <class Ternary, class A2, class A3>
   full1<binder2and3of3<Ternary,A2,A3> >
   operator()( const Ternary& f, const A2& a2, const A3& a3 ) const {
      return make_full1( binder2and3of3<Ternary,A2,A3>(f,a2,a3) );
   }
};

template <class Ternary, class Arg1, class Arg3>
class binder1and3of3 {
   const Ternary f;
   const Arg1 a1;
   const Arg3 a3;
public:
   template <class Arg2>
   struct sig 
   : public fun_type<
                    typename RT<Ternary,Arg1,Arg2,Arg3>::result_type> {};
                    // need RT above due to g++ bug on line below
                    // typename Ternary::sig<Arg1,Arg2,Arg3>::result_type> {};

   binder1and3of3(const Ternary& w, const Arg1& x, const Arg3& z) : 
     f(w), a1(x), a3(z) {}
   template <class Arg2>
   typename sig<Arg2>::result_type 
   operator()(const Arg2& y) const { return f(a1,y,a3); }
};

struct bind1and3of3_type {
   template <class Ternary, class A1, class A3>
   struct sig 
   : public fun_type<full1<binder1and3of3<Ternary,A1,A3> > > {};

   template <class Ternary, class A1, class A3>
   full1<binder1and3of3<Ternary,A1,A3> >
   operator()( const Ternary& f, const A1& a1, const A3& a3 ) const {
      return make_full1( binder1and3of3<Ternary,A1,A3>(f,a1,a3) );
   }
};

template <class Ternary, class Arg1>
class binder1of3 {
   const Ternary f;
   const Arg1 x;
public:
   binder1of3( const Ternary& a, const Arg1& b ) : f(a), x(b) {}

   template <class Arg2, class Arg3> struct sig 
   : public fun_type<
              typename Ternary::template sig<Arg1,Arg2,Arg3>::result_type> {};

   template <class Arg2, class Arg3>
   typename RT<Ternary,Arg1,Arg2,Arg3>::result_type
   operator()( const Arg2& y, const Arg3& z ) const {
      return f(x,y,z);
   }
};

struct bind1of3_type {
   template <class Ternary, class Arg1>
   struct sig 
   : public fun_type<full2<binder1of3<Ternary,Arg1> > > {};

   template <class Ternary, class Arg1>
   typename sig<Ternary,Arg1>::result_type
   operator()( const Ternary& f, const Arg1& x ) const {
      return make_full2( binder1of3<Ternary,Arg1>(f,x) );
   }
};

template <class Ternary, class Arg2>
class binder2of3 {
   const Ternary f;
   const Arg2 x;
public:
   binder2of3( const Ternary& a, const Arg2& b ) : f(a), x(b) {}

   template <class Arg1, class Arg3>
   struct sig 
   : public fun_type<
              typename Ternary::template sig<Arg1,Arg2,Arg3>::result_type> {};

   template <class Arg1, class Arg3>
   typename RT<Ternary,Arg1,Arg2,Arg3>::result_type
   operator()( const Arg1& y, const Arg3& z ) const {
      return f(y,x,z);
   }
};

struct bind2of3_type {
   template <class Ternary, class Arg2>
   struct sig 
   : public fun_type<full2<binder2of3<Ternary,Arg2> > > {};

   template <class Ternary, class Arg2>
   typename sig<Ternary,Arg2>::result_type
   operator()( const Ternary& f, const Arg2& x ) const {
      return make_full2( binder2of3<Ternary,Arg2>(f,x) );
   }
};

template <class Ternary, class Arg3>
class binder3of3 {
   const Ternary f;
   const Arg3 x;
public:
   binder3of3( const Ternary& a, const Arg3& b ) : f(a), x(b) {}

   template <class Arg1, class Arg2>
   struct sig 
   : public fun_type<
              typename Ternary::template sig<Arg1,Arg2,Arg3>::result_type> {};

   template <class Arg1, class Arg2>
   typename RT<Ternary,Arg1,Arg2,Arg3>::result_type
   operator()( const Arg1& y, const Arg2& z ) const {
      return f(y,z,x);
   }
};

struct bind3of3_type {
   template <class Ternary, class Arg3>
   struct sig 
   : public fun_type<full2<binder3of3<Ternary,Arg3> > > {};

   template <class Ternary, class Arg3>
   typename sig<Ternary,Arg3>::result_type
   operator()( const Ternary& f, const Arg3& x ) const {
      return make_full2( binder3of3<Ternary,Arg3>(f,x) );
   }
};

//////////////////////////////////////////////////////////////////////
// Note that the implementations of "thunkN" are still called
// "curryN_type", and still accept fewer than all of the args.  The
// intention is now that thunkN will always be called with all N args,
// returning a thunk.

struct curry3_type {
  template <class Ternary, class A1 = void, class A2 = void> struct sig
  : public fun_type<full1<binder1and2of3<Ternary,A1,A2> > > {};  

  template <class Ternary, class A1>
  struct sig<Ternary, A1, void> 
    : public fun_type<full2<binder1of3<Ternary,A1> > > {};  


  template <class Ternary, class A1, class A2, class A3>
  full0<binder1and2and3of3<Ternary,A1,A2,A3> >
  operator()( const Ternary& f, const A1& a1, const A2& a2, 
	      const A3& a3 ) const {
    return make_full0( binder1and2and3of3<Ternary,A1,A2,A3>(f,a1,a2,a3) );
  }

  template <class Ternary, class A1, class A2>
  typename sig<Ternary, A1, A2>::result_type
  operator()( const Ternary& f, const A1& a1, const A2& a2 ) const {
    return make_full1( binder1and2of3<Ternary,A1,A2>(f,a1,a2) );
  }

  template <class Ternary, class A1>
  typename sig<Ternary, A1>::result_type
  operator()( const Ternary& f, const A1& a1 ) const {
    return make_full2( binder1of3<Ternary,A1>(f,a1) );
  }
};

struct curry2_type {
  template <class Binary, class A1, class A2 = void>
  struct sig 
  : public fun_type<full0<binder1and2of2<Binary,A1,A2> > > {};

  template <class Binary, class A1>
  struct sig<Binary, A1, void> 
    : public fun_type<full1<binder1of2<Binary,A1> > > {};  


  template <class Binary, class A1, class A2>
  typename sig<Binary, A1, A2>::result_type
  operator()( const Binary& f, const A1& a1, const A2& a2 ) const {
    return make_full0( binder1and2of2<Binary,A1,A2>(f,a1,a2) );
  }

  template <class Binary, class A1>
  typename sig<Binary, A1>::result_type
  operator()( const Binary& f, const A1& a1 ) const {
    return make_full1( binder1of2<Binary,A1>(f,a1) );
  }
};

struct curry1_type {
  template <class Unary, class A1>
  struct sig : public fun_type<full0<binder1of1<Unary,A1> > > {};  

  template <class Unary, class A1>
  typename sig<Unary, A1>::result_type
  operator()( const Unary& f, const A1& a1 ) const {
    return make_full0( binder1of1<Unary,A1>(f,a1) );
  }
};

//////////////////////////////////////////////////////////////////////
// Finally, curryable3 (what a huge beast)
//////////////////////////////////////////////////////////////////////

template <class R, class F, class X, class Y, class Z>
struct Curryable3Helper {
   static inline R go( const F& f, const X& x, const Y& y, const Z& z ) {
      return f(x,y,z); 
   }
};

template <class R, class F, class X>
struct Curryable3Helper<R,F,X,auto_curry_type,auto_curry_type> {
   static R go( const F& f, const X& x, const auto_curry_type&, 
                 const auto_curry_type& ) {
      return make_full2( binder1of3<F,X>(f,x) );
   }
};

template <class R, class F, class Y>
struct Curryable3Helper<R,F,auto_curry_type,Y,auto_curry_type> {
   static R go( const F& f, const auto_curry_type&, const Y& y, 
                 const auto_curry_type& ) {
      return make_full2( binder2of3<F,Y>(f,y) );
   }
};

template <class R, class F, class Z>
struct Curryable3Helper<R,F,auto_curry_type,auto_curry_type,Z> {
   static R go( const F& f, const auto_curry_type&, const auto_curry_type&,
                 const Z& z ) {
      return make_full2( binder3of3<F,Z>(f,z) );
   }
};

template <class R, class F, class Y, class Z>
struct Curryable3Helper<R,F,auto_curry_type,Y,Z> {
   static R go( const F& f, const auto_curry_type&, const Y& y, 
                 const Z& z ) {
      return make_full1( binder2and3of3<F,Y,Z>(f,y,z) );
   }
};

template <class R, class F, class X, class Z>
struct Curryable3Helper<R,F,X,auto_curry_type,Z> {
   static R go( const F& f, const X& x, const auto_curry_type&, 
                 const Z& z ) {
      return make_full1( binder1and3of3<F,X,Z>(f,x,z) );
   }
};

template <class R, class F, class X, class Y>
struct Curryable3Helper<R,F,X,Y,auto_curry_type> {
   static R go( const F& f, const X& x, const Y& y, 
                 const auto_curry_type& ) {
      return make_full1( binder1and2of3<F,X,Y>(f,x,y) );
   }
};

template <class R, class F, class X, class Y>
struct Curryable3Helper2 {
   static R go( const F& f, const X& x, const Y& y ) {
      return make_full1( binder1and2of3<F,X,Y>(f,x,y) );
   }
};

template <class R, class F, class Y>
struct Curryable3Helper2<R,F,auto_curry_type,Y> {
   static R go( const F& f, const auto_curry_type&, const Y& y ) {
      return make_full2(binder2of3<F,Y>(f,y));
   }
};

template <class R, class F, class X>
struct Curryable3Helper2<R,F,X,auto_curry_type> {
   static R go( const F& f, const X& x, const auto_curry_type& ) {
      return make_full2(binder1of3<F,X>(f,x));
   }
};

// Note: curryable3 is gone.  It was deprecated and has been removed.

} // end namespace impl

// I just renamed "curryN" to "thunkN"
typedef impl::curry3_type thunk3_type;
typedef impl::curry2_type thunk2_type;
typedef impl::curry1_type thunk1_type;

// FIX THIS: the thunkNs are not full functoids.  Oh well.  Until I find
// a need to make them so, I'm not going through the effort.
BOOST_FCPP_MAYBE_EXTERN thunk1_type thunk1;
BOOST_FCPP_MAYBE_EXTERN thunk2_type thunk2;
BOOST_FCPP_MAYBE_EXTERN thunk3_type thunk3; 

// Note: the make_curryableNs are gone.  They were deprecated 
// and have been removed.

} // end namespace fcpp
} // end namespace boost

#endif
