#include "svm.h"
#include "mex.h"
#include "IntersectionClassifier.h"
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <algorithm>

using namespace std;

double minprod(svm_node* a, double* b,int dim){
  double retval=0;
  svm_node *a_temp=a;
  double *b_temp=b;
  for(int i = 0; i < dim; i++){
    if(a_temp->index == (i + 1)){
      retval+=fmin(a_temp->value,*b_temp);
      a_temp++;b_temp++;
    }
    else 
      b_temp++;
  }
  return retval;
}

class fWrap{
public:
  double val;
  int indx;
  fWrap(double aval, int aindx){
    val=aval;
    indx=aindx;
  }
  bool operator<(const fWrap& other) const{
    return val < other.val;
  }
};

/**
 *returns the permutation which caused the sorting
 *increasing order 
 */
void sortIndex(int * index, double *vals, int dim){
  fWrap *data  = (fWrap*)malloc(sizeof(fWrap)*dim);
  for(int i = 0; i < dim;i++)
    data[i] = fWrap(vals[i],i);
  sort(data, data+dim);
  for(int i = 0; i < dim;i++)
    index[i] = data[i].indx;
  free((void*)data);
}

IntersectionClassifier::IntersectionClassifier(svm_model* model,int nBins,int featureDim){
  dim = featureDim;
  precompute(model,nBins);
}
IntersectionClassifier::~IntersectionClassifier(){
  //free all the variables 
  for(int i = 0; i < dim ;i++){
    free(sortSV[i]);
    free(cumalphax[i]);
    free(cumalpha[i]);
    free(linearRank[i]);
    free(hDimValue[i]);
  }
  free(sortSV);
  free(cumalphax);
  free(cumalpha);
  free(hCoef);
  free(hOff);
  free(linearRank);
  free(hDimValue);
}

void IntersectionClassifier::print(){
  //print the model
  printf("SVs : %d featureDim: %d\n",nSV,dim);
  for(int i = 0 ; i <10;i++)
    printf("%f ",cumalphax[0][i]);
  printf("\n");

}
double* IntersectionClassifier::predict(svm_model * model,double **data, int num){

  double * predictVals = (double*)malloc(num*sizeof(double));
  for(int i = 0 ; i < num;i++){
    double thisVal=0;
    for(int j = 0; j < nSV; j++){
      thisVal+=model->sv_coef[0][j]*minprod(model->SV[j],data[i],dim);
    }
    predictVals[i] = thisVal - model->rho[0];
  } 
  return predictVals;
}

double* IntersectionClassifier::fastPredict(svm_model * model, double ** data, int num){
  double * predictVals = (double*)malloc(num*sizeof(double));
  int start,end,mid;
  double thisVal,jVal;
  for(int i = 0 ; i < num;i++){
    thisVal=0;
    for(int j = 0; j < dim; j++){
      jVal=data[i][j];
      double * sortSVj = sortSV[j];
      //do a binary search here...
      start=0; end=nSV-1;
      
      while(start <=end){
	mid = (start + end)/2;
	if(jVal >= sortSVj[mid])
	  start = mid +1;
	else
	  end = mid-1;
      }
      thisVal += cumalphax[j][start] - cumalpha[j][start]*jVal; 
    }
    predictVals[i] = thisVal - model->rho[0];
  }
  return predictVals;
}
/**
 *simulate a linear kernel computation time
 */
double* IntersectionClassifier::linSimulate(svm_model * model, double ** data, int num){
  double * predictVals = (double*)malloc(num*sizeof(double));
  double thisVal;
  for(int i = 0 ; i < num;i++){
    thisVal=0;
    for(int j = 0; j < dim; j++)
      thisVal += data[i][j]*hCoef[j];
    predictVals[i] = thisVal;
  }
  return predictVals;
}
//flips the order of computation 

double* IntersectionClassifier::fastPredictFlip(svm_model * model, double ** data, int num){
  double * predictVals = (double*)malloc(num*sizeof(double));
  for(int i = 0 ; i < num;i++)
    predictVals[i] = -model->rho[0];
  int start,end,mid;
  double thisVal,jVal;
  double *sortSVj, *cumalphaxj,* cumalphaj;
  for(int j = 0; j < dim; j++){
    
    sortSVj = sortSV[j];
    cumalphaxj = cumalphax[j];
    cumalphaj = cumalpha[j];

    for(int i = 0 ; i < num;i++){
      thisVal=0;
      
      jVal=data[i][j+1];
	
      //do a binary search here...
      start=0; end=nSV-1;
      
	while(start <=end){
	  mid = (start + end)/2;
	  if(jVal >= sortSVj[mid])
	    start = mid + 1;
	  else
	    end = mid - 1;
	}
	predictVals[i] += cumalphaxj[start] - cumalphaj[start]*jVal; 
      }
  }
  return predictVals;
}

double* IntersectionClassifier::linHashPredict(svm_model * model, double ** data, int num){
  double * predictVals = (double*)malloc(num*sizeof(double));
  for(int i = 0 ; i < num;i++)
    predictVals[i] = -model->rho[0];

  int start,end,mid;
  
  double jVal;
  int *linearRankj;
  double *sortSVj, *cumalphaxj,* cumalphaj;

  for(int j = 0; j < dim; j++){
      
    sortSVj = sortSV[j];
    linearRankj = linearRank[j];
    cumalphaxj = cumalphax[j];
    cumalphaj = cumalpha[j];

    for(int i = 0 ; i < num;i++){
      
      jVal=data[i][j];
      
      if(jVal >= sortSVj[nSV-1])
	predictVals[i] +=cumalphaxj[nSV];
      else if(jVal >= sortSVj[0]){
	//do a binary search here...

	int index=(int)(jVal*hCoef[j] + hOff[j]);

	//smart initialze search between prev and next ranks
	start=linearRankj[index]; end=linearRankj[index+1];
	
	while(start <=end){
	  mid = (start + end)/2;
	  if(jVal >= sortSVj[mid])
	    start = mid + 1;
	  else
	    end = mid - 1;
	}
	predictVals[i] += cumalphaxj[start] - cumalphaj[start]*jVal; 
      }
    }
  }
  return predictVals;
}


double* IntersectionClassifier::pwConstPredict(svm_model * model, double ** data, int num){
  double * predictVals = (double*)malloc(num*sizeof(double));
  for(int i = 0 ; i < num;i++)
    predictVals[i] = -model->rho[0];

  double jVal;
  double *hDimValuej,*cumalphaxj,*sortSVj;
  //clipData(data,num);
  for(int j = 0; j < dim; j++){
    hDimValuej = hDimValue[j];
    cumalphaxj = cumalphax[j];
    sortSVj    = sortSV[j];
    for(int i = 0 ; i < num;i++){
      
      jVal=data[i][j]; //starts from 0
      
      if(jVal > sortSVj[nSV-1])
	predictVals[i] += cumalphaxj[nSV];
      else if(jVal >= sortSVj[0]){
	int index=(int)(jVal*hCoef[j] + hOff[j]);
	predictVals[i] += hDimValuej[index+1];
      }
    }
  }
  //mexPrintf("Values Computed!\n");
  return predictVals;
}

double* IntersectionClassifier::pwLinPredict(svm_model * model, double ** data, int num){
  double * predictVals = (double*)malloc(num*sizeof(double));
  for(int i = 0 ; i < num;i++)
    predictVals[i] = -model->rho[0];

  double jVal,findex,dx;
  double *hDimValuej,*cumalphaxj,*sortSVj;
  int index;

  for(int j = 0; j < dim; j++){
    hDimValuej = hDimValue[j];
    cumalphaxj = cumalphax[j];
    sortSVj    = sortSV[j];
    for(int i = 0 ; i < num;i++){
      
      jVal=data[i][j]; //starts from 0
      
      if(jVal > sortSVj[nSV-1])
	predictVals[i] += cumalphaxj[nSV];
      else if(jVal >= sortSVj[0]){
	findex=jVal*hCoef[j] + hOff[j];
	index=(int)findex;
	dx=findex-index;
	predictVals[i] += hDimValuej[index]*(1-dx) + hDimValuej[index+1]*dx ;
      }
    }
  }
  return predictVals;
}

void IntersectionClassifier::precompute(svm_model* model,int nBins){
  //we have the following in the data file
  //dim lines 
  //each line has nSV values, feat_value,sum_alphax, sum_alpha
  //dim=model->maxIndex; //set the dim manually;
  
  nSV=model->l; 
  sortSV    = (double**)malloc(dim*sizeof(double*));
  cumalphax = (double**)malloc(dim*sizeof(double*));
  cumalpha  = (double**)malloc(dim*sizeof(double*));

  //precompte ranks 
  linearRank = (int**)malloc(dim*sizeof(int*));
  hCoef = (double*)malloc(dim*sizeof(double));
  hOff = (double*)malloc(dim*sizeof(double));
  
  //precompute values on unif spaced bins
  hDimValue = (double**)malloc(dim*sizeof(double*));
  for(int i = 0; i < dim ;i++){
    sortSV[i]    = (double*)malloc(nSV*sizeof(double));
    cumalphax[i] = (double*)malloc((nSV+1)*sizeof(double));
    cumalpha[i]  = (double*)malloc((nSV+1)*sizeof(double));
    linearRank[i]= (int *)malloc((nBins+1)*sizeof(int));
    hDimValue[i] = (double*)malloc((nBins+1)*sizeof(double));
  }

  double *sv_buffer = (double*)malloc(sizeof(double)*nSV);
  double *sv_sort_buffer = (double*)malloc(sizeof(double)*nSV);
  int* model_idx = (int *)malloc(sizeof(int)*nSV);
  int* index     = (int *)malloc(sizeof(int)*nSV);
  //index to the position in each model
  for(int j = 0; j < nSV; j++)
    model_idx[j] = 0;

  for(int i = 0; i < dim ;i++){
    for(int j = 0; j <nSV; j++){
      //the model is sparse..so check if the index matches
      if(model->SV[j][model_idx[j]].index != (i + 1))
	sv_buffer[j] = 0;
      else{
	sv_buffer[j] = model->SV[j][model_idx[j]].value;
	model_idx[j]++;
      }
    }
    sortIndex(index,sv_buffer,nSV);

    //march through the index and compute the cumsum 
    cumalphax[i][0] = 0;
    cumalpha[i][0]  = 0;

    for(int k = 0; k < nSV;k++){
      int ik=index[k];
      sortSV[i][k]      = sv_buffer[ik]; //sorted support vectors
      cumalphax[i][k+1] = cumalphax[i][k] + model->sv_coef[0][ik]*sv_buffer[ik]; //cumulative sums of alpha*x
      cumalpha[i][k+1]  = cumalpha[i][k] + model->sv_coef[0][ik]; //cumulative sums of alpha
    }
    //get a sorted vesion
    for(int k = 0; k < nSV;k++){
      sv_sort_buffer[k]=sv_buffer[index[k]];
    }
    double hmin=sv_sort_buffer[0];
    double hmax=sv_sort_buffer[nSV-1];
    double step = (hmax - hmin)/nBins;
    
    //set the coefs
    if(fabs(step) > 1e-10){ 
      hCoef[i]=1/step;
      hOff[i]= -hmin/step;
    }
    else{
      hCoef[i] = 0;
      hOff[i] = 0; 
    }
    //compute the ranks
    
    int lstart=0;
    for(int k=0; k < nBins;k++){
      double sv_val = hmin + k*step;
      for(int l=lstart; l<nSV; l++){
	if(sv_sort_buffer[l] >= sv_val){
	  linearRank[i][k]=l;
	  hDimValue[i][k] = cumalphax[i][l] - cumalpha[i][l]*sv_val;
	  lstart=l;
	  break;
	}
      }
      linearRank[i][nBins]=nSV-1;
      hDimValue[i][nBins]=cumalphax[i][nSV];
    }
  }
  //free stuff
  free(sv_sort_buffer);
  free(sv_buffer);
  free(index);
  free(model_idx);
}

void IntersectionClassifier::clipData(double **data, int num){
  double val;
  for(int i=0;i<num;i++)
    for(int j=0;j<dim;j++){
      //data[i][0] is the label
      val=data[i][j+1];
      if(val < sortSV[j][0])
	data[i][j+1]=sortSV[j][0];
      else if(val > sortSV[j][nSV-1])
	data[i][j+1]=sortSV[j][nSV-1];
    }
}

