#ifndef _MULTILINEAR_HPP_
#define _MULTILINEAR_HPP_

/*
Implements a multilinear interpolation.

This object can be queried to determine what weight should give to each point
when using a multilinear interpolation over a grid of points.

Notice that the getState function for Environment objects normalize their states to be in [0,1].
Still, this code works for arbitrary ranges. This is a bit redundant - we could have assumed here
that the inputs always are in the range [0,1], or we could have let the environments give states
in any range (not just [0,1]).
*/

#include <utility>
#include <Eigen/Dense>

class Multilinear {
public:
    Multilinear(int numInputs,                      // Dimension of input space
                int order,                          // Number of points to tile along each dimension
                const Eigen::VectorXd & minInputs,  // Minimum point for each dimension
                const Eigen::VectorXd & maxInputs); // Maximum point for each dimension

    // Get the number of points that are stored. This is the size of the weight vector returned by getWeights
    int getNumPoints() const;

    // Get the weights that would be applied to each point. The result is stored in buff, which is a std::pair---buff.first holds the indices with non-zero weights and buff.second holds the corresponding point weights.
    void getWeights(const Eigen::VectorXd & input,
                    std::pair<Eigen::VectorXi,Eigen::VectorXd> & buff) const;

    // Get the position of the index'th point
    Eigen::VectorXd getPoint(int index) const;

private:
    // Storage for inputs to constructor
    int numInputs;
    int order;
    Eigen::VectorXd minInputs;
    Eigen::VectorXd maxInputs;

    // pow(2, numInputs) --- only compute it once
    int pow2numInputs;

    int ipow(int a, int b) const;                           // Raise a to the b.
    void incrementBinaryCounter(Eigen::VectorXi & counter) const;
    int getIndex(const Eigen::VectorXi & indices) const;    // Convert the matrix-style indices into a single vector index
};

#endif // _MULTILINEAR_HPP_
