#include "Multilinear.hpp"

#include <iostream>

using namespace Eigen;
using namespace std;

Multilinear::Multilinear(int numInputs,
                         int order,
                         const VectorXd & minInputs,
                         const VectorXd & maxInputs) {
    // Copy over arguments and do a simple error-check
    this->numInputs = numInputs;
    this->order = order;
    if (order < 2) {
        cerr << "Order must be 2 or more." << endl;
        exit(1);
    }
    this->minInputs = minInputs;
    this->maxInputs = maxInputs;
    pow2numInputs = ipow(2,numInputs);
}

int Multilinear::getNumPoints() const {
    return ipow(order, numInputs);
}

void Multilinear::getWeights(const Eigen::VectorXd & input,
                    std::pair<Eigen::VectorXi,Eigen::VectorXd> & buff) const {
    // Make sure the output buffer is the right size
    if (buff.first.size() != pow2numInputs) {
        buff.first.resize(pow2numInputs);
        buff.second.resize(pow2numInputs);
    }

    // Normalize the input, and store it in x (now each element is in the range [0,order-1]).
    // We want this to be thread-able, so we allocate x every time---we do not make it a private member variable or static variable.
    VectorXd x = (double)(order-1)*(input - minInputs).array() / (maxInputs - minInputs).array();

    // Get the index for the lower-left corner of the hyper-cube containing the input point
    VectorXi ll = x.cast<int>();                        // The Eigen library's cast function will truncate the double

    // If for any index, i, input[i] = maxInput[i] exactly,
    // then x = (order-1)*1 = order-1.
    // We want ll[i] to be at most order-2,
    // since we will look at the hypercube starting from here
    // and going to order-1 at most.
    // So, if ll[i] = order-1 for any i, set it to order-2
    for (int i = 0; i < numInputs; i++)
        ll[i] = min(ll[i], order-2);

    // We will use a binary counter to loop over the corners of the hypercube with minimum-corner = ll.
    VectorXi binaryCounter = VectorXi::Zero(numInputs);
    for (int i = 0; i < pow2numInputs; i++, incrementBinaryCounter(binaryCounter)) {
        // Store the index for the point (ll + binaryCounter) --- we just map the giant hypercube of points onto a vector.
        // getIndex says what index in the array an integral point in numInputs-dimensional space has.
        buff.first[i] = getIndex(ll + binaryCounter);     // Get this point's index from its coordinates

        // Now compute the weight for the point (ll + binaryCounter)
        // It is the area of the opposite sub-rectangle (sub-hyper-rectangle)
        // Note: This is because each hyper-rectangle between two points has an area of one volume unit.
        buff.second[i] = 1.0;
        for (int j = 0; j < numInputs; j++) {
            // How wide is the (opposite) hyper-rectangle along the j'th dimension?
            if (binaryCounter[j] == 0)  // The current point is at the lower-edge of the hyper-rectangle
                buff.second[i] *= 1 - (x[j] - (double)ll[j]);
            else                        // The current point is at the upper-edge of the hyper-rectangle
                buff.second[i] *= x[j] - (double)ll[j];
        }
    }
}

// Convert an index for one of the returned weights to an actual point in the input space.
Eigen::VectorXd Multilinear::getPoint(int index) const {
    VectorXd result(numInputs);
    for (int i = numInputs-1; i >= 0; i--) {
        result[i] = index % order;
        index /= order;
    }
    // Now we need to undo the normalization
    result = result.array() / (double)(order-1);            // First normalize down to [0,1]
    result = result.array() * (maxInputs - minInputs).array() + minInputs.array();  // Then expand back to the real state range.
    return result;
}

int Multilinear::ipow(int a, int b) const {
	if (b == 0) return 1;
	if (b == 1) return a;
	int tmp = ipow(a, b / 2);
	if (b % 2 == 0) return tmp * tmp;
	return a * tmp * tmp;
}

// Treat counter as a Little-endian binary counter, and increment it
// This could be sped up by storing counter as a char * and dealing with
// bits directly, but this way is more clear... is it a computational bottleneck?
void Multilinear::incrementBinaryCounter(Eigen::VectorXi & counter) const {
    for (int i = 0; i < (int)counter.size(); i++) {
        if (counter[i] == 0) {
            counter[i] = 1;
            return;
        }
        counter[i] = 0;
    }
}

// This is not the opposite of getPoint. This is internal inside of getWeights
// and uses normalized points. So, all indices (input) should be in the range [0, order-1]
int Multilinear::getIndex(const Eigen::VectorXi & indices) const {
    int result = 0;
    for (int i = 0; i < numInputs; i++) {
        result *= order;
        result += indices[i];
    }
    return result;
}
