#include <iostream>                 // For console I/O
#include <stdarg.h>                 // For variadic functions

#include "ValueIteration.hpp"       // Contains code for value iteration
#include "PolicyEvaluation.hpp"     // Contains code to evaluate a policy
#include "Multilinear.hpp"          // Code useful for estimating Q(s,a) using multilinear interpolation

// All of the environment domains that we might use
#include "MountainCar.hpp"
#include "PuddleWorld.hpp"
#include "Acrobot.hpp"
#include "CartPole.hpp"
#include "Pendulum.hpp"
#include "Bicycle.hpp"

using namespace std;
using namespace Eigen;

// For the menu system: ask a question and return the answer.
// The "..." arguments are numOptions const char * objects
// that contain the possible answers.
int getSelection(const char * name, int numOptions, ...) {
    va_list vl;
    va_start(vl,numOptions);
    cout << "Select " << name << ":" << endl;
    for (int i = 0; i < numOptions; i++)
        cout << "[" << i << "] " << va_arg(vl, const char*) << endl;
    cout << "Selection: ";
    int result;
    cin >> result;
    getchar();
    return result;
}


// For use in the menu system: Ask a boolean question and return the answer
bool getBoolSelection(const char * name, const char * trueOption, const char * falseOption) {
    return getSelection(name, 2, trueOption, falseOption) == 0;
}

// Run a simple menu that lets the user select an experiment to run.
int runMenu(mt19937_64 & generator) {
    while (true) {
        ENVTYPE type;
        // Ask what environment to run
        switch (getSelection("environment", 6, "Mountain Car", "Puddle World", "Acrobot", "Cart Pole", "Pendulum Swing-Up and Balance", "Bicycle")) {
            case 0:     // Mountain Car
                type = et_MountainCar;
                break;
            case 1:
                type = et_PuddleWorld;
                break;
            case 2:
                type = et_Acrobot;
                break;
            case 3:
                type = et_CartPole;
                break;
            case 4:
                type = et_Pendulum;
                break;
            case 5:
                type = et_Bicycle;
                break;
            default:
                cout << "Unknown environment type." << endl << endl;
                continue; // @TODO ******************************************************* test that this kicks the while(true) to trigger again.
        }
        Environment * e = createEnvironment(type, generator);

        // As what order to use
        int order;
        cout << "Enter the order (state is " << e->getStateDim() << " dimensional): ";
        cin >> order;
        getchar();
        if (order < 2) {
            order = 2;
            cout << "Setting order = 2. Orders less than 2 are not allowed." << endl;
        }

        // Create the multilinear object
        Multilinear * d = new Multilinear(e->getStateDim(), order, VectorXd::Zero(e->getStateDim()), VectorXd::Ones(e->getStateDim()));

        // Ask whether the user wants to use value averaging
        bool valueAveraging;
        valueAveraging = getBoolSelection("value averaging", "Yes", "No");

        // Ask what method to use to compute maxQ
        MaxQVariantType maxQVariant;
        switch (getSelection("maxQ Variant", 2, "Ordinary Bellman", "Consistent Bellman")) {
            case 0:
                maxQVariant = MQV_Ordinary;
                break;
            case 1:
                maxQVariant = MQV_Consistent;
                break;
            default:    // It would be a paint to set it up to ask again or go up a level in the menu. Just quit.
                cout << "Unknown input. Exiting." << endl;
                delete e;   // Clean up memory first
                return 1;   // 1 denotes that an error occurred
        }

        // get numIterations
        int numIterations;
        cout << "Enter number of iterations of value iteration: ";
        cin >> numIterations;
        getchar();

        // get numThreads_VI
        int numThreads_VI;
        cout << "Enter number of threads for value-iteration (we suggest # available cores): ";
        cin >> numThreads_VI;
        getchar();

        int numThreads_PE;
        cout << "Enter number of threads for policy evaluation (we suggest # available cores): ";
        cin >> numThreads_PE;
        getchar();

        // Get evaluateFreq
        int evaluateFreq;
        cout << "Evaluate policy every how-many iterations? ";
        cin >> evaluateFreq;
        getchar();

        // Ask whether to print to a file or not
        bool printToFile = getBoolSelection("Print results to file", "Yes", "No");
        string fileName = "";
        if (printToFile) {
            cout << "Enter filename: ";
            cin >> fileName;
        }

        // Do the actual run
        cout << endl;
        runValueIteration(*e, type, *d, maxQVariant, valueAveraging, generator, numIterations, evaluateFreq, numThreads_VI, numThreads_PE, fileName);
        cout << endl;

        // Clean up memory
        delete e;
        delete d;
    }
}

// Entry point for the program. The arguments are ignored.
int main(int argc, const char * argv[]) {
    mt19937_64 generator(time(NULL));       // All seeds come from here. Change this seed to a fixed constant, and this entire program is deterministic (if you remove threading)
    return runMenu(generator);              // Run a simple menu system
    return 0;                               // Exit without saying there was an error
}
