#cython: boundscheck=False
#cython: wraparound=False
from libc cimport math

cdef double _norm =  2.328306549295727688e-10
cdef double _m1 =    4294967087.0
cdef double _m2 =   4294944443.0
cdef double _a12 =     1403580.0
cdef double _a13n =     810728.0
cdef double _a21 =      527612.0
cdef double _a23n =    1370589.0

cdef double _two17 =  131072.0
cdef double _two53 =  9007199254740992.0
cdef double _fact = 5.9604644775390625e-8

cdef double _nextSeed[6]
_nextSeed = [12345, 12345, 12345, 12345, 12345, 12345]

cdef double _A1p76[3][3]
_A1p76  = [
          [      82758667.0, 1871391091.0, 4127413238.0 ], 
          [    3672831523.0,   69195019.0, 1871391091.0 ], 
          [    3672091415.0, 3528743235.0,   69195019.0 ]]

cdef double _A2p76[3][3]
_A2p76  = [
          [    1511326704.0, 3759209742.0, 1610795712.0 ], 
          [    4292754251.0, 1511326704.0, 3889917532.0 ], 
          [    3859662829.0, 4292754251.0, 3708466080.0 ]]

cdef double _A1p127[3][3]
_A1p127 = [
          [    2427906178.0, 3580155704.0,  949770784.0 ], 
          [     226153695.0, 1230515664.0, 3580155704.0 ],
          [    1988835001.0,  986791581.0, 1230515664.0 ]]

cdef double _A2p127[3][3]
_A2p127 = [
          [    1464411153.0,  277697599.0, 1610723613.0 ],
          [      32183930.0, 1464411153.0, 1022607788.0 ],
          [    2824425944.0,   32183930.0, 2093834863.0 ]]

cdef double _MultModM (double a, double s, double c, double m):
    cdef double v
    cdef long a1
    v = a * s + c
    if (v >= _two53) or (v <= -_two53):
        a1 = <long> (a / _two17)
        a -= a1 * _two17
        v = a1 * s
        a1 = <long>(v / m)
        v -= a1 * m
        v = v * _two17 + a * s + c
    a1 = <long>(v / m)
    v -= a1 * m
    return (v + m if v < 0.0 else v)

cdef _MatVecModM (double A[3][3], double s[3], double m):
    cdef int i
    cdef double x[3]
    for i in range(3):
        x[i] = _MultModM (A[i][0], s[0], 0.0, m)
        x[i] = _MultModM (A[i][1], s[1], x[i], m)
        x[i] = _MultModM (A[i][2], s[2], x[i], m)
    return x

cdef _CheckSeed(seed):
    for i in range(3):
        if seed[i] >= _m1:
           raise RuntimeError("Seed[%d] >= m1, is not set"%i)
    for i in range(3,6):
        if seed[i] >= _m2:
            raise RuntimeError("Seed[%d] >= m2, is not set"%i)
    if seed[0] == 0 and seed[1] == 0 and seed[2] == 0:
        raise RuntimeError("First seeds = 0")
    if seed[3] == 0 and seed[4] == 0 and seed[5] == 0:
        raise RuntimeError("Last seeds = 0")
    return

def SetPackageSeed(seed):
    _CheckSeed(seed)
    for i in range(6): _nextSeed[i] = seed[i]

cdef class RngStream:
    cdef double _Cg[6]
    cdef double _Bg[6]
    cdef double _Ig[6]
    cdef int _Anti
    cdef int _IncPrec
    cdef str _name
    
    def __init__(self, gname = ""):
        for i in range(6):
            self._Cg[i] = _nextSeed[i]
            self._Bg[i] = _nextSeed[i]
            self._Ig[i] = _nextSeed[i]
        self._Anti = 0
        self._IncPrec = 0
        self._name = gname
        cdef double v[3]
        cdef double w[3]
        for i in range(3): w[i] = _nextSeed[i]
        v = _MatVecModM(_A1p127, w, _m1)
        _nextSeed[:3] = v
        for i in range(3,6): w[i-3] = _nextSeed[i]
        v = _MatVecModM(_A2p127, w, _m2)
        _nextSeed[3:] = v

    cdef _U01(self):
        cdef long k
        cdef double p1, p2, u
        p1 = _a12 * self._Cg[1] - _a13n * self._Cg[0]
        k = <long>(p1 / _m1)
        p1 -= k * _m1
        if p1 < 0.0: p1 += _m1
        self._Cg[0] = self._Cg[1]
        self._Cg[1] = self._Cg[2]
        self._Cg[2] = p1
        p2 = _a21 * self._Cg[5] - _a23n * self._Cg[3]
        k = <long>(p2 / _m2)
        p2 -= k * _m2
        if p2 < 0.0: p2 += _m2
        self._Cg[3] = self._Cg[4]
        self._Cg[4] = self._Cg[5]
        self._Cg[5] = p2
        u = ((p1 - p2) * _norm if p1 > p2 else (p1 - p2 + _m1) * _norm)
        return ((1 - u) if self._Anti else u)
    
    cdef _U01n(self, int n):
        cdef list l = [0]*n
        for i in range(n):
            l[i]  += self._U01()
        return l
    
    cdef _normal(self,double loc , double scale ,int size ):
        cdef int i = 0
        cdef float a
        cdef list l = []
        
        while i < size:
            U1 = self._U01()
            U2 = self._U01()
            a = scale* math.sqrt(-2*(math.log(U1)))
            l +=  [a * math.cos(2*math.pi * U2) +loc
                   ,a * math.sin(2*math.pi * U2)+loc]
            i += 2
            
        if size == 1:
            return l[0]
        return l[:size]

    cdef _U01d(self):
        cdef double u
        u = self._U01()
        if not self._Anti:
            u += self._U01() * _fact
            return (u if u < 1.0 else u - 1.0)
        else:
            u += (self.U01() - 1.0) * _fact
            return (u + 1.0 if u < 0.0 else u)

    def ResetStartStream(self):
        for i in range(6): self._Cg[i] = self._Bg[i] = self._Ig[i]

    def ResetNextSubstream(self):
        cdef double v[3]
        cdef double w[3]
        for i in range(3): v[i] = self._Bg[i]
        w = _MatVecModM (_A1p76, v, _m1)
        for i in range(3): self._Bg[i] = w[i]
        for i in range(3,6): v[i-3] = self._Bg[i]
        w = _MatVecModM (_A2p76, v, _m2)
        for i in range(3,6): self._Bg[i] = w[i-3]
        for i in range(6): self._Cg[i] = self._Bg[i]

    def ResetStartSubstream(self):
        for i in range(6): self._Cg[i] = self._Bg[i]

    def SetSeed(self, seed):
        _CheckSeed(seed)
        for i in range(6): self._Cg[i] = self._Bg[i] = self._Ig[i] = seed[i]

    def GetState (self, seed):
        for i in range(6): seed[i] = self._Cg[i]

    def WriteState(self):
        txt = "The current state of the Rngstream"
        if (self._name): txt = txt + " " + self._name
        txt = txt + ":"
        print txt
        txt = "   Cg = { "
        for i in range(5): txt = txt + "%d, "%self._Cg[i]
        txt = txt + "%d }\n\n"%self._Cg[5]
        print txt

    def WriteStateFull(self):
        txt = "The Rngstream"
        if (self._name): txt = txt + " %s"%self._name
        txt = txt + ":"
        print txt
        print "Anti = " + ("True" if self._Anti else "False")
        print "IncPrec = " + ("True" if self._IncPrec else "False")
        txt = "   Ig = { "
        for i in range(5): txt = txt + "%d, "%self._Ig[i]
        txt = txt + "%d }"%self._Ig[5]
        print txt
        txt = "   Bg = { "
        for i in range(5): txt = txt + "%d, "%self._Bg[i]
        txt = txt + "%d }"%self._Bg[5]
        print txt
        txt = "   Cg = { "
        for i in range(5): txt = txt + "%d, "%self._Cg[i]
        txt = txt + "%d }\n"%self._Cg[5]
        print txt

    def IncreasedPrecis(self, incp):
        self._IncPrec = incp

    def SetAntithetic(self, a):
        self._Anti = a

    def RandU01(self):
        if (self._IncPrec):
            return self._U01d()
        else:
            return self._U01()
        
    def RandInt(self, int i, int j):
        return i + int((j - i + 1.0) * self.RandU01())
    
    def nRandU01(self,n):
        return self._U01n(n)
    
    def normal(self,loc = 0, scale = 1,size = 1):
        return self._normal(loc , scale ,size )
