package structures;

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Random;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

public class TestHashTable {
	@Rule
	public Timeout globalTimeout = new Timeout(1000);

	public static final int INT_CAPACITY = 101;
	
	public Map<Integer, Integer> intMap;
	public Map<String, String> smallStringMap;

	@Before
	public void before() {
		intMap = new HashTable<Integer, Integer>(INT_CAPACITY);
		smallStringMap = new HashTable<String, String>(1);
	}

	@Test
	public void testSizeEmpty() {
		assertEquals(0, intMap.size());
		assertEquals(0, smallStringMap.size());
	}

	@Test
	public void testSizePutOne() {
		assertEquals(0, intMap.size());
		intMap.put(1, 1);
		assertEquals(1, intMap.size());

		assertEquals(0, smallStringMap.size());
		smallStringMap.put("a", "A");
		assertEquals(1, smallStringMap.size());
	}

	@Test
	public void testSizePutTwo() {
		assertEquals(0, intMap.size());
		intMap.put(1, 1);
		intMap.put(2, -1);
		assertEquals(2, intMap.size());

		assertEquals(0, smallStringMap.size());
		smallStringMap.put("a", "A");
		smallStringMap.put("B", "b");
		assertEquals(2, smallStringMap.size());
	}

	@Test
	public void testSizePutMany() {
		assertEquals(0, intMap.size());
		for (int i = 1; i <= 1000; i++) {
			intMap.put(i, 0);
			assertEquals(i, intMap.size());
		}
	}

	@Test
	public void testClearedSize() {
		assertEquals(0, intMap.size());
		final int count = 1000;
		for (int i = 1; i <= count; i++) {
			intMap.put(i, 0);
		}
		assertEquals(count, intMap.size());
		intMap.clear();
		assertEquals(0, intMap.size());
	}

	@Test
	public void testEmptyDoesNotContainKey() {
		assertFalse(intMap.containsKey(1));
	}
	
	@Test
	public void testContainsKeyOne() {
		intMap.put(1, 1);
		assertTrue(intMap.containsKey(1));
	}
	
	@Test
	public void testContainsManyKeys() {	
		final int count = 1000;
		for (int i = 0; i < count; i++) {
			intMap.put(i, 0);
			assertTrue(intMap.containsKey(i));
		}
		
		for (int i = 0; i < count; i++) {
			assertTrue(intMap.containsKey(i));
		}		
	}
	
	@Test
	public void testEmptyDoesNotContainValue() {
		assertFalse(intMap.containsValue(1));
	}
	
	@Test
	public void testContainsValueOne() {
		intMap.put(1, 1);
		assertTrue(intMap.containsValue(1));
	}

	@Test
	public void testContainsManyValues() {	
		final int count = 1000;
		for (int i = 0; i < count; i++) {
			intMap.put(i, i);
		}
		
		for (int i = 0; i < count; i++) {
			assertTrue(intMap.containsValue(i));
		}
	}

	@Test
	public void getFromEmpty() {
		assertNull(intMap.get(0));
	}
	
	@Test
	public void getOne() {
		Integer zero = new Integer(0);
		intMap.put(1, zero);
		assertEquals(zero, intMap.get(1));		
	}
	
	@Test
	public void getAfterCollision() {
		smallStringMap.put("a", "A");
		smallStringMap.put("b", "B");
		smallStringMap.put("c", "C");
		assertEquals("A", smallStringMap.get("a"));
		assertEquals("B", smallStringMap.get("b"));		
		assertEquals("C", smallStringMap.get("c"));
	}
	
	@Test
	public void testRemoveFromEmpty() {
		assertFalse(intMap.remove(0));
	}
	
	@Test 
	public void testRemoveOne() {
		intMap.put(0, 1);
		assertEquals(1, intMap.size());
		assertEquals(1, (int)intMap.get(0));
		
		intMap.remove(0);
		assertEquals(0, intMap.size());
		assertFalse(intMap.containsKey(0));
	}
	
	@Test
	public void testRemoveManyValues() {
		for (int iter = 0; iter < 100; iter++) {
			Random r = new Random(iter);
			final int count = (int)(INT_CAPACITY * 1.2);
			List<Integer> list = new ArrayList<Integer>(count);
			for (int i = 0; i < count; i++) {
				int randInt = r.nextInt();
				intMap.put(randInt, i);
				assertEquals(i+1, intMap.size());
				list.add(randInt);
			}

			Collections.shuffle(list, new Random(iter));
			for (int i=0; i < list.size(); i++) {
				intMap.remove(list.get(i));
				assertEquals(count - i - 1, intMap.size());
				assertFalse(intMap.containsKey(i));
			}
		}
	}
	
	@Test
	public void testIteratorNone() {
		Iterator<Entry<Integer,Integer>> i = intMap.iterator();
		assertFalse(i.hasNext());
	}

	@Test(expected = NoSuchElementException.class)
	public void testIteratorNoneNext() {
		Iterator<Entry<Integer,Integer>> i = intMap.iterator();
		i.next();
	}

	@Test
	public void testIteratorOne() {
		intMap.put(1,1);
		Iterator<Entry<Integer,Integer>> i = intMap.iterator();
		assertTrue(i.hasNext());
		assertEquals(new Entry<Integer, Integer>(1, 1), i.next());
	}

	@Test(expected = NoSuchElementException.class)
	public void testIteratorOneNext() {
		intMap.put(1,1);
		Iterator<Entry<Integer,Integer>> i = intMap.iterator();
		i.next();
		assertFalse(i.hasNext());
		i.next();
	}

	@Test
	public void testIteratorMany() {
		for (int iter = 0; iter < 100; iter++) {
			before();
			Random r = new Random(iter);
			final int count = (int)(INT_CAPACITY * 1.2);
			List<Integer> list = new ArrayList<Integer>(count);
			for (int i = 0; i < count; i++) {
				int randInt = r.nextInt();
				intMap.put(randInt, i);
				assertEquals(i+1, intMap.size());
				list.add(randInt);
			}

			List<Integer> iteratorList = new ArrayList<Integer>(count);
			for (Entry<Integer, Integer> e : intMap) {
				iteratorList.add(e.getKey());
			}
			
			assertEquals(count, iteratorList.size());
			for (Integer i : list) {
				assertTrue(iteratorList.contains(i));
			}
		}
	}
}
