package data;

import static org.junit.Assert.*;

import java.util.Date;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

public class TestURLRecord {
	@Rule
	public Timeout globalTimeout = new Timeout(1000);

	@Test
	public void testValidConstuction() {
		URLRecord a = new URLRecord("www.a.com", 80, "/a.html");
		assertNotNull(a);
	}

	@Test(expected = NullPointerException.class)
	public void testNullHost() {
		new URLRecord(null, 80, "/a.html");
	}

	@Test(expected = IllegalArgumentException.class)
	public void testEmptyHost() {
		new URLRecord("", 80, "/a.html");
	}

	@Test(expected = IllegalArgumentException.class)
	public void testNegativePort() {
		new URLRecord("www.a.com", -1, "/a.html");
	}

	@Test(expected = IllegalArgumentException.class)
	public void testTooLargePort() {
		new URLRecord("www.a.com", 65536, "/a.html");
	}

	@SuppressWarnings("deprecation")
	@Test
	public void testGetLastUpdatedReturnsCopy() {
		URLRecord a = new URLRecord("www.a.com", 80, "/a.html");
		Date d1 = a.getLastUpdated();
		assertEquals(d1, a.getLastUpdated());
		d1.setYear(1999);
		assertNotEquals(d1, a.getLastUpdated());
	}

	@Test
	public void testUpdate() throws Exception {
		URLRecord a = new URLRecord("www.a.com", 80, "/a.html");
		Date d1 = a.getLastUpdated();
		Thread.sleep(100);
		a.update();
		Date d2 = a.getLastUpdated();
		assertNotEquals(d1, d2);
		assertTrue(d1.compareTo(d2) < 0);
	}

	@Test
	public void testEquals() throws Exception {
		URLRecord a1 = new URLRecord("www.a.com", 80, "/a.html");
		URLRecord a2 = new URLRecord("www.a.com", 80, "/a.html");
		assertEquals(a1, a2);

		URLRecord a3 = new URLRecord("www.a.com", 80, "/a.html");
		Thread.sleep(100);
		URLRecord a4 = new URLRecord("www.a.com", 80, "/a.html");
		assertEquals(a3, a4);
	}

	@Test
	public void testNotEquals() {
		URLRecord a = new URLRecord("www.a.com", 80, "/a.html");

		URLRecord a1 = new URLRecord("www.a1.com", 80, "/a.html");
		assertNotEquals(a, a1);

		URLRecord a81 = new URLRecord("www.a.com", 81, "/a.html");
		assertNotEquals(a, a81);

		URLRecord aZip = new URLRecord("www.a.com", 80, "/a.zip");
		assertNotEquals(a, aZip);
	}

	@Test
	public void testEqualHashCodes() {
		URLRecord a1 = new URLRecord("www.a.com", 80, "/a.html");
		URLRecord a2 = new URLRecord("www.a.com", 80, "/a.html");
		assertEquals(a1.hashCode(), a2.hashCode());
	}

	/**
	 * Note that in principle, it's possible that different URLRecords will have
	 * the same hash code. But this case is extremely unlikely given this
	 * assignment's parameters.
	 */
	@Test
	public void testUnequalHashCodes() {
		URLRecord a = new URLRecord("www.a.com", 80, "/a.html");

		URLRecord a1 = new URLRecord("www.a1.com", 80, "/a.html");
		assertNotEquals(a.hashCode(), a1.hashCode());

		URLRecord a81 = new URLRecord("www.a.com", 81, "/a.html");
		assertNotEquals(a.hashCode(), a81.hashCode());

		URLRecord aZip = new URLRecord("www.a.com", 80, "/a.zip");
		assertNotEquals(a.hashCode(), aZip.hashCode());
	}
}
