package structures;

/**
 * An interface representing a mapping between keys and values (as in a hash
 * table).
 * 
 * Map extends Iterable and must therefore have an iterator method. The behavior
 * of the returned Iterator is undefined if the Map is modified before the
 * iterator is exhausted.
 * 
 * Note that in a real application, you'd use java.util.Map.
 * 
 * @param <K>
 *            the type of the keys
 * @param <V>
 *            the type of the values
 */
public interface Map<K, V> extends Iterable<Entry<K, V>> {
	/**
	 * Removes all mappings, emptying this map.
	 */
	void clear();

	/**
	 * 
	 * @param key
	 * @return true iff the map contains this key
	 */
	boolean containsKey(K key);

	/**
	 * 
	 * @param value
	 * @return true iff the map contains this value
	 */
	boolean containsValue(V value);

	/**
	 * 
	 * @param key
	 * @return the value stored with this key in the map (or null if no such
	 *         key)
	 */
	V get(K key);

	/**
	 * Store a key/value pair in this map. Neither key nor value may be null.
	 * 
	 * Duplicate key among key/value pairs are not permitted. Calling put(k1,
	 * v1) then put(k1, v2) will result in (k1, v2) being stored in this map;
	 * the mapping between k1 and v1 will be lost.
	 * 
	 * @param key
	 * @param value
	 */
	void put(K key, V value);

	/**
	 * Attempts to remove the key (and associated value) from the map, returning
	 * true iff there was a key and value removed.
	 * 
	 * @param key
	 * @return true iff a key/value pair was removed from the map
	 */
	boolean remove(K key);

	/**
	 * 
	 * @return the number of key/value pairs stored in this map
	 */
	int size();
}
