// -*- C++ -*-

#ifndef _REGIONHEAP_H_
#define _REGIONHEAP_H_

/**
 * @file regionheap.h
 * @brief The support classes for reap.
 * @author Emery Berger (emery@cs.umass.edu)
 */

#include <assert.h>

#include "heaplayers.h"
#include "chunkheap.h"
#include "leamallocheap.h"
#include "nestedheap.h"
#include "slopheap.h"

/**
 * @class LeaHeap2
 * @brief An implementation of the Lea heap, using only an sbrk heap.
 */

template <class Sbrk>
class LeaHeap2 :
  public
    Threshold<4096,
	      DLSmallHeapType<DLBigHeapType<CoalesceableHeap<Sbrk> > > >
{};


/**
 * @class AddHeader
 * @brief Adds LeaHeap-compatible metadata to objects.
 */

template <class SuperHeap>
class AddHeader : public SuperHeap {
public:

  inline AddHeader (void) 
    : prevSize (sizeof(Header))
  {
  }

  inline void * malloc (const size_t sz) {
    Header h (prevSize, sz);
    prevSize = sz;
    Header * p = (Header *) SuperHeap::malloc (sz + sizeof(Header));
    *p = h;
    return (p + 1);
  }

  inline void clear (void) {
    prevSize = sizeof(Header);
    SuperHeap::clear();
  }

  // FIX ME - should be private.
  inline void free (void * ptr) { fprintf (stderr, "what the?\n"); abort(); }

private:

  /// Object headers for use by the Lea allocator.
  class Header {
  public:
    Header (size_t p, size_t s)
      : prevSize (p),
	size (s)
    {}
    /// The size of the previous (contiguous) object.
    size_t prevSize;

    /// The size of the current (immediately-following) object.
    size_t size;
  };

  /// The previous size allocated.
  size_t prevSize;
};


/**
 * @class ClearOptimize
 * @brief A heap layer that is optimized for region-like allocation patterns.
 * @param Heap1 This heap provides memory for freshly allocated objects.
 * @param Heap2 This heap provides memory for recycled objects.
 */

template <class Heap1, class Heap2>
class ClearOptimize: public Heap1, Heap2 {
public:

  inline ClearOptimize (void)
    : nothingOnHeap (true)
  {
  }
  
  inline void * malloc (const size_t sz) {
    if (nothingOnHeap) {
      return Heap1::malloc (sz);
    } else {
      void * ptr = Heap2::malloc (sz);
	if (ptr == NULL) {
	  // Right now we assume that this means that the heap is in fact
	  // exhausted. This is not necessarily true, but it's a reasonable
	  // approximation, and is a lot cheaper than tracking the complete
	  // state of the parent heap.
	  nothingOnHeap = true;
	  ptr = Heap1::malloc (sz);
	}
	return ptr;
    }
  }
  
  inline void free (void * ptr) {
#if 1
    nothingOnHeap = false;
#else
    if (nothingOnHeap) {
      nothingOnHeap = false;
    }
#endif
    Heap2::free (ptr);
  }
  
  inline int remove (void * ptr) {
    return Heap2::remove (ptr); // heap2.remove (ptr);
  }
  
  inline void clear (void) {
    Heap1::clear();
    if (!nothingOnHeap) {
      Heap2::clear(); ///heap2.clear();
	}
    nothingOnHeap = true;
  }

  inline size_t getSize (const void * ptr) {
    return Heap2::getSize (ptr);
  }

private:

  /// True iff there is nothing on heap 2.
#ifdef WIN32
  BOOL nothingOnHeap;
#else
  bool nothingOnHeap;
#endif
  //  Heap2 heap2;
};


#if 0 // USE stack for non-touching deletion stuff (for drag measurements)

// NB: The magic MaxGlobalElements here supports 1GB worth of 8K chunks.

template <class TYPE, int MaxGlobalElements = 131072>
class StaticStack
{
public:

  StaticStack (void)
    : head (NULL, NULL)
  {
  }

  /**
   * @brief Push an item onto the stack.
   * @return 0 iff the stack is full.
   */

  inline int push (TYPE v) {
    Entry * e = new Entry (v, head.next);
    head.next = e;
    return 1;
  }

  /**
   * @brief Pops an item off of the stack.
   * @return 0 iff the stack is empty.
   */

  inline int pop (TYPE& t) {
    Entry * e = head.next;
    if (e == NULL) {
      return 0;
    }
    head.next = head.next->next;
    TYPE r = e->datum;
    delete e;
    t = r;
    return 1;
  }

private:

  //	class Entry : public PerClassHeap<FreelistHeap<StaticHeap<MaxGlobalElements * (sizeof(TYPE) + sizeof(Entry *))> > > {
  class Entry : public PerClassHeap<FreelistHeap<ZoneHeap<SbrkHeap, 8192> > > {
  public:
    explicit Entry (TYPE d, Entry * n)
      : datum (d), next (n)
    {}
    TYPE datum;
    Entry * next;
  };

  Entry head;
};


template <class SuperHeap>
class RegionHeap : public SuperHeap {
public:

  ~RegionHeap (void)
  {
#if 0
    char fname[255];
    sprintf (fname, "regionstats-%d", GetCurrentProcessId());
    FILE * f = fopen (fname, "a+");
    fprintf (f, "delete region: %x\n", this);
    fclose (f);
#endif
    clear();
  }

  RegionHeap (void)
  {
#if 0
    char fname[255];
    sprintf (fname, "regionstats-%d", GetCurrentProcessId());
    FILE * f = fopen (fname, "a+");
    fprintf (f, "create region: %x\n", this);
    fclose (f);
#endif
  }

  inline void * malloc (const size_t sz) {
    void * ptr = SuperHeap::malloc (sz);
    if (!stk.push (ptr)) {
      return NULL;
    } else {
      return ptr;
    }
  }
  
  inline void clear (void) {
    void * ptr;
    while (stk.pop(ptr)) {
      SuperHeap::free (ptr);
    }
  }

private:

  StaticStack<void *> stk;

};

#else

/**
 * @class RegionHeap
 * @brief A heap layer that provides region semantics.
 */

template <class SuperHeap>
class RegionHeap : public SuperHeap {
public:

  RegionHeap (void)
    : prev (NULL)
  {}

  ~RegionHeap (void)
  {
    clear();
  }

  inline void * malloc (const size_t sz) {
    char * ch = (char *) SuperHeap::malloc (sz + sizeof(Header));

    if (ch == NULL) {
      return NULL;
    }
    
    // Put the "header" at the end of the object.
    // This is just so we can overwrite the start of the object
    // with a boundary tag (metadata) that will let us free it
    // into a coalescing heap.
    
    // The datum member points to the actual start of the object,
    // while the prev member is our linked-list pointer.
    
    Header * ptr = (Header *) (ch + sz);
    ptr->datum = ch;
    ptr->prev = prev;
    prev = ptr;

    return (void *) ch;
  }

  /**
   * @brief Checks to see if an object was allocated from this heap.
   * @return 1 iff the obj was in one of our chunks.
   */
  int find (void * obj) const {
    Header * curr = prev;
    while (curr) {
      if ((curr->datum <= obj) && (curr > obj)) {
	return 1;
      }
      curr = curr->prev;
    }
    return 0;
  }

  /// Delete everything.
  inline void clear (void) {
    while (prev) {
      Header * ptr = prev->prev;
      SuperHeap::free (prev->datum);
      prev = ptr;
    }
  }

  inline void free (void * ptr) {
    // This is not allowed here.
    abort();
  }

private:

  /// The header for allocated objects.
  class Header {
  public:
    /// Points to the start of the allocated object.
    void * datum;

    /// The previous header.
    Header * prev;
  };

  Header * prev;
  };
#endif


template <class MainStore>
class ReapTopHeap :
  public ChunkHeap<8192 - 20,
		   SlopHeap<RegionHeap<MainStore>, 16> > {};

/**
 * @class ReapBaseType
 * @brief The base implementation for reap.
 *
 */

template <class MainStore>
class ReapBaseType :
  public ClearOptimize<AddHeader<ReapTopHeap<MainStore> >,
		       LeaHeap2<ReapTopHeap<MainStore> > >
{};

/**
 * @class Reap
 * @brief A hybrid region-heap.
 *
 * This class uses a per-class freelist heap to optimize reap
 * allocation, and adds nested (hiearchical) heaps and ANSI
 * compliance.
 */

template <class MainStore>
class Reap :
  public PerClassHeap<FreelistHeap<MainStore> >, 
  public ANSIWrapper<NestedHeap<ReapBaseType<MainStore> > >
{};

#endif
