/* -*- C++ -*- */

#ifndef _MMAPHEAP_H_
#define _MMAPHEAP_H_

#ifdef _WIN32
#include <windows.h>
#else
// UNIX
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#endif


#if ALLOCATION_STATS
extern int totalAllocated;
extern int maxAllocated;
#endif


/**
 * @class MmapHeap
 * @brief A "source heap" that manages memory via calls to the VM interface.
 * @author Emery Berger
 */

class MmapHeap {
public:

#ifdef _WIN32 

#define USE_RED_ZONES 0

  void * malloc (size_t sz) {
    //  printf ("size request = %d\n", sz);
#if USE_RED_ZONES
    char * ptr = (char *) VirtualAlloc (NULL, sz + 2 * 4096, MEM_RESERVE | MEM_COMMIT | MEM_TOP_DOWN, PAGE_EXECUTE_READWRITE);
    // Establish red-zones.
    DWORD oldProtect;
    VirtualProtect ((void *) ptr, 4096, PAGE_NOACCESS, &oldProtect);
    VirtualProtect ((void *) ptr[sz + 4096], 4096, PAGE_NOACCESS, &oldProtect);
    //printf ("VirtualAlloc %x\n", &ptr[4096]);
    return (void *) &ptr[4096]; 
#else
    // Round up sz to the nearest 64K.
    //sz = (sz + 65535) & ~65535 ;
    //    cout << "sz = " << sz << endl;
    char * ptr = (char *) VirtualAlloc (NULL, sz, MEM_COMMIT | MEM_TOP_DOWN, PAGE_EXECUTE_READWRITE);
//    cout << "MMAP ptr = " << (void *) ptr << endl;
    //  fprintf (stderr, "VirtualAlloc %d = %x\n", sz, ptr); // FIX ME
#if ALLOCATION_STATS
    totalAllocated += sz;
    if (totalAllocated > maxAllocated) {
      maxAllocated = totalAllocated;
    }
#endif
    return (void *) ptr;
#endif
  }
  
  void free (void * ptr) {
    MEMORY_BASIC_INFORMATION mbi;
    VirtualQuery (ptr, &mbi, sizeof(mbi));
    //    fprintf (stderr, "VirtualFree %x (size = %d)\n", ptr, mbi.RegionSize);
#if ALLOCATION_STATS
    totalAllocated -= mbi.RegionSize;
#endif
    VirtualFree (ptr, mbi.RegionSize, MEM_DECOMMIT);
  }

  int remove (void * ptr) {
    free (ptr);
    return 1;
  }

  inline static size_t getSize (const void * ptr) {
    MEMORY_BASIC_INFORMATION mbi;
    VirtualQuery (ptr, &mbi, sizeof(mbi));
    return (size_t) mbi.RegionSize;
  }

#else

  MmapHeap (void) {
    // Get memory that is not backed by swap.
    fileDescriptor = ::open ("/dev/zero", O_RDWR);
  }

  ~MmapHeap (void) {
    ::close (fileDescriptor);
  }

  // UNIX
  void * malloc (const size_t sz) {
    // printf ("mmap size request = %d\n", sz); fflush(stdout);
    double * ptr = (double *) mmap (NULL, sz + sizeof(double), PROT_READ | PROT_WRITE | PROT_EXEC, MAP_PRIVATE, fileDescriptor, 0);
    // printf ("ptr mmapped = %x\n", ptr + 1);
    // Write the size of the mmapped space into the header.
    *((int *) ptr) = sz;
    return (void *) (ptr + 1);
  }
  
  void free (void * ptr) {
    munmap ((char *) ((double *) ptr - 1), getSize(ptr));
  }

  int remove (void * ptr) {
    free (ptr);
    return 1;
  }

  /// Return the size of an object.
  inline static size_t getSize (void * ptr) {
    return *((int *) ((double *) ptr - 1));
  }

private:

  /// A file descriptor, used for calls to <TT>mmap</TT> and <TT>munmap</TT>.
  int fileDescriptor;


#endif

};


#endif
