//
//  Heap Layers
//
//  www.heaplayers.org
//  www.cs.umass.edu/~emery
//
//  Copyright (C) 2000 - 2002 by Emery Berger
//

/**
 * @file customheap.cpp
 * @brief Link with this file to replace malloc etc. in your application.
 */

#ifdef _MSC_VER
// 4786: Disable warnings about long (> 255 chars) identifiers.
// 4512: Disable warnings about assignment operators.
#pragma warning(disable:4786 4512)
// Set inlining to the maximum possible depth.
#pragma inline_depth(255)
#endif

#include <stdlib.h>
#include <string.h>
#include <new.h>

/**
 * @def   REPLACE_ALLOCATOR
 * @brief Define this as 1 to replace malloc and friends.
 */

#ifndef REPLACE_ALLOCATOR
#define REPLACE_ALLOCATOR 0
#endif

/**
 * @def   REPLACE_PREFIX
 * @brief Define this as to replace the implementation of malloc and friends.
 *
 * Possible prefixes include:<BR>
 *   dl = Lea allocator (dlmalloc)<BR>
 *   k  = Kingsley allocator (kmalloc)<BR>
 *   hl = Heap Layers allocator (hlmalloc)<BR>
 */

#ifndef REPLACE_PREFIX
#define REPLACE_PREFIX(n) hl##n
// #define REPLACE_PREFIX(n) dl##n
// #define REPLACE_PREFIX(n) k##n
#endif


/**
 * @def ALLOCATION_STATS
 * @brief Define this as 1 to track memory allocated & requested.
 */

#ifndef ALLOCATION_STATS
#define ALLOCATION_STATS 0
#endif


#include "heaplayers.h"
#include "kingsleyheap.h"
#include "dlheap.h"
#include "sbrkheap.h"


#if 0 // KINGSLEY + Deferred COALESCING

class TopHeap :
  public RequireCoalesceable<UniqueHeap<CoalesceableHeap<SbrkHeap> > > {};

/// A "loose" Kingsley heap (power-of-two, but with splitting and coalescing).
template <class PerClassHeap, class BigHeap>
class KingsleyHeapLoose :
  public SegHeap<29,
		 Kingsley::pow2Class,
		 Kingsley::class2Size,
		 PerClassHeap,
		 BigHeap> {};

template <class SuperHeap>
class KingsleyThreshold : public SuperHeap {
public:

  KingsleyThreshold (void)
    : totalRequested (0),
      maxRequested (0),
      totalFreed (0)
  {}

  inline void * malloc (size_t sz) {
    void * ptr = SuperHeap::malloc (sz);
    if (ptr != NULL) {
      totalRequested += getSize (ptr);
      if (totalRequested > maxRequested) {
	maxRequested = totalRequested;
      }
      if (totalFreed > getSize(ptr)) {
	totalFreed -= getSize(ptr);
      }
    }
    return ptr;
  }

  inline void free (void * ptr) {
    totalFreed += getSize (ptr);
    SuperHeap::free (ptr);
    if (totalFreed * 4 > maxRequested) {
      // printf ("free all !\n");
      SuperHeap::freeAll();
      totalFreed = 0;
    }
  }

private:
  size_t totalRequested;
  size_t maxRequested;
  size_t totalFreed;
};


class KH2: public KingsleyHeapLoose<AdaptHeap<DLList, TopHeap>, AdaptHeap<DLList, TopHeap> > {};
class KH2X : public UniqueHeap<KH2> {};
class KH1 : public KingsleyHeap<KH2X, KH2X> {};
class T1 : public KingsleyThreshold<KH1> {};

class TheCustomHeapType :
  public ANSIWrapper<CoalesceHeap<TryHeap<RequireCoalesceable<T1>, TopHeap> > > {};

#endif



#if 0 // KINGSLEY + COALESCING

class TopHeap : public RequireCoalesceable<UniqueHeap<CoalesceableHeap<SlopHeap<SbrkHeap> > > > {};

template <class PerClassHeap, class BigHeap>
class KingsleyHeap2 : public SegHeap<29, Kingsley::pow2Class, Kingsley::class2Size, PerClassHeap, BigHeap> {};

class TheCustomHeapType :
  public ANSIWrapper<CoalesceHeap<RequireCoalesceable<KingsleyHeap2<AdaptHeap<DLList, TopHeap>, TryHeap<AdaptHeap<DLList, TopHeap>, TopHeap> > > > > {};

#endif

#if 0 // KINGSLEY, no coalescing

class TopHeap : public SizeHeap<UniqueHeap<SbrkHeap> > {};

class TheCustomHeapType :
  public ANSIWrapper<KingsleyHeap<AdaptHeap<DLList, TopHeap>, TopHeap> > {};

#endif

#if 0 // Locked KINGSLEY + COALESCING
class TopHeap : public RequireCoalesceable<UniqueHeap<CoalesceableHeap<SbrkHeap> > > {};

template <class PerClassHeap, class BigHeap>
class KingsleyHeap2 : public SegHeap<29, Kingsley::pow2Class, Kingsley::class2Size, PerClassHeap, BigHeap> {};

class TheCustomHeapType :
  public ANSIWrapper<DebugHeap<LockedHeap<CoalesceHeap<RequireCoalesceable<KingsleyHeap2<AdaptHeap<DLList, TopHeap>, TryHeap<DebugHeap<AdaptHeap<DLList, TopHeap> >, TopHeap> > > > > > > {};

#endif


#if 0 // Locked KINGSLEY, no coalescing
class TopHeap : public SizeHeap<UniqueHeap<SbrkHeap> > {};

class TheCustomHeapType :
  public ANSIWrapper<LockedHeap<KingsleyHeap<AdaptHeap<DLList, TopHeap>, TopHeap> > > {};
#endif


#if 1 // LeaHeap

#include "oneheap.h"
#include "regionheap.h"
#include "chunkheap.h"

// Lea Heap
// class TheCustomHeapType : public ANSIWrapper<LeaHeap<SlopHeap<SbrkHeap>, MmapHeap> > {};

/// The Lea heap.
class TheCustomHeapType :
  public ANSIWrapper<
  LeaHeap<
  SlopHeap<RegionHeap<ChunkHeap<65536, MmapHeap> > >,
  RegionHeap<MmapHeap> > > {};

#endif

#if 0 // Locked Lea Heap
class TheCustomHeapType : public ANSIWrapper<LockedHeap<LeaHeap<SbrkHeap, MmapHeap> > > {};
#endif

#if 0 // log heap
#include "logheap.h"
#define USE_ADIOS 1

class TheCustomHeapType: public ANSIWrapper<LogHeap<LeaHeap<SbrkHeap, MmapHeap> > > {};
#endif


#if 0 // sanity check heap
#include "sanitycheckheap.h"
//class TheCustomHeapType: public SanityCheckHeap<ANSIWrapper<LeaHeap<SbrkHeap, MmapHeap> > > {};
class TheCustomHeapType: public SanityCheckHeap<ANSIWrapper<mallocHeap> > {};
#endif


#if 0 // pure-private heaps Lea allocator
#include "threadspecificheap.h"
#include "winlock.h"
class TheCustomHeapType : public ANSIWrapper<ThreadHeap<32, LockedHeap<WinLockType, LeaHeap<SbrkHeap, MmapHeap> > > > {};

#endif


#if 0
extern char logfilename[255] = "espressolog";
class TheCustomHeapType : public TraceHeap<mallocHeap, logfilename> {};
#endif

//#define HEAPLAYERS_PREFIX(n) hl##n
#define HEAPLAYERS_PREFIX(n) n

#define CUSTOM_MALLOC(x) HEAPLAYERS_PREFIX(malloc)(x)
#define CUSTOM_FREE(x) HEAPLAYERS_PREFIX(free)(x)
#define CUSTOM_REALLOC(x,y) HEAPLAYERS_PREFIX(realloc)(x,y)
#define CUSTOM_CALLOC(x,y) HEAPLAYERS_PREFIX(calloc)(x,y)
#define CUSTOM_GETSIZE(x) HEAPLAYERS_PREFIX(malloc_usable_size)(x)

static char theCustomHeapBuffer[sizeof(TheCustomHeapType)];

// A static method that guarantees initialization of the custom heap
// and at the same time prevents it from being destroyed (possibly prematurely).
inline static TheCustomHeapType * getCustomHeap (void) {
  static TheCustomHeapType * theHeap = new ((void *) &theCustomHeapBuffer[0]) TheCustomHeapType;
  return theHeap;
}

#ifdef USE_ADIOS
// This should only be used for the LogHeap.
class Adios {
public:
  ~Adios (void) {
    getCustomHeap()->TheCustomHeapType::~TheCustomHeapType();
  }
};
Adios adios;
#endif

/// malloc wrapper.
extern "C" void * CUSTOM_MALLOC (size_t sz)
{
  static TheCustomHeapType * theCustomHeap = getCustomHeap();
  void * ptr = theCustomHeap->malloc (sz);
  return ptr;
}

/// freeAll wrapper.
extern "C" void CUSTOM_FREEALL (void)
{
  static TheCustomHeapType * theCustomHeap = getCustomHeap();
  theCustomHeap->clear();
}

/// calloc wrapper.
extern "C" void * CUSTOM_CALLOC (size_t nelem, size_t elsize)
{
  static TheCustomHeapType * theCustomHeap = getCustomHeap();
  void * ptr = theCustomHeap->malloc (nelem * elsize);
  // Zero out the malloc'd block.
  if (ptr != NULL) {
    memset (ptr, 0, nelem * elsize);
  }
  return ptr;
}

/// getsize wrapper.
extern "C" size_t CUSTOM_GETSIZE (void * ptr)
{
  static TheCustomHeapType * theCustomHeap = getCustomHeap();
  if (ptr == NULL) {
    return 0;
  }
  return theCustomHeap->getSize(ptr);
}

/// free wrapper.
extern "C" void CUSTOM_FREE (void * ptr)
{
  static TheCustomHeapType * theCustomHeap = getCustomHeap();
  theCustomHeap->free (ptr);
}

/// realloc wrapper.
extern "C" void * CUSTOM_REALLOC (void * ptr, size_t sz)
{
  static TheCustomHeapType * theCustomHeap = getCustomHeap();
  if (ptr == NULL) {
    ptr = theCustomHeap->malloc (sz);
    return ptr;
  }
  if (sz == 0) {
    theCustomHeap->free (ptr);
    return NULL;
  }


#if 0
  
  // We use size doubling to provide hysteresis in
  // object growth/shrinking.
  // If the old size is at least as big as the new size
  // and not more than twice as big, just return it.

  size_t objSize = theCustomHeap->getSize (ptr);
  if ((objSize >= sz) && (objSize < (size_t) (2 * sz))) {
    return ptr;
  }

  // Allocate a new block of size 2 * sz.

  void * buf = theCustomHeap->malloc ((size_t) (2 * sz));
#else

  size_t objSize = theCustomHeap->getSize(ptr);
  if (objSize >= sz) {
    return ptr;
  }

  void * buf = theCustomHeap->malloc ((size_t) (sz));

#endif

  // Copy the contents of the original object
  // up to the size of the new block.

  size_t minSize = (objSize < sz) ? objSize : sz;
  memcpy (buf, ptr, minSize);

  // Free the old block.

  theCustomHeap->free(ptr);

  // Return a pointer to the new one.

  return buf;
}


#if REPLACE_ALLOCATOR

#define REPLACE_MALLOC(x) REPLACE_PREFIX(malloc)(x)
#define REPLACE_FREE(x) REPLACE_PREFIX(free)(x)
#define REPLACE_REALLOC(x,y) REPLACE_PREFIX(realloc)(x,y)
#define REPLACE_CALLOC(x,y) REPLACE_PREFIX(calloc)(x,y)
#define REPLACE_GETSIZE(x) REPLACE_PREFIX(malloc_usable_size)(x)

extern "C" void * REPLACE_MALLOC(size_t);
extern "C" void REPLACE_FREE(void *);
extern "C" void * REPLACE_REALLOC(void *, size_t);
extern "C" void * REPLACE_CALLOC(size_t, size_t);
extern "C" size_t REPLACE_GETSIZE(void *);


extern "C" void * malloc (size_t sz) {
  return REPLACE_MALLOC (sz);
}

extern "C" void free (void * ptr) {
  REPLACE_FREE (ptr);
}

extern "C" void * realloc (void * ptr, size_t sz) {
  return REPLACE_REALLOC (ptr, sz);
}

extern "C" void * calloc (size_t num, size_t sz) {
  return REPLACE_CALLOC (num, sz);
}

#endif // REPLACE_ALLOCATOR

