/* -*- C++ -*- */

#ifndef _STLALLOCATOR_H_
#define _STLALLOCATOR_H_

#include <new.h>
#include <stdlib.h>

#include <memory> // STL

/**
 * @class STLAllocator
 * @brief An allocator adapter for STL.
 * 
 * This mixin 
 * Example:
 * <TT>
 *   typedef STLAllocator<int, MyHeapType> MyAllocator;<BR>
 *   list<int, MyAllocator> l;<BR>
 * </TT>
 */

template <class T, class SuperHeap>
class STLAllocator {
public:

  typedef T value_type;
  typedef size_t size_type;
  typedef ptrdiff_t difference_type;
  typedef T * pointer;
  typedef const T * const_pointer;
  typedef T& reference;
  typedef const T& const_reference;

  inline STLAllocator (void) { }

#if _WIN32
  char * _Charalloc (size_type n) {
    return (char *) allocate (n);
  }
#endif

  inline pointer allocate (size_type n,
			   std::allocator<void>::const_pointer hint = 0) {
    return (pointer) getHeap().malloc (sizeof(T) * n);
  }
  inline void deallocate (void * p, size_type n) {
    getHeap().free (p);
  }
  inline void deallocate (pointer p, size_type n) {
    getHeap().free (p);
  }
  
  void construct (pointer p, const T& val) { new (p) T (val); }
  void destroy (pointer p) { p->~T(); }

  /// Make the maximum size be the largest possible object.
  size_type max_size(void) const
  {
    size_type n = (size_type) (-1);
    return (n > 0 ? n : (unsigned) n);
  }

  template <class U> STLAllocator( const STLAllocator<U, SuperHeap> & other) {}
  template <class U> struct rebind { typedef STLAllocator<U,SuperHeap> other; };

private:

  /// Maintain exactly one instance of the SuperHeap.
  SuperHeap& getHeap (void) {
    static SuperHeap theHeap;
    return theHeap;
  }
};


#endif
