/**
 * @file regionheap.cpp
 * @author Emery Berger
 * 
 * The implementation of the reap functionality.  Note that the API
 * uses "region*" and not "reap*" as the naming convention.
 *
 * Note that we use DLmalloc (via LeaMallocHeap) as the main store
 * (for obtaining chunks of memory) to take advantage of the memory
 * accounting that we have added to dlmalloc.c. To use the generic
 * memory allocation functions, replace MAIN_ALLOCATOR with
 * mallocHeap. We also define replacement functions for the malloc
 * family, called reapmalloc, etc.
 *
 * @sa regionheap.h
 */

#include <iostream.h>
#include <new.h>

#define MAIN_ALLOCATOR LeaMallocHeap
// #define MAIN_ALLOCATOR mallocHeap

extern "C" void   regionCreate   (void ** reg, void ** parent);
extern "C" void   regionDestroy  (void ** reg);
extern "C" void * regionAllocate (void ** reg, size_t sz);
extern "C" void   regionFreeAll  (void ** reg);
extern "C" void   regionFree     (void ** reg, void * ptr);
extern "C" int    regionFind     (void ** reg, void * ptr);

#include "heaplayers.h"
#include "slopheap.h"
#include "regionheap.h"
#include "chunkheap.h"
#include "kingsleyheap.h"
#include "oneheap.h"
#include "regionheapapi.h" 
#include "nestedheap.h"

#define LOCAL_ALLOCATION_STATS 0

extern "C" void dlmalloc_stats(void);
extern "C" void * dlmalloc(size_t);
extern "C" void dlfree (void *);
extern "C" size_t dlmalloc_usable_size(void *);

int ReapMaxMemory = 0;
int ReapCurrentMemory = 0;

#if LOCAL_ALLOCATION_STATS

template <class S>
class MeasureMemory : public S {
public:

  MeasureMemory (void) {
    //		fprintf (stderr, "REAP!\n");
  }

  ~MeasureMemory (void) {
    char str[255];
    sprintf (str, "moop-%d", GetCurrentProcessId());
    FILE * f = fopen(str, "w+");
    fprintf (f, "Max memory in use = %d\n", ReapMaxMemory);
    fclose (f);
    // dlmalloc_stats();
  }

  inline void * malloc (size_t sz) {
    void * ptr = S::malloc (sz);
    ReapCurrentMemory += sz;
    if (ReapCurrentMemory > ReapMaxMemory) {
      ReapMaxMemory = ReapCurrentMemory;
    }
    return ptr;
  }
  inline void free (void * ptr) {
    ReapCurrentMemory -= getSize (ptr);
    S::free (ptr);
  }
};
#else
template <class S>
class MeasureMemory : public S {};
#endif

class ReportStats {
public:
  ReportStats (void) {
    st = (size_t) sbrk(0);
  }

  ~ReportStats (void) {
    //		dlmalloc_stats();
    fprintf (stderr, "sbrk = %d\n", (size_t) sbrk(0));
    fprintf (stderr, "start = %d\n", st);
    fprintf (stderr, "Max Reap memory in use = %d\n", ReapMaxMemory);
  }
private:
  size_t st;
};


class LeaMallocHeap1 {
public:
  LeaMallocHeap1 (void)
    : currentMemory (0),
      maxMemory (0)
  {}

  ~LeaMallocHeap1 (void) {
    char str[255];
    sprintf (str, "moop-%d-%d", GetCurrentProcessId(), (size_t) this);
    FILE * f = fopen(str, "w+");
    fprintf (f, "Curr = %d, maxMemory = %d\n", currentMemory, maxMemory);
    fclose (f);
  }

  inline void * malloc (size_t sz) {
    void * ptr = dlmalloc (sz);
    currentMemory += getSize(ptr);
    if (currentMemory > maxMemory) {
      maxMemory = currentMemory;
    }
    return ptr;
  }

  inline void free (void * p) {
    currentMemory -= getSize(p);
    dlfree (p);
  }

  inline size_t getSize (const void * p) {
    return dlmalloc_usable_size ((void *) p);
  }
private:
  size_t currentMemory;
  size_t maxMemory;
};


// The Real Deal.

#if LOCAL_ALLOCATION_STATS
class MainStore :  public OneHeap<MeasureMemory<LeaMallocHeap> > {};
#else
class MainStore : public LeaMallocHeap {};
#endif

class TheMmapHeap : public MainStore {};
class OneStore : public MainStore {};
class TopHeap :
  public ChunkHeap<8192 - 20, SlopHeap<RegionHeap<OneStore>, 16> > {};

template <class SuperHeap>
class CountingFreelistHeap : public SuperHeap {
public:
  
  inline CountingFreelistHeap (void)
    : myFreeList (NULL)
    ,nObjects (0)
  {}

  inline ~CountingFreelistHeap (void)
  {
  }

  inline void * malloc (const size_t sz) {
    // Check the free list first.
    void * ptr = myFreeList;
    if (ptr == NULL) {
      assert (nObjects == 0);
      ptr = SuperHeap::malloc (sz);
    } else {
      assert (nObjects > 0);
      myFreeList = myFreeList->next;
      nObjects--;
    }
    return ptr;
  }
  
  inline void free (void * ptr) {
    // Add this object to the free list.
    assert (ptr != NULL);
    nObjects++;
    ((freeObject *) ptr)->next = myFreeList;
    myFreeList = (freeObject *) ptr;
  }

  inline void clear (void) {
    freeObject * ptr = myFreeList;
    while (ptr) {
      void * p = ptr;
      ptr = ptr->next;
      SuperHeap::free (p);
    }
    nObjects = 0;
    //    SuperHeap::clear();
  }

  inline int getCached (void) const {
    return nObjects;
  }

private:

  /// The linked list pointer we embed in the freed objects.
  class freeObject {
  public:
    freeObject * next;
  };

  /// The head of the linked list of freed objects.
  freeObject * myFreeList;
#ifndef NDEBUG
  int nObjects;
#endif

};


template <float f, class SuperHeap>
class ThresholdFreelistHeap : public SuperHeap {
public:
  ThresholdFreelistHeap (void)
    :
    currAllocated (0),
    maxAllocated (0)
  {}
  
  inline void free (void * ptr) {
    // If we exceed the limit, clear the superheap.
    if (getCached() + 1 > f * maxAllocated) {
      SuperHeap::clear ();
    } else {
      SuperHeap::free (ptr);
    }
    currAllocated--;
  }
  
  inline void malloc (const size_t sz) {
    void * ptr = SuperHeap::malloc (sz);
    currAllocated++;
    if (currAllocated > maxAllocated) {
      maxAllocated = currAllocated;
    }
    return ptr;
  }
  
  inline void clear (void) {
    currAllocated = 0;
    SuperHeap::clear();
  }
  
private:
  int currAllocated;
  int maxAllocated;
};

class ReapletStore : public ThresholdFreelistHeap<.2,
		     CountingFreelistHeap<MainStore> > {};

class ReapBaseType1 :
  public  PerClassHeap<ReapletStore>, 
  // This one is also good: uses leaheap instead of leamallocheap.
  public ClearOptimize<AddHeader<TopHeap>, LeaHeap2<TopHeap> >
{};


// Replace malloc, free, and friends.

#define CUSTOM_MALLOC(x)    reapmalloc(x)
#define CUSTOM_CALLOC(x,y)  reapcalloc(x,y)
#define CUSTOM_GETSIZE(x)   reapmalloc_usable_size(x)
#define CUSTOM_REALLOC(x,y) reaprealloc(x,y)
#define CUSTOM_FREE(x)      reapfree(x)


inline static Reap<MAIN_ALLOCATOR> * getHeapFunction (void) 
{
  static char buf[sizeof(Reap<MAIN_ALLOCATOR>)];
  static Reap<MAIN_ALLOCATOR> * theHeap = ::new (buf) Reap<MAIN_ALLOCATOR>;
  return theHeap;
}


#if LOCAL_ALLOCATION_STATS
class FOOP {
public:
  FOOP (void) {
    //printf ("HEAP: %d\n", sizeof(Reap<MAIN_ALLOCATOR>));
    //printf ("heap base: %d\n", sizeof(ReapBaseType1));
    //printf ("TopHeap: %d\n", sizeof(TopHeap));
    //printf ("LittleHeap: %d\n", sizeof(DLSmallHeapType<MainStore>));
    //printf ("BigHeap: %d\n", sizeof(DLBigHeapType<MainStore>));
    st = sbrk(0);
  }
  ~FOOP (void) {
    char fname[255];

    sprintf (fname, "rstats-%d", GetCurrentProcessId());

    FILE * f = fopen (fname, "a+");

    fprintf (f, "start = %d\n", (size_t) st);
    fprintf (f, "end = %d\n", (size_t) sbrk(0));
    //dlmalloc_stats();
    fprintf (f, "Max memory in use = %d\n", ReapMaxMemory);
    fclose (f);
    //fprintf (stderr, "sizeof heap = %d\n", sizeof(Reap<MAIN_ALLOCATOR>));
    //fprintf (stderr, "max memory = %d\n", maxMemory);
  }
private:
  void * st;
};

FOOP foop;
#endif



#if 1

extern "C" void * CUSTOM_MALLOC (size_t sz)
{
  return getHeapFunction()->malloc (sz);
}

extern "C" void * CUSTOM_CALLOC (size_t nelem, size_t elsize)
{
  Reap<MAIN_ALLOCATOR> * theCustomHeap = getHeapFunction();
  void * ptr = theCustomHeap->malloc (nelem * elsize);
  // Zero out the malloc'd block.
  if (ptr != NULL) {
    memset (ptr, 0, nelem * elsize);
  }
  return ptr;
}

extern "C" size_t CUSTOM_GETSIZE (void * ptr)
{
  Reap<MAIN_ALLOCATOR> * theCustomHeap = getHeapFunction();
  if (ptr == NULL) {
    return 0;
  }
  return theCustomHeap->getSize((void *) ptr);
}


extern "C" void CUSTOM_FREE (void * ptr)
{
  getHeapFunction()->free (ptr);
}


extern "C" void * CUSTOM_REALLOC (void * ptr, size_t sz)
{
  return getHeapFunction()->realloc (ptr, sz);

#if 0
  Reap<MAIN_ALLOCATOR> * theCustomHeap = getHeapFunction();
  if (ptr == NULL) {
    ptr = theCustomHeap->malloc (sz);
    return ptr;
  }
  if (sz == 0) {
    theCustomHeap->free (ptr);
    return NULL;
  }

  size_t objSize = theCustomHeap->getSize(ptr);
  if (objSize >= sz) {
    return ptr;
  }

  void * buf = theCustomHeap->malloc ((size_t) (sz));

  // Copy the contents of the original object
  // up to the size of the new block.

  size_t minSize = (objSize < sz) ? objSize : sz;
  memcpy (buf, ptr, minSize);

  // Free the old block.

  theCustomHeap->free(ptr);

  // Return a pointer to the new one.

  return buf;
#endif
}
#endif

extern "C" void regionCreate (void ** reg, void ** parent)
{
  Reap<MAIN_ALLOCATOR> * psr;
  psr = new Reap<MAIN_ALLOCATOR> ();
  if (parent) {
    (*((Reap<MAIN_ALLOCATOR> **) parent))->addChild (psr);
  }

  //  fprintf(stderr, "regionCreate %x = %x\n", parent, psr);

  *((Reap<MAIN_ALLOCATOR> **) reg) = psr;
}


extern "C" void regionDestroy (void ** reg)
{
  delete ((Reap<MAIN_ALLOCATOR> *) *reg);
  *reg = NULL;
}

extern "C" void * regionAllocate (void ** reg, size_t sz)
{
  void * ptr = ((Reap<MAIN_ALLOCATOR> *) *reg)->malloc (sz);
  return ptr;
}

extern "C" void regionFreeAll (void ** reg)
{
  ((Reap<MAIN_ALLOCATOR> *) *reg)->clear ();
}

extern "C" void regionFree (void ** reg, void * ptr)
{
  ((Reap<MAIN_ALLOCATOR> *) *reg)->free (ptr);
}


extern "C" int regionFind (void ** reg, void * ptr)
{
  return ((Reap<MAIN_ALLOCATOR> *) *reg)->find (ptr);
}




