/* -*- C++ -*- */

//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2002, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
 * @file mallocwrap.cpp
 *
 * Wrappers that re-route everything, including the Win32 API, to
 * ordinary ANSI C calls.
 *
 * @author Emery Berger
 */

#if defined(WIN32)
#include <windows.h>
#endif

#include <stdlib.h>
#include <new>
#include <stdlib.h>
#include <stdio.h>

#define USE_REAP 0
#define USE_DLMALLOC 1
#define USE_VMALLOC 0
#define USE_HMALLOC 0

#if (USE_REAP + USE_DLMALLOC + USE_VMALLOC + USE_HMALLOC > 1)
#error "Pick only one heap implementation."
#endif

#if USE_REAP
#define CUSTOM_PREFIX(x) reap##x
#endif

#if USE_DLMALLOC
#define CUSTOM_PREFIX(x) dl##x
#endif

#if USE_VMALLOC
#define CUSTOM_PREFIX(x) vm_##x
#endif

#if USE_HMALLOC
#define CUSTOM_PREFIX(x) hoard_##x
#endif

#if (USE_REAP + USE_DLMALLOC + USE_VMALLOC + USE_HMALLOC > 0)

// We are using a substitute heap, so implement replacement wrappers.

#include <crtdbg.h>
#include <malloc.h>

extern "C" {

void * CUSTOM_PREFIX(malloc) (size_t);
void   CUSTOM_PREFIX(free) (void *);
void * CUSTOM_PREFIX(calloc) (size_t, size_t);
void * CUSTOM_PREFIX(realloc) (void *, size_t);

#if !(USE_VMALLOC)
size_t CUSTOM_PREFIX(malloc_usable_size) (void *);
#endif

void * malloc (size_t sz) {
  return CUSTOM_PREFIX(malloc) (sz);
}

void free (void * ptr) {
  CUSTOM_PREFIX(free) (ptr);
}

void * calloc (size_t s, size_t n) {
  return CUSTOM_PREFIX(calloc) (s, n);
}

void * realloc (void * ptr, size_t sz) {
  return CUSTOM_PREFIX(realloc) (ptr, sz);
}

#if !(USE_VMALLOC)
size_t malloc_usable_size (void * ptr) {
  return CUSTOM_PREFIX(malloc_usable_size) (ptr);
}
#endif
}

void * operator new (size_t size)
{
  return malloc (size);
}

void * operator new (size_t size, const std::nothrow_t&) throw() {
  return malloc (size);
} 

void * operator new[] (size_t size)
{
  return malloc (size);
}

void * operator new[] (size_t size, const std::nothrow_t&) throw() {
  return malloc (size);
} 

void operator delete (void * ptr)
{
  free (ptr);
}

void operator delete[] (void * ptr)
{
  free (ptr);
}


#if 0 // defined(WIN32)

#if defined(_MSC_VER) // Microsoft Visual C++
#pragma warning(disable: 4074)	// initializers put in compiler reserved area
#pragma init_seg(compiler)
#pragma comment(linker, "/merge:.CRT=.data")
#pragma comment(linker, "/disallowlib:libc.lib")
#pragma comment(linker, "/disallowlib:libcd.lib")
#pragma comment(linker, "/disallowlib:libcmt.lib")
#pragma comment(linker, "/disallowlib:libcmtd.lib")
#pragma comment(linker, "/disallowlib:msvcrt.lib")
#pragma comment(linker, "/disallowlib:msvcrtd.lib")
//#define EXPORT __declspec(dllexport)
#define EXPORT

#else

#define EXPORT

#endif

// Replace the CRT debugging allocation routines, just to be on the
// safe side.  This is not a complete solution, but should avoid
// inadvertent mixing of allocations.

EXPORT void * _heap_alloc_base (size_t size)
{
  return malloc (size);
}


#if !(USE_VMALLOC)
EXPORT size_t _msize (void * ptr) { return malloc_usable_size (ptr); }
EXPORT size_t _msize_dbg (void * ptr, int) { return malloc_usable_size (ptr); }
EXPORT size_t _msize_base (void * ptr) { return malloc_usable_size (ptr); }
#endif

void * operator new (unsigned int cb, int, const char *, int)
{
  return ::operator new (cb);
}

#if 0
void operator delete(void * p, int, const char *, int)
{
  ::operator delete(p);
}
#endif

EXPORT void * debug_operator_new (unsigned int cb, int, const char *, int)
{ return malloc (cb); }
EXPORT void   debug_operator_delete (void * p, int, const char *, int) { free (p); }
EXPORT void * _malloc_dbg (size_t sz, int, const char *, int) { return malloc (sz); }
EXPORT void * _malloc_base (size_t sz) { return malloc (sz); }
EXPORT void *  _calloc_dbg (size_t num, size_t size, int, const char *, int) { return calloc (num, size); }
EXPORT void *  _calloc_base (size_t num, size_t size) { return calloc (num, size); }
EXPORT void *  _realloc_dbg (void * ptr, size_t newSize, int, const char *, int) { return realloc (ptr, newSize); }
EXPORT void *  _realloc_base (void * ptr, size_t newSize) { return realloc (ptr, newSize); }
EXPORT void    _free_dbg (void * ptr, int) { free (ptr); }
EXPORT void    _free_base (void * ptr) { free (ptr); }

EXPORT int     _CrtCheckMemory(void) { return 1; }
EXPORT void    _CrtDoForAllClientObjects (void(*pfn)(void*,void*), void *context) {}
EXPORT int     _CrtDumpMemoryLeaks(void) { return 0; }
EXPORT int     _CrtIsMemoryBlock (const void *, unsigned int, long *, char **, int *) { return 1; }
EXPORT int     _CrtIsValidHeapPointer (const void *) { return 1; }
EXPORT void    _CrtMemCheckpoint (_CrtMemState *) { }
EXPORT int     _CrtMemDifference (_CrtMemState *, const _CrtMemState *, const _CrtMemState *) { return 0; }
EXPORT void    _CrtMemDumpAllObjectsSince (const _CrtMemState *) { }
EXPORT void    _CrtMemDumpStatistics (const _CrtMemState *) { }
EXPORT _CRT_ALLOC_HOOK _CrtSetAllocHook (_CRT_ALLOC_HOOK) { return 0; }
EXPORT long    _CrtSetBreakAlloc (long) { return 0; }
EXPORT int     _CrtSetDbgFlag (int) { return _CRTDBG_ALLOC_MEM_DF; }
EXPORT _CRT_DUMP_CLIENT _CrtSetDumpClient (_CRT_DUMP_CLIENT) { return 0; }
EXPORT int     _heapchk (void) { return _HEAPOK; }
EXPORT int     _heapmin (void) { return 0; }
EXPORT int     _heapset (unsigned int) { return _HEAPOK; }
EXPORT int     _heapwalk (_HEAPINFO *) { return _HEAPEND; }

EXPORT void * new_nothrow (unsigned int sz,struct std::nothrow_t const &) { return malloc (sz); }

EXPORT void *  _expand (void *, size_t) { return NULL; }
EXPORT void *  _expand_base (void *, size_t) { return NULL; }
EXPORT void *  _expand_dbg (void *, size_t, int, const char *, int) { return NULL; }

EXPORT void *  _nh_malloc (size_t sz, int) { return malloc (sz); }
EXPORT void *  _nh_malloc_base (size_t sz, int) { return malloc (sz); }
EXPORT void *  _nh_malloc_dbg (size_t sz, size_t, int, int, const char *, int) { return malloc (sz); }

EXPORT void assert_failed (const char *, const char *, int) {  __asm int 3 }

// We have to replace atexit & _onexit with a version that doesn't use
// the heap.  So we make a reasonably large static buffer and hope for
// the best.

typedef void (__cdecl * exitFunction)(void);

enum { MAX_EXIT_FUNCTIONS = 16384 };

exitFunction exitFunctionBuffer[MAX_EXIT_FUNCTIONS];
int exitFunctionCount = 0;


void __cdecl _exit(int);
void __cdecl exit(int);

EXPORT void __cdecl doexit(int code,int quick,int retcaller)
{
  if (quick) {
    _exit(code);
  } else {
    exit(code);
  }
}


EXPORT int __cdecl atexit (void (__cdecl * fn)(void)) {
  if (exitFunctionCount == MAX_EXIT_FUNCTIONS) {
    return 1;
  } else {
    exitFunctionBuffer[exitFunctionCount] = fn;
    exitFunctionCount++;
    return 0;
  }
}


EXPORT _onexit_t __cdecl _onexit(_onexit_t fn)
{
  if (atexit((exitFunction) fn) == 0) {
    return fn;
  } else {
    return NULL;
  }
}

void __cdecl _exit(int);
int __cdecl _fcloseall (void);

EXPORT void __cdecl exit (int r) {

  // As prescribed, we call all exit functions in LIFO order,
  // flush all open buffers to disk, and then really exit.

  for (int i = exitFunctionCount - 1; i >= 0; i--) {
    (*exitFunctionBuffer[i])();
  }
  //  _fcloseall();

  _exit(r);
}

#endif

#endif
