#ifndef _KINGSLEYHEAP_H_
#define _KINGSLEYHEAP_H_

#include "segheap.h"

/**
 * @file kingsleyheap.h
 * @brief Classes to implement a Kingsley (power-of-two, segregated fits) allocator.
 */

/**
 * @namespace Kingsley
 * @brief Functions to implement KingsleyHeap.
 */

namespace Kingsley {

  size_t class2Size (const int i);

  /**
   * A speed optimization:
   * we use this array to quickly return the size class of objects
   * from 8 to 128 bytes.
   */
  const int cl[16] = { 0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4 };

  inline int pow2Class (const size_t sz) {
#if 0
    int c = 0;
    size_t sz1 = sz - 1;
    while (sz1 > 7) {
      sz1 >>= 1;
      c++;
    }
    assert (class2Size(c) >= sz);
    return c;
#else
    if (sz <= 128 ) {
      assert (class2Size(cl[sz >> 3]) >= sz);
      return cl[(sz - 1) >> 3];
    } else {
      //
      // We know that the object is more than 128 bytes long,
      // so we can avoid iterating 5 times.
      //
      int c = 5;
      size_t sz1 = ((sz - 1) >> 5);
      while (sz1 > 7) {
	sz1 >>= 1;
	c++;
      }
      assert (class2Size(c) >= sz);
      return c;
    }
#endif

  }

  inline size_t class2Size (const int i) {
    return (size_t) (1 << (i+3));
  }

  enum { NUMBINS = 29 };

}

/**
 * @class KingsleyHeap
 * @brief The Kingsley-style allocator.
 * @param PerClassHeap The heap to use for each size class.
 * @param BigHeap The heap for "large" objects.
 * @see Kingsley
 */

template <class PerClassHeap, class BigHeap>
  class KingsleyHeap : 
   public StrictSegHeap<Kingsley::NUMBINS,
			Kingsley::pow2Class,
			Kingsley::class2Size,
			PerClassHeap,
			BigHeap> {};

#endif
