/* -*- C++ -*- */

#ifndef _ANSIWRAPPER_H_
#define _ANSIWRAPPER_H_

#if ALLOCATION_STATS
extern int totalRequested;
extern int maxRequested;
#endif

#include <assert.h>
#include <stdlib.h>
#include <string.h>

/**
 * @class ANSIWrapper
 * @brief Provide ANSI C behavior for malloc & free.
 *
 * Implements all prescribed ANSI behavior, including zero-sized
 * requests & aligns all request sizes to a double word.
 */

template <class SuperHeap>
class ANSIWrapper : public SuperHeap {
 public:
  inline ANSIWrapper (void)
    {
#if ALLOCATION_STATS
      totalRequested = maxRequested = 0;
#endif
    }

  inline void * malloc (const size_t sz) {
    //	  printf ("M\n");
    //printf ("malloc %d\n", sz);

    //
    // Round up small sizes and align larger ones
    // to a double-word boundary.
    //

    const size_t sz1 = (sz < sizeof(double)) ?
      sizeof(double) :
      (sz + sizeof(double) - 1) & ~(sizeof(double) - 1);

#if !ALLOCATION_STATS || NDEBUG
    return SuperHeap::malloc (sz1);
#else
    void * ptr = SuperHeap::malloc (sz);
    //	totalRequested += sz;
#if ALLOCATION_STATS
    if (ptr != NULL) {
      totalRequested += getSize(ptr);
      if (totalRequested > maxRequested) {
	maxRequested = totalRequested;
      }
    }
#endif
#ifndef NDEBUG
    if (ptr != NULL) {
      size_t actualSize = getSize(ptr);
      assert (actualSize >= sz);
    }
#endif
    return ptr;
#endif
  }


  inline void free (void * ptr) {
    //	  printf ("F\n");
    if (ptr != NULL) {
#if ALLOCATION_STATS
      totalRequested -= getSize(ptr);
#endif
      SuperHeap::free (ptr);
    }
  }

  inline void * calloc (const size_t s1, const size_t s2) {
    void * ptr = malloc (s1 * s2);
    memset (ptr, 0, s1 * s2);
    return ptr;
  }

  inline void * realloc (void * ptr, const size_t sz) {
    if (ptr == NULL) {
      return malloc (sz);
    }
    if (sz == 0) {
      free (ptr);
      return NULL;
    }

    size_t objSize = getSize (ptr);
    if (objSize == sz) {
      return ptr;
    }

    // Allocate a new block of size sz.

    void * buf = malloc (sz);

    // Copy the contents of the original object
    // up to the size of the new block.

    size_t minSize = (objSize < sz) ? objSize : sz;

    memcpy (buf, ptr, minSize);

    // Free the old block.

    free (ptr);

    return buf;
  }

  inline size_t getSize (const void * ptr) {
    return SuperHeap::getSize (ptr);
  }

};


#endif
