/******************************
   Library:  TAQAccess_tick.h
   author:   Boulat A. Bash
   email:    boulat@alum.dartmouth.org
   created:  July     2000
   updated:  February 2004
   comments: Header file for C Routines to process TAQ data
             Based on taq.c/tick.c by Terence M. Lim (terence@alum.mit.edu),
             taq.c/tick.c created June 1996;
******************************/

#ifndef TAQAccess_TICK
#define TAQAccess_TICK

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef MSVC6
  #define LONGLONG __int64
#elif  C99
  #define LONGLONG long long
#else
  #warning LONGLONG undefined, use -DC99 or -DMSVC6
#endif

#define TAQ_ERROR      -1
#define TAQ_SUCCESS     0

#define DAWN            1       /* earliest time possible */
#define DUSK       235900       /* latest time possible */

/* Integer-based tick structures and routines
     --  useful for converting TAQ data to other formats */

/* structure to hold a single trade */
typedef struct tickctstruct * TAQ_TICK_CT;
struct     tickctstruct {
  long     tim;
  LONGLONG prc;
  long     siz, tseq;
  short    g127;
  short    corr;
  char     cond[4];
  char     ex;
};

/* structure to hold a quote */
typedef struct tickcqstruct * TAQ_TICK_CQ;
struct   tickcqstruct {
  long   tim;
  LONGLONG bid, ofr;
  long   qseq, bidsiz, ofrsiz;
  short  mode;
  char   ex, mmid[4];
};

/* structure to hold a day's data for one stock */
typedef struct tickstruct * TICK; 
struct tickstruct {
  char   symbol[7], cusip[8], name[30];
  long   date, uot, nt, nq;
  TAQ_TICK_CT ct; 
  TAQ_TICK_CQ cq;
};

/* related routines */
TICK  tick_new  (int nt,int nq);  /* allocates memory to store data */
void  tick_free (TICK t);         /* frees memory */
void  tick_sort (TICK t);
int   okspread  (LONGLONG bid, LONGLONG ofr);

/* Float-based structures and routines
     -- useful for manipulating TAQ data in C */

/* structure to hold a single trade */
typedef struct tickfctstruct * TAQ_TICK_F_CT;
struct   tickfctstruct {
  long   tim;
  double prc;
  long   siz, tseq;
  short  g127;
  short  corr;
  char   cond[4];
  char   ex;
};

/* structure to hold a quote */
typedef struct tickfcqstruct * TAQ_TICK_F_CQ;
struct   tickfcqstruct {
  long   tim;
  double bid, ofr;
  long   qseq, bidsiz, ofrsiz;
  short  mode;
  char   ex, mmid[4];
};

/* structure to hold a day's data for one stock */
typedef struct tickfstruct * TICK_F; 
struct tickfstruct {
  char   symbol[7], cusip[8], name[30];
  long   date, uot, nt, nq;
  TAQ_TICK_F_CT ct; 
  TAQ_TICK_F_CQ cq;
};

/* related routines */
TICK_F tick_f_new  (int nt, int nq);  /* allocates memory to store data */
void   tick_f_free (TICK_F t);        /* frees memory */
int    tick_f_new_from_int (TICK_F* t, TICK s);
void   tick_f_sort (TICK_F t);
int    okspread_f  (double bidf, double ofrf);


/* other utility routines */
long  clocktotime        (long t);
long  timetoclock        (long t);
char *todate             (int d, int t);
void  taqerr             (char *s);

void   swap     (char *to,char *from,int n);
void   strnul   (char *s, size_t n);

long     mklong   (char *s);
LONGLONG mkvlong (char *s);
short    mkshort  (char *s);
char*    mkstring (char *s,char *t,int n);
                        
#endif

