/******************************
   Program:  taq2csv.c
   author:   Boulat A. Bash
   email:    boulat@alum.dartmouth.org
   created:  August 2000
   comments: Program to convert TAQ data to comma-separated text (csv) files
******************************/

#define VERSION "1.1"
#define UPDATED "July 2004"
#define MAX_PATH 256

#include "TAQAccess_tick.h"
#include "TAQAccess_cd.h"
#include "TAQAccess_ascii.h"

void print_taq2csv_help ();

int main (int argc, char *argv[]) {

  FILE *dfp = NULL, *sfp = NULL, *tfp = NULL, *qfp = NULL;
  long date, i;
  char taqpath[MAX_PATH], datepath[MAX_PATH], outfile[MAX_PATH];
  char *datefile = NULL, *soutfile = NULL, *toutfile = NULL, *qoutfile = NULL;
  char id[9];
  int dbt, dbq, p, ps, pt, pq, print_head = 1;
  TICK t = NULL;
  SUM_PRINTER sp;
  TRADE_PRINTER tp;
  QUOTE_PRINTER qp;

  /* Set default values */
  strcpy (taqpath, "/home/boulat/TAQAccess/TAQ_199610A/");
  //DEB: strcpy (taqpath, "");
  strcpy (datepath, "");
  dbt = 0; dbq = 0;
  ps = 0; pt = 0; pq = 0;
  sp = sum_print_none();
  tp = trade_print_none();
  qp = quote_print_none();
  strcpy (outfile, "out");

  fprintf (stderr, "taq2csv version %s Copyright (c) %s Boulat A. Bash\n",
           VERSION, UPDATED);
  fprintf (stderr,
           "This  is  free software; see the source for copying ");
  fprintf (stderr,
           "conditions.\nThere is NO warranty; not even for MERCHANTABILITY ");
  fprintf (stderr, "or FITNESS FOR A\nPARTICULAR PURPOSE.\n\n");

  /* Process command-line arguments */
  argc--; argv++;
  if (0 == argc) fprintf (stderr, "Enter taq2csv -h for help\n\n");
  while ((argc > 0) && (**argv == '-')) {
    switch (*(*argv+1)) {

      case 'p':
        if (argc > 1 && *argv[1] != '-') {
          strncpy (taqpath, argv[1], MAX_PATH - 1);
          argc--; argv++;
        }
        break;

      case 'd':
        if (argc > 1 && *argv[1] != '-') {
          strncpy (datepath, argv[1], MAX_PATH - 1);
          argc--; argv++;
        }
        break;
      
      case 'n':
        if (argc > 1 && *argv[1] != '-') {
          strncpy (outfile, argv[1], MAX_PATH - 1);
          argc--; argv++;
        }
        break;

      case 'b':
        while (*argv[0] != '\0') {
          dbt = (*argv[0] == 't' || dbt) ? 1 : 0;
          dbq = (*argv[0] == 'q' || dbq) ? 1 : 0;
          *argv[0]++;
        }
        break;
      
      case 's':
        ps = 1;
        *argv[0]++; *argv[0]++;
        if      (*argv[0] == '-') { p = 0; sp = sum_print_all(); }
        else if (*argv[0] == '+')   p = 1;
             else {sp = sum_print_all(); break; }
        *argv[0]++;
        while (*argv[0] != '\0') {
          switch (*argv[0]) {
            case 's': sp.s      = p; break;
            case 'c': sp.c      = p; break;
            case 'n': sp.n      = p; break;
            case 'd': sp.d      = p; break;
            case 'u': sp.u      = p; break;
            case 't': sp.nt     = p; break;
            case 'q': sp.nq     = p; break;
            default : 
              fprintf (stderr, "Invalid argument: enter taq2csv -h for help\n");
              return (TAQ_ERROR);
          }
          *argv[0]++;
        }
        break;
      
      case 't':
        pt = 1;
        *argv[0]++; *argv[0]++;
        if      (*argv[0] == '-') { p = 0; tp = trade_print_all(); }
        else if (*argv[0] == '+')   p = 1;
             else {tp = trade_print_all(); break; }
        *argv[0]++;
        while (*argv[0] != '\0') {
          switch (*argv[0]) {
            case 's': tp.s      = p; break;
            case 'c': tp.c      = p; break;
            case 'd': tp.d      = p; break;
            case 'x': tp.ex     = p; break;
            case 't': tp.tim    = p; break;
            case 'p': tp.prc    = p; break;
            case 'z': tp.siz    = p; break;
            case 'o': tp.cond   = p; break;
            case 'r': tp.corr   = p; break;
            case 'q': tp.tseq   = p; break;
            case 'g': tp.g127   = p; break;
            default : 
              fprintf (stderr, "Invalid argument: enter taq2csv -h for help\n");
              return (TAQ_ERROR);
          }
          *argv[0]++;
        }
        break;
      
      case 'q':
        pq = 1;
        *argv[0]++; *argv[0]++;
        if      (*argv[0] == '-') { p = 0; qp = quote_print_all(); }
        else if (*argv[0] == '+')   p = 1;
             else {qp = quote_print_all(); break; }
        *argv[0]++;
        while (*argv[0] != '\0') {
          switch (*argv[0]) {
            case 's': qp.s      = p; break;
            case 'c': qp.c      = p; break;
            case 'd': qp.d      = p; break;
            case 'x': qp.ex     = p; break;
            case 't': qp.tim    = p; break;
            case 'b': qp.bid    = p; break;
            case 'i': qp.bidsiz = p; break;
            case 'o': qp.ofr    = p; break;
            case 'f': qp.ofrsiz = p; break;
            case 'e': qp.mode   = p; break;
            case 'q': qp.qseq   = p; break;
            case 'm': qp.mmid   = p; break;
            default : 
              fprintf (stderr, "Invalid argument: enter taq2csv -h for help\n");
              return (TAQ_ERROR);
          }
          *argv[0]++;
        }
        break;

      case 'h':
        print_taq2csv_help ();
        return (TAQ_SUCCESS);
    }
    argc--; argv++;
  }
  
  /* open date2.dat */
  datefile = (char *) malloc (strlen (datepath) + strlen ("date2.dat") + 2);
  if (NULL == datefile) {
    fprintf (stderr, "malloc error: datefile\n");
    return (TAQ_ERROR);
  }
  sprintf (datefile, "%s%s", datepath, "date2.dat");
  dfp = fopen (datefile, "r");
  if (NULL == dfp) {
    fprintf (stderr, "file open error: %s\n", datefile);
    return (TAQ_ERROR);
  }
  
  while (2 == fscanf (stdin, "%8s %ld", id, &date)) {
    fprintf (stderr, "Processing id=%s date=%ld...  ", id, date);
    /* read the transaction data */
    if (read_stockday (dfp, taqpath, id, date, DAWN, DUSK, dbt, dbq, 
                       0, (void *) &t)) {
      tick_sort (t); /* sort trades and quotes by time and sequence numbers */
      
      /* open output files on the first iteration of the loop */
      if (ps && NULL == sfp) {
        soutfile = (char *) malloc (strlen (outfile) + strlen (".s.csv") + 2);
        if (NULL == soutfile) {
          fprintf (stderr, "malloc error: soutfile\n"); return (TAQ_ERROR);
        }
        sprintf (soutfile, "%s.s.csv", outfile);
        sfp = fopen (soutfile, "w");
        if (NULL == sfp) {
          fprintf (stderr, "file open error: %s\n", soutfile);
          return (TAQ_ERROR);
        }
      }
      if (pt && NULL == tfp) {
        toutfile = (char *) malloc (strlen (outfile) + strlen (".t.csv") + 2);
        if (NULL == toutfile) {
          fprintf (stderr, "malloc error: toutfile\n"); return (TAQ_ERROR);
        }
        sprintf (toutfile, "%s.t.csv", outfile);
        tfp = fopen (toutfile, "w");
        if (NULL == tfp) {
          fprintf (stderr, "file open error: %s\n", toutfile);
          return (TAQ_ERROR);
        }
      }
      if (pq && NULL == qfp) {
        qoutfile = (char *) malloc (strlen (outfile) + strlen (".q.csv") + 2);
        if (NULL == qoutfile) {
          fprintf (stderr, "malloc error: qoutfile\n"); return (TAQ_ERROR);
        }
        sprintf (qoutfile, "%s.q.csv", outfile);
        qfp = fopen (qoutfile, "w");
        if (NULL == qfp) {
          fprintf (stderr, "file open error: %s\n", qoutfile);
          return (TAQ_ERROR);
        }
      }

      /* write headers (only on first iteration), then write data */
      if (ps) {
        if (print_head) s_header_write_ascii (sfp, 0, ',', '\n', sp);
        tick_sum_write_ascii (sfp, t, 0, ',', '\n', sp);
      }
      if (pt) {
        if (print_head) t_header_write_ascii (tfp, 20000101, 0, ',', '\n', tp);
        for (i = 0; i < t->nt; i++)
          trade_write_ascii (tfp, t, i, 0, ',', '\n', tp);
      }
      if (pq) {
        if (print_head) q_header_write_ascii (qfp, 20000101, 0, ',', '\n', qp);
        for (i = 0; i < t->nq; i++)
          quote_write_ascii (qfp, t, i, 0, ',', '\n', qp);
      }
      print_head = 0;
      tick_free (t);
      t = NULL;
      fprintf (stderr, "ok.\n");
    }
    else fprintf (stderr, "FAILURE!\n");
  }

  /* clean up */
  if (ps && (NULL != sfp)) fclose (sfp);
  if (pt && (NULL != tfp)) fclose (tfp);
  if (pq && (NULL != qfp)) fclose (qfp);
  fclose (dfp);
  
  return (0);
}


void print_taq2csv_help () {

  fprintf (stdout, "NAME\n\ttaq2csv - convert TAQ data to ");
  fprintf (stdout, "comma-separated text (csv) files\n\n");
  
  fprintf (stdout, "SYNOPSIS\n\ttaq2csv [options]\n\n");
  
  fprintf (stdout, "DESCRIPTION\n");
  fprintf (stdout, "\ttaq2csv converts the specified TAQ trading data for ");
  fprintf (stdout, "stocks on\n\tcorresponding dates to comma-separated ");
  fprintf (stdout, "text (csv) format.  It\n\treads the list of ");
  fprintf (stdout, "stock symbols/cusips and dates from standard input.  ");
  fprintf (stdout, "\n\tThe following is example input:\n");
  fprintf (stdout, "\t\tIBM      20020604\n");
  fprintf (stdout, "\t\t59491810 19961001\n");
  fprintf (stdout, "\tThe stock symbols should be at most 7 characters long; ");
  fprintf (stdout, "the cusips should\n\tbe *exactly* 8 characters long.");
  fprintf (stdout, "The date should be in YYYYMMDD format.\n");
  fprintf (stdout, "\tThe program writes the specified summary, trade, and ");
  fprintf (stdout, "quote records\n\tinto OUTPUTFILE.s.csv, OUTPUTFILE.t.csv,");
  fprintf (stdout, "and OUTPUTFILE.q.csv\n\trespectively.\n\n");

  fprintf (stdout, "COMMAND-LINE OPTIONS\n\n");
  fprintf (stdout, "\t-p TAQPATH\n");
  fprintf (stdout, "\t\tSet the location of TAQ data to TAQPATH.\n");
  fprintf (stdout, "\t\tDefault is present working directory.\n\n");
  fprintf (stdout, "\t-d DATEPATH\n");
  fprintf (stdout, "\t\tSet the location of date2.dat file to DATEPATH.\n");
  fprintf (stdout, "\t\tDefault is present working directory.\n\n");
  fprintf (stdout, "\t-n OUTPUTFILE\n");
  fprintf (stdout, "\t\tSet the prefix of the output files.  ");
  fprintf (stdout, "Default is `out'.\n");
  
  fprintf (stdout, "\t-bt\n");
  fprintf (stdout, "\t\tDo not write bad trades.\n");
  fprintf (stdout, "\t-bq\n");
  fprintf (stdout, "\t\tDo not write bad quotes.\n");
  fprintf (stdout, "\t-btq, -bqt\n");
  fprintf (stdout, "\t\tDo not write both bad trades and bad quotes.\n");

  fprintf (stdout, "\t-s[+-][scndutq]\n");
  fprintf (stdout, "\t\tControl which summary records to write for ");
  fprintf (stdout, "specified stock and\n\t\ttrading date.  Default ");
  fprintf (stdout, "is not to write any.\n");
  fprintf (stdout, "\t\tOption -s simply writes all summary records.\n");
  fprintf (stdout, "\t\tOperator `-' causes records specified by ");
  fprintf (stdout, "sequent letters to be\n\t\tremoved from full set ");
  fprintf (stdout, "of summary records.  Remaining ones get\n\t\twritten.\n");
  fprintf (stdout, "\t\tOperator `+' causes only records specified by ");
  fprintf (stdout, "sequent letters\n\t\tto be written.\n");
  fprintf (stdout, "\t\tFinally, letters 'scndutq' select summary records:\n");
  fprintf (stdout, "\t\tticker symbol (s), cusip (c), ");
  fprintf (stdout, "company name (n),\n\t\ttrading date (d), round lot size ");
  fprintf (stdout, "a.k.a. unit-of-trade (u),\n\t\tnumber of trades (t), ");
  fprintf (stdout, "and number of quotes (q).\n");
   
  fprintf (stdout, "\t-t[+-][scdxtpzorqg]\n");
  fprintf (stdout, "\t\tControl which trade records to write for ");
  fprintf (stdout, "specified stock and\n\t\ttrading date.  Default ");
  fprintf (stdout, "is not to write any.\n");
  fprintf (stdout, "\t\tOption -t simply writes all trade records.\n");
  fprintf (stdout, "\t\tOperator `-' causes records specified by ");
  fprintf (stdout, "sequent letters to be\n\t\tremoved from full set ");
  fprintf (stdout, "of trade records.  Remaining ones get\n\t\twritten.\n");
  fprintf (stdout, "\t\tOperator `+' causes only records specified by ");
  fprintf (stdout, "sequent letters\n\t\tto be written.\n");
  fprintf (stdout, "\t\tFinally, letters 'scdxtpzorqg' select trade ");
  fprintf (stdout, "records: ticker\n\t\tsymbol (s), cusip (c), ");
  fprintf (stdout, "trading date (d), exchange on which\n\t\tthe trade occurred ");
  fprintf (stdout, "(x), time in seconds since midnight (t),\n\t\t");
  fprintf (stdout, "price (p), number of shares traded (z), sale condition ");
  fprintf (stdout, "(o),\n\t\tcorrection indicator (r), MDS sequence ");
  fprintf (stdout, "number (q),\n\t\tand combined \"G\", rule 127, and ");
  fprintf (stdout, "stopped trade indicator (g).\n");
   
  fprintf (stdout, "\t-q[+-][scdxtbiofeqm]\n");
  fprintf (stdout, "\t\tControl which quote records to write for ");
  fprintf (stdout, "specified stock and\n\t\ttrading date.  Default ");
  fprintf (stdout, "is not to write any.\n");
  fprintf (stdout, "\t\tOption -q simply writes all quote records.\n");
  fprintf (stdout, "\t\tOperator `-' causes records specified by ");
  fprintf (stdout, "sequent letters to be\n\t\tremoved from full set ");
  fprintf (stdout, "of quote records.  Remaining ones get\n\t\twritten.\n");
  fprintf (stdout, "\t\tOperator `+' causes only records specified by ");
  fprintf (stdout, "sequent letters\n\t\tto be written.\n");
  fprintf (stdout, "\t\tFinally, letters 'scdxtpzorqg' select quote ");
  fprintf (stdout, "records: ticker\n\t\tsymbol (s), cusip (c), ");
  fprintf (stdout, "trading date (d), exchange on which\n\t\tthe quote ");
  fprintf (stdout, "occurred (x), time in seconds since midnight (t),\n\t\t");
  fprintf (stdout, "bid (b), bid size in number of round lots (i), offer (o),");
  fprintf (stdout, "\n\t\toffer size in number of round lots (f), quote ");
  fprintf (stdout, "condition (e),\n\t\tMDS sequence number (q), and ");
  fprintf (stdout, "market-maker identifier (m).\n");

  fprintf (stdout, "EXAMPLES\n");
  fprintf (stdout, "\tTo write all of trade and quote data to myout.t.csv ");
  fprintf (stdout, "and myout.q.csv\n\twith date2.dat located in setup/ and");
  fprintf (stdout, "TAQ data files located in data/:\n");
  fprintf (stdout, "\t\ttaq2csv -n myout -d setup/ -p data/ -t -q\n\n");
  fprintf (stdout, "\tTo write all of summary data in addition to ticker ");
  fprintf (stdout, "symbol, date,\n\tand trade time, price and size:\n");
  fprintf (stdout, "\t\ttaq2csv -s -t+sdtpz\n\n");
  fprintf (stdout, "\tTo do the equivalent to the previous task, excluding ");
  fprintf (stdout, "bad trades:\n");
  fprintf (stdout, "\t\ttaq2csv -s -t-cxorqg -bt\n");
}

