/******************************
   Library:  TAQAccess_tick.c
   author:   Boulat A. Bash
   email:    boulat@alum.dartmouth.org
   created:  July   2000
   updated:  February 2004
   comments: C routines to process TAQ data
             Based on taq.c/tick.c by Terence M. Lim (terence@alum.mit.edu),
             taq.c/tick.c created June 1996;
             Particular thanks to Bob Burnham (robert.burnham@dartmouth.edu)
               for work on migration to TAQ2 (TAQ format after 01-01-2000).
******************************/

#include "TAQAccess_tick.h"


/* ------------------------------------------------
   Routines related to integer-based tick structure
   -----------------------------------------------*/

/*
/// ----------------------
/// TICK tick_new (nt, nq)
/// ======================
///    input parameters
///       int nt : maximum number of trades
///       int nq : maximum number of quotes
///    description
///       allocates and returns pointer to a new transactions data structure
/// ___________________
*/
TICK tick_new (int nt, int nq)
{
  TICK t;
  
  t = (TICK) malloc (sizeof (*t));
  if (NULL == t) 
    return (NULL);
    
  memset (t->symbol, '\0', 7);
  memset (t->cusip,  '\0', 8);
  memset (t->name,   '\0', 30);
  t->uot = 100;
  t->nt = t->nq = 0;
  
  t->ct = (TAQ_TICK_CT) malloc (sizeof (*t->ct) * nt);
  if (NULL == t->ct) {
    free (t);
    return (NULL);
  }
  t->cq = (TAQ_TICK_CQ) malloc (sizeof (*t->cq) * nq);
  if (NULL == t->cq) {
    free (t);
    free (t->ct);
    return (NULL);
  }
  return (t);
}

/*
/// ------------------
/// void tick_free (t)
/// ==================
///    input parameters
///       TICK t : transactions data structure
///    description
///       frees memory previously allocated to this structure
/// ___________________
*/
void tick_free (TICK t)
{
  free (t->ct);
  free (t->cq);
  free (t);
}

/*
/// ------------------
/// void tick_sort (t)
/// ==================
///    input parameters
///       TICK *t : transactions data structure
///    description
///       sorts trades and quotes by time-stamp (if timestamps are equal, uses
///       trade/quote sequence numbers (tseq and qseq) to sort; static 
///       functions cmptickct and cmptickcq are used by quicksort in tick_sort.
/// _________________
*/
static int cmptickct (const void *a, const void *b) {
  if ((((TAQ_TICK_CT) a)->tim - ((TAQ_TICK_CT) b)->tim) == 0)
    return (((TAQ_TICK_CT) a)->tseq - ((TAQ_TICK_CT) b)->tseq);
  else return (((TAQ_TICK_CT) a)->tim - ((TAQ_TICK_CT) b)->tim);
}
static int cmptickcq (const void *a, const void *b) {
  if ((((TAQ_TICK_CQ) a)->tim - ((TAQ_TICK_CQ) b)->tim) == 0)
    return (((TAQ_TICK_CQ) a)->qseq - ((TAQ_TICK_CQ) b)->qseq);
  else return (((TAQ_TICK_CQ) a)->tim - ((TAQ_TICK_CQ) b)->tim);
}
void tick_sort (TICK t) {
  qsort (t->ct, t->nt, sizeof (*t->ct), cmptickct);
  qsort (t->cq, t->nq, sizeof (*t->cq), cmptickcq);
}

/*
/// -----------------------------
/// int okspread (bid, ofr, date)
/// =============================
///    input parameters
///       LONGLONG bid : bid price
///       LONGLONG ofr : offer price
///    description
///       checks that bid and offer prices are valid:
///       (1) positive bid and offer
///       (2) offer greater than bid
///       (3) if midquote less than ten dollars, spread no more than $2
///       (4) if midquote greater than ten dollars, spread no more than 20%
/// ______________________
*/
int okspread (LONGLONG bid, LONGLONG ofr) {

  double bidf, ofrf, midq;

  bidf = ((double) bid) / 100000000.0;
  ofrf = ((double) ofr) / 100000000.0;
    
  midq = (bidf + ofrf) / 2.0;
  if (bidf <= 0.0 || ofrf<=0.0 || bidf >= ofrf) return (0);
  if (midq <= 10.0 && (ofrf - bidf) > 2.0) return (0);
  if (midq > 10.0 && (ofrf - bidf) > (0.2 * midq)) return (0);
  return (1);
}


/* ----------------------------------------------
   Routines related to float-based tick structure
   ---------------------------------------------*/

/*
/// --------------------------
/// TICK_F tick_f_new (nt, nq)
/// ==========================
///    input parameters
///       int nt : maximum number of trades
///       int nq : maximum number of quotes
///    description
///       allocates and returns pointer to a new transactions data structure
/// ___________________
*/
TICK_F tick_f_new (int nt, int nq)
{
  TICK_F t;
  
  t = (TICK_F) malloc (sizeof(*t));
  if (NULL == t) 
    return (NULL);
    
  memset (t->symbol, '\0', 7);
  memset (t->cusip,  '\0', 8);
  memset (t->name,   '\0', 30);
  t->uot = 100;
  t->nt = t->nq = 0;
  
  t->ct = (TAQ_TICK_F_CT) malloc (sizeof (*t->ct) * nt);
  if (NULL == t->ct) {
    free (t);
    return (NULL);
  }
  t->cq = (TAQ_TICK_F_CQ) malloc (sizeof (*t->cq) * nq);
  if (NULL == t->cq) {
    free (t);
    free (t->ct);
    return (NULL);
  }
  return (t);
}

/*
/// --------------------
/// void tick_f_free (t)
/// ====================
///    input parameters
///       TICK_F t : transactions data structure
///    description
///       frees memory previously allocated to this structure
/// ___________________
*/
void tick_f_free (TICK_F t) {
  free (t->ct);
  free (t->cq);
  free (t);
}

/*
/// -------------------------------
/// void tick_f_new_from_int (t, s)
/// ===============================
///    input parameters
///       TICK_F t : transactions data structure
///    description
///       allocates and returns pointer to a new transactions data structure
///       containing the given integer-based structure converted to float-based
/// _______________________________
*/
int tick_f_new_from_int (TICK_F* t, TICK s) {
  
  long i;

  if (*t != NULL) {
    
    tick_f_free (*t);
    *t = NULL;
  }
  
  *t = tick_f_new (s->nt, s->nq);
  if (NULL == *t) 
    return (TAQ_ERROR);

  memcpy ((*t)->symbol, s->symbol, 7);
  memcpy ((*t)->cusip,  s->cusip,  8);
  memcpy ((*t)->name,   s->name,   30);
  (*t)->date = s->date;
  (*t)->uot  = s->uot;
  (*t)->nt   = s->nt;
  (*t)->nq   = s->nq;

  for (i = 0; i < (*t)->nt; i++) {

    (*t)->ct[i].tim = s->ct[i].tim;
    (*t)->ct[i].prc = (double) s->ct[i].prc / 100000000.0;
    (*t)->ct[i].siz  = s->ct[i].siz;
    (*t)->ct[i].tseq = s->ct[i].tseq;
    (*t)->ct[i].g127 = s->ct[i].g127;
    (*t)->ct[i].corr = s->ct[i].corr;
    memcpy ((*t)->ct[i].cond, s->ct[i].cond, 4);
    (*t)->ct[i].ex   = s->ct[i].ex;
  }
  for (i = 0; i < (*t)->nq; i++) {

    (*t)->cq[i].tim = s->cq[i].tim;
    (*t)->cq[i].bid = (double) s->cq[i].bid / 100000000.0;
    (*t)->cq[i].ofr = (double) s->cq[i].ofr / 100000000.0;
    (*t)->cq[i].qseq   = s->cq[i].qseq;
    (*t)->cq[i].bidsiz = s->cq[i].bidsiz;
    (*t)->cq[i].ofrsiz = s->cq[i].ofrsiz;
    (*t)->cq[i].mode   = s->cq[i].mode;
    (*t)->cq[i].ex     = s->cq[i].ex;
    memcpy((*t)->cq[i].mmid, s->cq[i].mmid, 4);
  }
  return (TAQ_SUCCESS);
}

/*
/// --------------------
/// void tick_f_sort (t)
/// ====================
///    input parameters
///       TICK_F *t : transactions data structure
///    description
///       sorts trades and quotes by time-stamp (if timestamps are equal, uses
///       trade/quote sequence numbers (tseq and qseq) to sort; static 
///       functions cmptickct and cmptickcq are used by quicksort in tick_sort.
/// _________________
*/
static int cmptickfct (const void *a, const void *b) {
  if ((((TAQ_TICK_F_CT) a)->tim - ((TAQ_TICK_F_CT) b)->tim) == 0)
    return (((TAQ_TICK_F_CT) a)->tseq - ((TAQ_TICK_F_CT) b)->tseq);
  else return (((TAQ_TICK_F_CT) a)->tim - ((TAQ_TICK_F_CT) b)->tim);
}
static int cmptickfcq (const void *a, const void *b) {
  if ((((TAQ_TICK_F_CQ) a)->tim - ((TAQ_TICK_F_CQ) b)->tim) == 0)
    return (((TAQ_TICK_F_CQ) a)->qseq - ((TAQ_TICK_F_CQ) b)->qseq);
  else return (((TAQ_TICK_F_CQ) a)->tim - ((TAQ_TICK_F_CQ) b)->tim);
}
void tick_f_sort (TICK_F t) {
  qsort (t->ct, t->nt, sizeof (*t->ct), cmptickfct);
  qsort (t->cq, t->nq, sizeof (*t->cq), cmptickfcq);
}

/*
/// -------------------------
/// int okspread_f (bid, ofr)
/// =========================
///    input parameters
///       double bid : bid price
///       double ofr : offer price
///    description
///       checks that bid and offer prices are valid:
///       (1) positive bid and offer
///       (2) offer greater than bid
///       (3) if midquote less than ten dollars, spread no more than $2
///       (4) if midquote greater than ten dollars, spread no more than 20%
/// ______________________
*/
int okspread_f (double bidf, double ofrf) {

  double midq;
  midq = (bidf + ofrf) / 2.0;
  if (bidf <= 0.0 || ofrf<=0.0 || bidf >= ofrf) return (0);
  if (midq <= 10.0 && (ofrf - bidf) > 2.0) return (0);
  if (midq > 10.0 && (ofrf - bidf) > (0.2 * midq)) return (0);
  return (1);
}


/* -----------------------------------------
   Other utility routines
   ----------------------------------------*/

/*
/// --------------------
/// long clocktotime (t)
/// ====================
///    input parameters
///       long t : clock time (hhmmss)
///    description
///       converts and returns as seconds after midnight
/// ___________________
*/
long clocktotime (long t) {
  return ((t % 100L) + ((t / 100L) % 100L) * 60L + (t / 10000L) * 3600L);
}

/*
/// --------------------
/// long timetoclock (t)
/// ====================
///    input parameters
///       long t : time, in seconds after midnight
///    description
///       converts and returns clock time (hhmmss)
/// ___________________
*/
long timetoclock (long t) 
{
  return ((t % 60L) + ((t / 60L) % 60L) * 100L + (t / 3600L) * 10000L);
}

/*
/// -------------------
/// char *todate (d, t)
/// ===================
///    input parameters
///       int d : date (yyyymmdd or yymmdd)
///       int t : time, in seconds after midnight
///    description
///       returns pointer to static character string representing date and time
/// ___________________
*/
char *todate (int d, int t) {
  static char tmp[32];
  static char *mmm[] = {
    "Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
  int yy, mm, dd, h, m, s;
  yy = (d / 10000) % 100;
  mm = (d / 100) % 100;
  dd = d % 100;
  s = t % 60;
  m = (t / 60) % 60;
  h = t / 3600;
  sprintf (tmp, "%2d %3s %4d %02d:%02d:%02d", dd, mmm[mm-1],
           yy >= 93 && yy <= 99 ? 1900 + yy : 2000 + yy, h, m, s);
  return (tmp);
}

/*
/// ---------------
/// void taqerr (s)
/// ===============
///    input parameters
///       char *s : error message
///    description
///       prints error message and exits
/// ___________________
*/
void taqerr (char *s) {
  fprintf (stderr, "\n%s\nExiting...\n", s);
  exit (0);
}



/* variables used to detect byte order representation on local machine */
static long checklong=1L;
static char *noswap=(char *) &checklong;


/*
/// -----------------------
/// void swap (to, from, n)
/// =======================
///    input parameters
///      char *from : pointer to a Intel byte-order structure
///      int      n : number of bytes in the structure
///    output 
///      char   *to : pointer to a structure with byte order approprite to
///                   to host machine
///    description
///      Intel and DEC Alpha use little-endian, while PowerPC, Sun, and
///      SGI use big-endian byte ordering scheme.  TAQ CDs are produced for
///      DOS/Win32-based (Intel) machines, thus we may need to reverse bytes.
///      Static variable noswap indicates whether such reversal is needed;
///      The procedure is also used to always write XDF binaries in
///      little-endian byte order.
/// ___________________
*/
void swap (char *to, char *from, int n) {
  int i;
  for (i = 0; i < n; i++) {
    if (*noswap) to[i] = from[i];
    else         to[i] = from[n-i-1];
  }
}


/*
/// ------------------
/// void strnul (s, n)
/// ==================
///    input parameters
///      char *t : pointer to a string
///      int   n : number of bytes in the string
///    output 
///      fills the given string in with nulls from first non-alphanumeric, non-.
///      char to the last byte;
/// ___________________
*/
void strnul (char* s, size_t n) {

  int i, eflag = 0;
  for (i = 0; i < n; i++) {
    if (!(isalnum (s[i]) || s[i] == '.') || eflag) {
      eflag = 1;
      s[i] = '\0';
    }
  }
}


/*
/// -------------------
/// __int64   mkvlong (s) --> in MS-VC 6
/// long long mkvlong (s) --> in C99 
/// ===================
///    input parameters
///      char *t  : pointer to an 8-byte unsigned long integer
///    output 
///      returns very long (64 bit) integer;
///      NOTE:  This reads the exo-standard format for prices
///             in TAQ2; the output is a very long integer with 8 implied
///             decimal spaces -- thus the number must be divided
///             by 10^8=100,000,000.0 to obtain the correct price.
///      NOTE2: We cast our unsigned very long integer into signed
///             into signed very long integer with expectation that
///             price of a single share of stock shall never exceed 
///             92233720368.54775807 USD.
/// ___________________
*/
LONGLONG mkvlong (char *s)
{
  unsigned LONGLONG x;
  swap   ((char *) &x, s, 8);
  return ((LONGLONG) x);
}


/*
/// -------------------
/// long mklong (s)
/// ===================
///    input parameters
///      char *s  : pointer to a long
///    output 
///      returns the given long with reversed byte ordering if necessary;
/// ___________________
*/
long mklong(char *s)
{
  long x;
  swap((char *) &x,s,4);
  return(x);
}


/*
/// -------------------
/// short mkshort (s)
/// ===================
///    input parameters
///      char *s  : pointer to a short
///    output 
///      returns the given short with reversed byte ordering if necessary;
/// ___________________
*/
short mkshort(char *s)
{
  short x;
  swap((char *) &x,s,2);
  return(x);
}

/*
/// -----------------------
/// char mkstring (s, t, n)
/// =======================
///    input parameters
///      char *t  : pointer to a string (source)
///      int   n : number of bytes in the structure
///    output
///      char *s  : pointer to a string (destination)
///    return
///      pointer to destination string
///    description
///      copies string t into string s. Destination string s is also filled-in
///      with nulls from the end until the last alphanumeric character;
///      WARNING: make sure that the c-string terminator '\0' is at the end
///               of s if s is to be printed or used in atoi/atof/etc.
///               this procedure does NOT take care of this. Use with care.
/// ___________________
*/
char *mkstring(char *s,char *t,int n)
{
  int eflag = 0;
  for(s[--n] = '\0'; n >= 0; n--) {
    if (isalnum(t[n]) || t[n] == '.') {
      s[n]  = t[n];
      eflag = 1;
    }
    else {
      if (eflag)  s[n] = t[n];
      else        s[n] = '\0';
    }
  }
  return(s);
}


