/******************************
   Library:  TAQAccess_cd.h
   author:   Boulat A. Bash
   email:    boulat@alum.dartmouth.org
   created:  July   2000
   updated:  February 2004
   comments: Header file for C Routines to process TAQ CD
             Based on taq.c/tick.c by Terence M. Lim (terence@alum.mit.edu),
             taq.c/tick.c created June 1996;
             Particular thanks to Bob Burnham (robert.burnham@dartmouth.edu)
               for work on migration to TAQ2 (TAQ format after 01-01-2000).
******************************/

#ifndef TAQAccess_CD
#define TAQAccess_CD

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include "TAQAccess_tick.h"


/*-----------------------------------------
  Reading TAQ CD
------------------------------------------*/

typedef struct taqcdstruct {
  char  path[256];           /* CD location */
  int   month;
  char  disk;                /* TAQ disk designation: a, b, c, ... */
  
  int   mastlen;
  int   divlen;
  int   idxlen;
  int   ctbinlen;
  int   cqbinlen;

  FILE *ctbinfp;             /* File pointers */
  FILE *ctidxfp;
  FILE *cqbinfp;
  FILE *cqidxfp;
  FILE *mastfp;
  FILE *divfp;
} TAQCD;


/* 19971028 NASD timestamp kludge */
typedef struct taqindexstruct {
  char key[16];
  int loc;
} TAQINDEX;

/* structure of TAQ CD data files, as defined in TAQ manual */
typedef struct ctbinbuf {
  long  ttim;
  LONGLONG price;
  long siz, tseq;
  short g127, corr;
  unsigned char cond[4], ex;
} CTBIN;

typedef struct cqbinbuf {
  long qtim;
  LONGLONG bid, ofr;
  long qseq, bidsiz, ofrsiz;
  short mode;
  unsigned char ex, mmid[4];
} CQBIN;

typedef struct idxbuf {
  char symbol[7];
  long tdate, brec, erec;
} IDX;

typedef struct mastbuf {
  char symbol[7], name[30], cusip[12], etx[10], its, icode[4];
  long sharesout;
  char uot[4], denom, type;
  long datef;
} MASTTAB;

typedef struct divbuf {
  char symbol[7], cusip[12];
  double div, split;
  long datef;
} DIVTAB;


/* routines to access TAQ CD data */
int  taq_cd_init    (char *p,int m, char d, TAQCD *cd);
void taq_cd_cleanup (TAQCD *cd);
int  read_stockday  (FILE *dfp, char *p,
                     char *ident, int date, int begtime, int endtime,
                     int dbt, int dbq,
                     int isfloat, void *t);
int  tick_read_cd   (char *symbol, char *cusip, IDX ctidx, IDX cqidx, TAQCD cd,
                     int begtime, int endtime, int ex, long date, long uot,
                     char *name, int dbt, int dbq, TICK *t);
int  tick_f_read_cd (char *symbol, char *cusip, IDX ctidx, IDX cqidx, TAQCD cd,
                     int begtime, int endtime, int ex, long date, long uot,
                     char *name, int dbt, int dbq, TICK_F *t);

/* Filter raw TAQ CD records */
int okctbin (CTBIN ctbin);
int okcqbin (CQBIN cqbin, long date);

/* routines to read raw records on TAQ CD */
int   getctbin     (long irec, TAQCD cd, CTBIN *ctbin);
int   getcqbin     (long irec, TAQCD cd, CQBIN *cqbin);
int   getcqidx     (long irec, TAQCD cd, IDX *cqidx);
int   getctidx     (long irec, TAQCD cd, IDX *ctidx);
int   getdivtab    (TAQCD cd, DIVTAB *d);
int   getmasttab   (TAQCD cd, MASTTAB *m);

/* 10-28-1997 NASD timestamp kludge */
typedef struct nasdindexstruct {
  char key[16];
  int loc;
} NASDINDEX;
int   getnasdctidx (char *nasd1028file, IDX *ctidx);
int   getnasdctbin (long irec, TAQCD cd, CTBIN *ctbin);
int   nasdcmp      (const void *a, const void *b);

/* Useful transformations of TAQ data */
long  mastuot      (char *uot);
char  mastetx      (char *etx);
long  mastdenom    (char  denom);
char* masttype     (char *type);

/* Utility routines */
int   idxcmp       (const void *a, const void *b);


#endif
