/******************************
   File:     TAQAccess_ascii.c
   author:   Boulat A. Bash
   email:    boulat@alum.dartmouth.org
   created:  July 2003
   updated:  July 2003
   comments: C Routines to output TICK data to text file
******************************/

#include "TAQAccess_ascii.h"

/*
/// --------------------------------------------------
/// int tick_sum_write_ascii (fp, t, i, spc, d, e, print)
/// ==================================================
///    input parameters
///       FILE *fp          : output data file
///       TICK  t           : tick data stored here (invalid if NULL)
///       long  i           : index of trade in t->ct to write
///       int   spc         : flag to print spaces in the fields
///       char  d           : delimiter
///       char  e           : line terminator
///       SUM_PRINTER print : data items to be printed
///    return
///       TAQ_ERROR or TAQ_SUCCESS 
///    description
///       writes summary to file stream in ASCII format
/// __________________________________________________
*/
int tick_sum_write_ascii (FILE *fp, TICK t, int spc, char d, char e,
                          SUM_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if (t == NULL) return (TAQ_ERROR);

  if (print.s)  print_symbol (fp, t->symbol, spc,
                              print.c || print.n  || print.d  ||
                              print.u || print.nt || print.nq ? delim : end);
  if (print.c)  print_cusip  (fp, t->cusip,  spc,
                              print.n  || print.d  || print.u || 
                              print.nt || print.nq ? delim : end);
  if (print.n)  print_name   (fp, t->name,   spc,
                              print.d  || print.u || 
                              print.nt || print.nq ? delim : end);
  if (print.d)  print_date   (fp, t->date,   spc,
                              print.u || print.nt || print.nq ? delim : end);
  if (print.u)  print_uot    (fp, t->uot,    spc,
                              print.nt || print.nq ? delim : end);
  if (print.nt) print_n      (fp, t->nt,     spc, print.nq ? delim : end);
  if (print.nq) print_n      (fp, t->nq,     spc, end);

  return (TAQ_SUCCESS);
}                        

/*
/// --------------------------------------------------
/// int trade_write_ascii (fp, t, i, spc, d, e, print)
/// ==================================================
///    input parameters
///       FILE *fp            : output data file
///       TICK  t             : tick data stored here (invalid if NULL)
///       long  i             : index of trade in t->ct to write
///       int   spc           : flag to print spaces in the fields
///       char  d             : delimiter
///       char  e             : line terminator
///       TRADE_PRINTER print : data items to be printed
///    return
///       TAQ_ERROR or TAQ_SUCCESS 
///    description
///       writes i'th trade to file stream in ASCII format
/// __________________________________________________
*/
int trade_write_ascii (FILE *fp, TICK t, long i, int spc, char d, char e, 
                       TRADE_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if ((t == NULL) || (i >= t->nt)) return (TAQ_ERROR);

  if (print.s)    print_symbol (fp, t->symbol,     spc, 
                                print.c    || print.d    || print.ex   ||
                                print.tim  || print.prc  || print.siz  ||
                                print.cond || print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.c)    print_cusip  (fp, t->cusip,      spc,
                                print.d    || print.ex   || print.tim  ||
                                print.prc  || print.siz  || print.cond || 
                                print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.d)    print_date   (fp, t->date,       spc,
                                print.ex   || print.tim  || print.prc  || 
                                print.siz  || print.cond || print.corr || 
                                print.tseq || print.g127 ? delim : end);
  if (print.ex)   print_ex     (fp, t->ct[i].ex,   spc,
                                print.tim  || print.prc  || print.siz  ||
                                print.cond || print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.tim)  print_tim    (fp, t->ct[i].tim,  spc,
                                print.prc  || print.siz  || print.cond ||
                                print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.prc)  print_prc    (fp, (double) t->ct[i].prc / 100000000.0, spc,
                                print.siz  || print.cond || print.corr || 
                                print.tseq || print.g127 ? delim : end);
  if (print.siz)  print_siz    (fp, t->ct[i].siz,  spc,
                                print.cond || print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.cond) print_cond   (fp, t->ct[i].cond, spc,
                                print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.corr) print_corr   (fp, t->ct[i].corr, spc, 
                                print.tseq || print.g127 ? delim : end);
  if (print.tseq) print_seq    (fp, t->ct[i].tseq, spc,
                                print.g127 ? delim : end);
  if (print.g127) print_g127   (fp, t->ct[i].g127, spc, end);
  
  return (TAQ_SUCCESS);
}


/*
/// --------------------------------------------------
/// int quote_write_ascii (fp, t, i, spc, d, e, print)
/// ==================================================
///    input parameters
///       FILE *fp            : output data file
///       TICK  t             : tick data stored here (invalid if NULL)
///       long  i             : index of quote in t->cq to write
///       int   spc           : flag to print spaces in the fields
///       char  d             : delimiter
///       char  e             : line terminator
///       QUOTE_PRINTER print : data items to be printed
///    return
///       TAQ_ERROR or TAQ_SUCCESS 
///    description
///       writes i'th quote to file stream in ASCII format
/// __________________________________________________
*/
int quote_write_ascii (FILE *fp, TICK t, long i, int spc, char d, char e,
                       QUOTE_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if ((t == NULL) || (i >= t->nq)) return (TAQ_ERROR);

  if (print.s)    print_symbol (fp, t->symbol,     spc, 
                                print.c      || print.d      || print.ex     || 
                                print.tim    || print.bid    || print.bidsiz ||
                                print.ofr    || print.ofrsiz || print.mode   ||
                                print.qseq   || print.mmid ? delim : end);
  if (print.c)    print_cusip  (fp, t->cusip,      spc,
                                print.d      || print.ex     || print.tim    ||
                                print.bid    || print.bidsiz || print.ofr    || 
                                print.ofrsiz || print.mode   || print.qseq   ||
                                print.mmid ? delim : end);
  if (print.d)    print_date   (fp, t->date,       spc,
                                print.ex     || print.tim    || print.bid    ||
                                print.bidsiz || print.ofr    || print.ofrsiz ||
                                print.mode   || print.qseq   ||  
                                print.mmid ? delim : end);
  if (print.ex)   print_ex     (fp, t->cq[i].ex,   spc,
                                print.tim    || print.bid    || print.bidsiz || 
                                print.ofr    || print.ofrsiz || print.mode   || 
                                print.qseq   || print.mmid ? delim : end);
  if (print.tim)  print_tim    (fp, t->cq[i].tim,  spc,
                                print.bid    || print.bidsiz || print.ofr    || 
                                print.ofrsiz || print.mode   || print.qseq   || 
                                print.mmid ? delim : end);
  if (print.bid)  print_prc    (fp, (double) t->cq[i].bid / 100000000.0, spc,
                                print.bidsiz || print.ofr    || print.ofrsiz ||
                                print.mode   || print.qseq   || 
                                print.mmid ? delim : end);
  if (print.bidsiz) print_qsiz (fp, t->cq[i].bidsiz, spc,
                                print.ofr    || print.ofrsiz || print.mode   || 
                                print.qseq   || print.mmid ? delim : end);
  if (print.ofr)    print_prc  (fp, (double) t->cq[i].ofr / 100000000.0, spc,
                                print.ofrsiz || print.mode   || print.qseq   || 
                                print.mmid ? delim : end);
  if (print.ofrsiz) print_qsiz (fp, t->cq[i].ofrsiz, spc,
                                print.mode   || print.qseq   ||
                                print.mmid ? delim : end);
  if (print.mode)   print_mode (fp, t->cq[i].mode, spc,
                                print.qseq   || print.mmid ? delim : end);
  if (print.qseq)   print_seq  (fp, t->cq[i].qseq, spc,
                                print.mmid ? delim : end);
  if (print.mmid)   print_mmid (fp, t->cq[i].mmid, spc, end);

  return (TAQ_SUCCESS);
}

/*
/// ----------------------------------------------------
/// int tick_f_sum_write_ascii (fp, t, i, spc, d, e, print)
/// ====================================================
///    input parameters
///       FILE  *fp          : output data file
///       TICK_F t           : tick data stored here (invalid if NULL)
///       long   i           : index of trade in t->ct to write
///       int    spc         : flag to print spaces in the fields
///       char   d           : delimiter
///       char   e           : line terminator
///       SUM_PRINTER print  : data items to be printed
///    return
///       TAQ_ERROR or TAQ_SUCCESS 
///    description
///       writes summary to file stream in ASCII format
/// __________________________________________________
*/
int tick_f_sum_write_ascii (FILE *fp, TICK_F t, int spc, char d, char e,
                            SUM_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if (t == NULL) return (TAQ_ERROR);

  if (print.s)  print_symbol (fp, t->symbol, spc,
                              print.c || print.n  || print.d  ||
                              print.u || print.nt || print.nq ? delim : end);
  if (print.c)  print_cusip  (fp, t->cusip,  spc,
                              print.n  || print.d  || print.u || 
                              print.nt || print.nq ? delim : end);
  if (print.n)  print_name   (fp, t->name,   spc,
                              print.d  || print.u || 
                              print.nt || print.nq ? delim : end);
  if (print.d)  print_date   (fp, t->date,   spc,
                              print.u || print.nt || print.nq ? delim : end);
  if (print.u)  print_uot    (fp, t->uot,    spc,
                              print.nt || print.nq ? delim : end);
  if (print.nt) print_n      (fp, t->nt,     spc, print.nq ? delim : end);
  if (print.nq) print_n      (fp, t->nq,     spc, end);

  return (TAQ_SUCCESS);
}                        

/*
/// ----------------------------------------------------
/// int trade_f_write_ascii (fp, t, i, spc, d, e, print)
/// ====================================================
///    input parameters
///       FILE  *fp            : output data file
///       TICK_F t             : tick data stored here (invalid if NULL)
///       long   i             : index of trade in t->ct to write
///       int    spc           : flag to print spaces in the fields
///       char   d             : delimiter
///       char   e             : line terminator
///       TRADE_PRINTER print  : data items to be printed
///    return
///       TAQ_ERROR or TAQ_SUCCESS 
///    description
///       writes i'th trade to file stream in ASCII format
/// ____________________________________________________
*/
int trade_f_write_ascii (FILE *fp, TICK_F t, long i, int spc, char d, char e, 
                         TRADE_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if ((t == NULL) || (i >= t->nt)) return (TAQ_ERROR);

  if (print.s)    print_symbol (fp, t->symbol,     spc, 
                                print.c    || print.d    || print.ex   ||
                                print.tim  || print.prc  || print.siz  ||
                                print.cond || print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.c)    print_cusip  (fp, t->cusip,      spc,
                                print.d    || print.ex   || print.tim  ||
                                print.prc  || print.siz  || print.cond || 
                                print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.d)    print_date   (fp, t->date,       spc,
                                print.ex   || print.tim  || print.prc  || 
                                print.siz  || print.cond || print.corr || 
                                print.tseq || print.g127 ? delim : end);
  if (print.ex)   print_ex     (fp, t->ct[i].ex,   spc,
                                print.tim  || print.prc  || print.siz  ||
                                print.cond || print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.tim)  print_tim    (fp, t->ct[i].tim,  spc,
                                print.prc  || print.siz  || print.cond ||
                                print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.prc)  print_prc    (fp, t->ct[i].prc, spc,
                                print.siz  || print.cond || print.corr || 
                                print.tseq || print.g127 ? delim : end);
  if (print.siz)  print_siz    (fp, t->ct[i].siz,  spc,
                                print.cond || print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.cond) print_cond   (fp, t->ct[i].cond, spc,
                                print.corr || print.tseq || 
                                print.g127 ? delim : end);
  if (print.corr) print_corr   (fp, t->ct[i].corr, spc, 
                                print.tseq || print.g127 ? delim : end);
  if (print.tseq) print_seq    (fp, t->ct[i].tseq, spc,
                                print.g127 ? delim : end);
  if (print.g127) print_g127   (fp, t->ct[i].g127, spc, end);
  
  return (TAQ_SUCCESS);
}


/*
/// ----------------------------------------------------
/// int quote_f_write_ascii (fp, t, i, spc, d, e, print)
/// ====================================================
///    input parameters
///       FILE  *fp            : output data file
///       TICK_F t             : tick data stored here (invalid if NULL)
///       long   i             : index of quote in t->cq to write
///       int    spc           : flag to print spaces in the fields
///       char   d             : delimiter
///       char   e             : line terminator
///       QUOTE_PRINTER print  : data items to be printed
///    return
///       TAQ_ERROR or TAQ_SUCCESS 
///    description
///       writes i'th quote to file stream in ASCII format
/// __________________________________________________
*/
int quote_f_write_ascii (FILE *fp, TICK_F t, long i, int spc, char d, char e,
                         QUOTE_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if ((t == NULL) || (i >= t->nq)) return (TAQ_ERROR);

  if (print.s)    print_symbol (fp, t->symbol,     spc, 
                                print.c    || print.d      || print.ex     || 
                                print.tim  || print.bid    || print.bidsiz ||
                                print.ofr  || print.ofrsiz || print.mode   ||
                                print.qseq || print.mmid ? delim : end);
  if (print.c)    print_cusip  (fp, t->cusip,      spc,
                                print.d      || print.ex     || print.tim  ||
                                print.bid    || print.bidsiz || print.ofr  || 
                                print.ofrsiz || print.mode   || print.qseq ||
                                print.mmid ? delim : end);
  if (print.d)    print_date   (fp, t->date,       spc,
                                print.ex     || print.tim  || print.bid    ||
                                print.bidsiz || print.ofr  || print.ofrsiz ||
                                print.mode   || print.qseq ||  
                                print.mmid ? delim : end);
  if (print.ex)   print_ex     (fp, t->cq[i].ex,   spc,
                                print.tim  || print.bid    || print.bidsiz || 
                                print.ofr  || print.ofrsiz || print.mode   || 
                                print.qseq || print.mmid ? delim : end);
  if (print.tim)  print_tim    (fp, t->cq[i].tim,  spc,
                                print.bid    || print.bidsiz || print.ofr  || 
                                print.ofrsiz || print.mode   || print.qseq || 
                                print.mmid ? delim : end);

  if (print.bid)  print_prc    (fp, t->cq[i].bid, spc,
                                print.bidsiz || print.ofr    || print.ofrsiz ||
                                print.mode   || print.qseq   || 
                                print.mmid ? delim : end);
  if (print.bidsiz) print_qsiz (fp, t->cq[i].bidsiz, spc,
                                print.ofr  || print.ofrsiz || print.mode || 
                                print.qseq || print.mmid ? delim : end);
  if (print.ofr)    print_prc  (fp, t->cq[i].ofr, spc,
                                print.ofrsiz || print.mode || print.qseq || 
                                print.mmid ? delim : end);
  if (print.bidsiz) print_qsiz (fp, t->cq[i].ofrsiz, spc,
                                print.mode || print.qseq ||
                                print.mmid ? delim : end);
  if (print.mode)   print_mode (fp, t->cq[i].mode, spc,
                                print.qseq || print.mmid ? delim : end);
  if (print.qseq)   print_seq  (fp, t->cq[i].qseq, spc,
                                print.mmid ? delim : end);
  if (print.mmid)   print_mmid (fp, t->cq[i].mmid, spc, end);

  return (TAQ_SUCCESS);
}


/*
/// -----------------------------------------------------
/// int s_header_write_ascii (fp, t, i, spc, d, e, print)
/// =====================================================
///    input parameters
///       FILE *fp          : output data file
///       TICK  t           : tick data stored here (invalid if NULL)
///       long  i           : index of trade in t->ct to write
///       int   spc         : flag to print spaces in the fields
///       char  d           : delimiter
///       char  e           : line terminator
///       SUM_PRINTER print : data items to be printed
///    description
///       writes summary header to file stream in ASCII format
/// _____________________________________________________
*/
void s_header_write_ascii (FILE *fp, int spc, char d, char e,
                          SUM_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if (print.s)  fprintf (fp, spc ? "%-7.7s%s"      : "%s%s", "@SYMBOL", 
                         print.c || print.n  || print.d  ||
                         print.u || print.nt || print.nq ? delim : end);
  if (print.c)  fprintf (fp, spc ? "%-8.8s%s"     : "%s%s", "CUSIP",
                         print.n  || print.d  || print.u || 
                         print.nt || print.nq ? delim : end);
  if (print.n)  fprintf (fp, spc ? "%-30.30s%s"     : "%s%s", "NAME",
                         print.d  || print.u || 
                         print.nt || print.nq ? delim : end);
  if (print.d)  fprintf (fp, spc ? "      %8.8s%s" : "%s%s", "DATE",
                         print.u || print.nt || print.nq ? delim : end);
  if (print.u)  fprintf (fp, spc ? "%6.6s%s" : "%s%s", "UOT",
                         print.nt || print.nq ? delim : end);
  if (print.nt) fprintf (fp, spc ? "%8.8s%s" : "%s%s", "NUMTRD",
                         print.nq ? delim : end);
  if (print.nq) fprintf (fp, spc ? "%8.8s%s" : "%s%s", "NUMQT", end);
}                        

/*
/// -------------------------------------------------
/// void t_header_write_ascii (fp, date, d, e, print)
/// =================================================
///    input parameters
///       FILE *fp            : output data file
///       long  date          : trading date
///       int   spc           : flag to print spaces in the fields
///       char  d             : delimiter
///       char  e             : line terminator
///       TRADE_PRINTER print : data items to be printed
///    description
///       writes trade header to file stream in ASCII format
/// _________________________________________________
*/
void t_header_write_ascii (FILE *fp, long date, int spc, char d, char e, 
                           TRADE_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if (print.s)    fprintf (fp, spc ? "%-7.7s%s"      : "%s%s", "@SYMBOL", 
                           print.c    || print.d    || print.ex   ||
                           print.tim  || print.prc  || print.siz  ||
                           print.cond || print.corr || print.tseq || 
                           print.g127 ? delim : end);
  if (print.c)    fprintf (fp, spc ? "%-8.8s%s"     : "%s%s", "CUSIP",
                           print.d    || print.ex   || print.tim  ||
                           print.prc  || print.siz  || print.cond || 
                           print.corr || print.tseq || 
                           print.g127 ? delim : end);
  if (print.d)    fprintf (fp, spc ? "      %8.8s%s" : "%s%s", "DATE",
                           print.ex   || print.tim  || print.prc  || 
                           print.siz  || print.cond || print.corr || 
                           print.tseq || print.g127 ? delim : end);
  if (print.ex)   fprintf (fp, spc ? "%2.2s%s"       : "%s%s", "EX" ,
                           print.tim  || print.prc  || print.siz  ||
                           print.cond || print.corr || print.tseq || 
                           print.g127 ? delim : end);
  if (print.tim)  fprintf (fp, spc ? "   %7.7s%s"    : "%s%s", "TIME",
                           print.prc  || print.siz  || print.cond ||
                           print.corr || print.tseq || 
                           print.g127 ? delim : end);
  if (print.prc)  fprintf (fp, spc ? "%15.15s%s"     : "%s%s", "PRICE",
                           print.siz  || print.cond || print.corr || 
                           print.tseq || print.g127 ? delim : end);
  if (print.siz)  fprintf (fp, spc ? "%10.10s%s"     : "%s%s", "SIZE", 
                           print.cond || print.corr || print.tseq || 
                           print.g127 ? delim : end);
  if (print.cond) {
  /* for consistency with TAQ2 v.1.42b */
    if (date < 20000101)
                  fprintf (fp, spc ? "%-4.4s%s"      : "%s%s", "C",
                           print.corr || print.tseq || 
                           print.g127 ? delim : end);
    else          fprintf (fp, spc ? "%-4.4s%s"      : "%s%s", "COND",
                           print.corr || print.tseq || 
                           print.g127 ? delim : end);
  }
  if (print.corr) fprintf (fp, spc ? "%2.2s%s"       : "%s%s", "CR",
                           print.tseq || print.g127 ? delim : end);
  if (print.tseq) fprintf (fp, spc ? "%9.9s%s"       : "%s%s", "TSEQ", 
                           print.g127 ? delim : end);
  if (print.g127) fprintf (fp, spc ? "%4.4s%s"       : "%s%s", "G127", end);
}


/*
/// -------------------------------------------------
/// void q_header_write_ascii (fp, date, spc, d, e, print)
/// =================================================
///    input parameters
///       FILE *fp            : output data file
///       long  date          : trading date
///       int   spc           : flag to print spaces in the fields
///       char  d             : delimiter
///       char  e             : line terminator
///       TRADE_PRINTER print : data items to be printed
///    description
///       writes quote header to file stream in ASCII format
/// _________________________________________________
*/
void q_header_write_ascii (FILE *fp, long date, int spc, char d, char e, 
                           QUOTE_PRINTER print) {
  char delim[3], end[2];
  if (d == '\0') sprintf (delim, " %c", d); /* keep the space */
  else           sprintf (delim, "%c ", d);
  if (!spc) delim[1] = '\0';
  sprintf (end  , "%c" , e);

  if (print.s)      fprintf (fp, spc ? "%-7.7s%s"      : "%s%s", "@SYMBOL",
                             print.c    || print.d      || print.ex     || 
                             print.tim  || print.bid    || print.bidsiz ||
                             print.ofr  || print.ofrsiz || print.mode   ||
                             print.qseq || print.mmid ? delim : end);
  if (print.c)      fprintf (fp, spc ? "%-8.8s%s"     : "%s%s", "CUSIP",
                             print.d      || print.ex     || print.tim  ||
                             print.bid    || print.bidsiz || print.ofr  || 
                             print.ofrsiz || print.mode   || print.qseq ||
                             print.mmid ? delim : end);
  if (print.d)      fprintf (fp, spc ? "      %8.8s%s" : "%s%s", "DATE",
                             print.ex     || print.tim  || print.bid    ||
                             print.bidsiz || print.ofr  || print.ofrsiz ||
                             print.mode   || print.qseq ||  
                             print.mmid ? delim : end);
  if (print.ex)     fprintf (fp, spc ? "%2.2s%s"       : "%s%s", "EX",
                             print.tim  || print.bid    || print.bidsiz || 
                             print.ofr  || print.ofrsiz || print.mode || 
                             print.qseq || print.mmid ? delim : end);
  if (print.tim)    fprintf (fp, spc ? "   %7.7s%s"    : "%s%s", "TIME", 
                             print.bid    || print.bidsiz || print.ofr  || 
                             print.ofrsiz || print.mode   || print.qseq || 
                             print.mmid ? delim : end);
  if (print.bid)    fprintf (fp, spc ? "%15.15s%s"     : "%s%s", "BID", 
                             print.bidsiz || print.ofr    || print.ofrsiz ||
                             print.mode   || print.qseq   || 
                             print.mmid ? delim : end);
  /* NOTE: in TAQ2 v.1.42b "SIZE" is printed instead of "BSIZE" */
  if (print.bidsiz) fprintf (fp, spc ? "%5.5s%s"       : "%s%s", "BSIZE",
                             print.ofr  || print.ofrsiz || print.mode || 
                             print.qseq || print.mmid ? delim : end);
  if (print.ofr)    fprintf (fp, spc ? "%15.15s%s"     : "%s%s", "OFFER",
                             print.ofrsiz || print.mode || print.qseq || 
                             print.mmid ? delim : end);
  /* NOTE: in TAQ2 v.1.42b "SIZE" is printed instead of "OSIZE" */
  if (print.ofrsiz) fprintf (fp, spc ? "%5.5s%s"       : "%s%s", "OSIZE", 
                             print.mode || print.qseq ||
                             print.mmid ? delim : end);
  if (print.mode) {
  /* for consistency with TAQ2 v.1.42b */
    if (date < 20000101) 
                    fprintf (fp, spc ? " %-4.4s%s"     : "%s%s", "MD",
                             print.qseq || print.mmid ? delim : end);
    else            fprintf (fp, spc ? " %-4.4s%s"     : "%s%s", "MODE",
                             print.qseq || print.mmid ? delim : end);
  }
  if (print.qseq)   fprintf (fp, spc ? "%7.7s%s"       : "%s%s", "QSEQ", 
                             print.mmid ? delim : end);
  if (print.mmid)   fprintf (fp, spc ? "%-4.4s%s"      : "%s%s", "MMID", end);
}


/*
/// ------------------------------------
/// void print_<var> (fp, <var>, spc, d)
/// ====================================
///    input parameters
///       FILE *fp            : output data file
///       <var>               : variable to output
///       int   spc           : flag to print spaces in the fields
///       char *d             : string terminator
///    description
///       writes <var> to file stream,
///         where <var> = {symbol, cusip, date, ex, tim, seq, prc 
///                        siz, cond, g127, qsiz, mode, mmid, name, uot, n}
/// ____________________________________
*/
/* common between trades and quotes */
void print_symbol (FILE *fp, char *s,  int spc, char *d) {
  if (spc) fprintf (fp, "%-7.7s%s",      s, d);
  else     fprintf (fp, "%.7s%s",        s, d);
}
void print_cusip  (FILE *fp, char *c,  int spc, char *d) {
  if (spc) fprintf (fp, "%-8.8s%s",     c, d);
  else     fprintf (fp, "%.8s%s",       c, d);
}
void print_date   (FILE *fp, long dt,  int spc, char *d) {
  if (spc) fprintf (fp, "      %8ld%s", dt, d);
  else     fprintf (fp, "%ld%s",        dt, d);
}
void print_ex     (FILE *fp, char ex,  int spc, char *d) {
           fprintf (fp, "%c%s",         ex, d);
}
void print_tim    (FILE *fp, long t,   int spc, char *d) {
  if (spc) fprintf (fp, "    %7ld%s",    t, d);
  else     fprintf (fp, "%ld%s",         t, d);
}
void print_prc    (FILE *fp, double p, int spc, char *d) {
  if (spc) fprintf (fp, "%15.8f%s",      p, d);
  else     fprintf (fp, "%0.8f%s",       p, d);
}
/* prints both tseq and qseq */
void print_seq    (FILE *fp, long s,   int spc, char *d) {
  if (spc) fprintf (fp, "%9ld%s",        s, d); 
  else     fprintf (fp, "%ld%s",         s, d); 
}
/* trade-specific */
void print_siz    (FILE *fp, long s,   int spc, char *d) {
  if (spc) fprintf (fp, "%10ld%s",       s, d);
  else     fprintf (fp, "%ld%s",         s, d);
}
void print_cond   (FILE *fp, char *c,  int spc, char *d) {
  if (spc) fprintf (fp, "%-4.4s%s",      c, d);
  else     fprintf (fp, "%.4s%s",        c, d);
}
void print_corr   (FILE *fp, short c,  int spc, char *d) {
  if (spc) fprintf (fp, "%2hd%s",        c, d);
  else     fprintf (fp, "%hd%s",         c, d);
}
void print_g127   (FILE *fp, short g,  int spc, char *d) {
  if (spc) fprintf (fp, "%4hd%s",        g, d);
  else     fprintf (fp, "%hd%s",         g, d);
}
/* quote-specific */
void print_qsiz   (FILE *fp, long s,   int spc, char *d) {
  if (spc) fprintf (fp, "%5ld%s",        s, d);
  else     fprintf (fp, "%ld%s",         s, d);
}
void print_mode   (FILE *fp, short m,  int spc, char *d) {
  if (spc) fprintf (fp, " %2hd%s",       m, d);
  else     fprintf (fp, "%hd%s",         m, d);
}
void print_mmid   (FILE *fp, char *m,  int spc, char *d) {
  if (spc) fprintf (fp, "%-4.4s%s",      m, d);
  else     fprintf (fp, "%.4s%s",        m, d);
}
/* summary-specific */
void print_name   (FILE *fp, char *n,  int spc, char *d) {
  if (spc) fprintf (fp, "%-30.30s%s",    n, d);
  else     fprintf (fp, "%.30s%s",       n, d);
}
void print_uot    (FILE *fp, long u,   int spc, char *d) {
  if (spc) fprintf (fp, "%6ld%s",        u, d);
  else     fprintf (fp, "%ld%s",         u, d);
}
void print_n     (FILE *fp, long t,    int spc, char *d) {
  if (spc) fprintf (fp, "%8ld%s",        t, d);
  else     fprintf (fp, "%ld%s",         t, d);
}


/*
/// -----------------------------
/// SUM_PRINTER sum_print_all  ()
/// =============================
///    description
///      returns SUM_PRINTER structure set up to print all of the summary items
/// _______________________________
*/
SUM_PRINTER sum_print_all  () {

  SUM_PRINTER print;
  print.s = print.c = print.n = print.d = print.u = print.nt = print.nq = 1;
  return (print);
}

/*
/// ------------------------------
/// SUM_PRINTER sum_print_none  ()
/// ==============================
///    description
///      returns SUM_PRINTER structure set up to print none of the summary items
/// _______________________________
*/
SUM_PRINTER sum_print_none  () {

  SUM_PRINTER print;
  print.s = print.c = print.n = print.d = print.u = print.nt = print.nq = 0;
  return (print);
}

/*
/// ---------------------------------
/// TRADE_PRINTER trade_print_all  ()
/// =================================
///    description
///      returns TRADE_PRINTER structure set up to print all of the trade items
/// _______________________________
*/
TRADE_PRINTER trade_print_all  () {

  TRADE_PRINTER print;
  print.s   = print.c    = print.d    = print.ex   = print.tim  = print.prc =
  print.siz = print.cond = print.corr = print.tseq = print.g127 = 1;
  return (print);
}

/*
/// ---------------------------------
/// TRADE_PRINTER trade_print_none ()
/// =================================
///    description
///      returns TRADE_PRINTER structure set up to print none of the trade items
/// _______________________________
*/
TRADE_PRINTER trade_print_none () {

  TRADE_PRINTER print;
  print.s   = print.c    = print.d    = print.ex   = print.tim  = print.prc =
  print.siz = print.cond = print.corr = print.tseq = print.g127 = 0;
  return (print);
}

/*
/// ---------------------------------
/// QUOTE_PRINTER quote_print_all  ()
/// =================================
///    description
///      returns QUOTE_PRINTER structure set up to print all of the quote items
/// _______________________________
*/
QUOTE_PRINTER quote_print_all  () {

  QUOTE_PRINTER print;
  print.s      = print.c    = print.d      = print.ex   =
  print.tim    = print.bid  = print.bidsiz = print.ofr  =
  print.ofrsiz = print.mode = print.qseq   = print.mmid = 1;
  return (print);
}

/*
/// ---------------------------------
/// QUOTE_PRINTER quote_print_none ()
/// =================================
///    description
///      returns QUOTE_PRINTER structure set up to print none of the quote items
/// _______________________________
*/
QUOTE_PRINTER quote_print_none () {

  QUOTE_PRINTER print;
  print.s      = print.c    = print.d      = print.ex   =
  print.tim    = print.bid  = print.bidsiz = print.ofr  =
  print.ofrsiz = print.mode = print.qseq   = print.mmid = 0;
  return (print);
}


/* Format based on TAQ2 v.1.42b fixed-width ASCII format
   NOTE: TAQ2 v.1.42b does not print CUSIP. Make sure to set up 
         {TRADE, QUOTE}_PRINTERs accordingly */

/*
/// -----------------------------------------
/// int trades_write_taq2 (fp, t, spc, print)
/// =========================================
///    input parameters
///       FILE *fp            : output data file
///       TICK  t             : tick data stored here (invalid if NULL)
///       int   spc           : flag to print spaces in the fields
///       TRADE_PRINTER print : data items to be printed
///    return
///       TAQ_ERROR (null TICK) or TAQ_SUCCESS 
///    description
///       writes all trades to file stream in TAQ2 ASCII format
/// _________________________________________
*/
int trades_write_taq2 (FILE *fp, TICK t, int spc, TRADE_PRINTER print)
{
  long i;
  if (t == NULL) return (TAQ_ERROR);
  for (i = 0; i < (t->nt); i++) 
    trade_write_ascii (fp, t, i, spc, '\0', '\n', print);
  return (TAQ_SUCCESS);
}

/*
/// -----------------------------------------
/// int quotes_write_taq2 (fp, t, spc, print)
/// =========================================
///    input parameters
///       FILE *fp            : output data file
///       TICK  t             : tick data stored here (invalid if NULL)
///       int   spc           : flag to print spaces in the fields
///       QUOTE_PRINTER print : data items to be printed
///    return
///       TAQ_ERROR (null TICK) or TAQ_SUCCESS 
///    description
///       writes all quotes to file stream in TAQ2 ASCII format
/// _________________________________________
*/
int quotes_write_taq2 (FILE *fp, TICK t, int spc, QUOTE_PRINTER print)
{
  long i;
  if (t == NULL) return (TAQ_ERROR);
  for (i = 0; i < (t->nq); i++) 
    quote_write_ascii (fp, t, i, spc, '\0', '\n', print);
  return (TAQ_SUCCESS);
}

