/******************************
   Program:  TAQAccess_sample.c
   author:   Boulat A. Bash
   email:    boulat@alum.dartmouth.org
   created:  August 2000
   updated:  July 2004
   comments: Program to sign trades as buyer/seller-initiated using a variation
             of Lee and Ready (1991) tick test.
******************************/

#define VERSION "1.0"
#define UPDATED "July 2004"
#define MAX_PATH 256

#include "TAQAccess_tick.h"
#include "TAQAccess_cd.h"
#include "TAQAccess_ascii.h"

void print_taqsign_help ();

/* Structure to hold a day's signed trading data for one stock */
typedef struct signstruct * SIGNS;
struct signstruct {
  char* sign;
  long voltot;
  long numbuy;
  long volbuy;
  long numsell;
  long volsell;
  long late;
  long cantsign;
  long unsignable;
  TICK_F t;
};

/* Routines related to signing */
SIGNS signs_new (TICK_F t);
void  signs_free (SIGNS s);
int   isafter (char *cond);
void tick_sign (SIGNS s);

int main (int argc, char *argv[]) {

  FILE *dfp = NULL, *ssfp = NULL, *stfp = NULL;
  long date, i;
  char taqpath[MAX_PATH], datepath[MAX_PATH], outfile[MAX_PATH];
  char *datefile = NULL, *ssoutfile = NULL, *stoutfile = NULL;
  char id[9];
  int  pss = 0, pst = 0, print_head = 1;
  TICK_F tf = NULL;
  SIGNS s = NULL;
  SUM_PRINTER sp;
  TRADE_PRINTER tp;

  /* Set default values */
  strcpy (taqpath, "/home/boulat/TAQAccess/TAQ_199610A/");
  //DEB: strcpy (taqpath, "");
  strcpy (datepath, "");
  strcpy (outfile, "out");
  sp = sum_print_all();
  tp = trade_print_all();


  fprintf (stderr, "taqsign version %s Copyright (c) %s Boulat A. Bash\n",
           VERSION, UPDATED);
  fprintf (stderr,
           "This  is  free software; see the source for copying ");
  fprintf (stderr,
           "conditions.\nThere is NO warranty; not even for MERCHANTABILITY ");
  fprintf (stderr, "or FITNESS FOR A\nPARTICULAR PURPOSE.\n\n");

  /* Process command-line arguments */
  argc--; argv++;
  while ((argc > 0) && (**argv == '-')) {
    switch (*(*argv+1)) {

      case 'p':
        if (argc > 1 && *argv[1] != '-') {
          strncpy (taqpath, argv[1], MAX_PATH - 1);
          argc--; argv++;
        }
        break;

      case 'd':
        if (argc > 1 && *argv[1] != '-') {
          strncpy (datepath, argv[1], MAX_PATH - 1);
          argc--; argv++;
        }
        break;

      case 'n':
        if (argc > 1 && *argv[1] != '-') {
          strncpy (outfile, argv[1], MAX_PATH - 1);
          argc--; argv++;
        }
        break;

      case 's':
        pss = 1;
        break;

      case 't':
        pst = 1;
        break;

      case 'h':
        print_taqsign_help ();
        return (TAQ_SUCCESS);
    }
    argc--; argv++;
  }
  if (!pss && !pst) {
    fprintf (stderr, "Enter taqsign -h for help\n\n");
    return (TAQ_SUCCESS);
  }

  /* open date2.dat */
  datefile = (char *) malloc (strlen (datepath) + strlen ("date2.dat") + 2);
  if (NULL == datefile) {
    fprintf (stderr, "malloc error: datefile\n");
    return (TAQ_ERROR);
  }
  sprintf (datefile, "%s%s", datepath, "date2.dat");
  dfp = fopen (datefile, "r");
  if (NULL == dfp) {
    fprintf (stderr, "file open error: %s\n", datefile);
    return (TAQ_ERROR);
  }

  while (2 == fscanf (stdin, "%8s %ld", id, &date)) {
    fprintf (stderr, "Reading id=%s date=%ld...  ", id, date);
    /* read the transaction data */
    if (read_stockday (dfp, taqpath, id, date, DAWN, DUSK, 1, 1,
                       1, (void *) &tf)) {
      tick_f_sort (tf); /* sort trades and quotes by time and sequence nums */
      s = signs_new (tf);
      if (NULL == s) {
        fprintf (stderr, "malloc error: signs\n");
        return (TAQ_ERROR);
      }

      /* sign trades */
      fprintf (stderr, "Signing...  ");
      tick_sign (s);

      /* prepare for output */
      for (i = 0; i < s->t->nt; i++) {
        s->voltot += s->t->ct[i].siz;
        switch (s->sign[i]) {
          case '.' : s->late++; break;
          case '?' : s->unsignable++; break;
          case '=' : s->cantsign++; break;
          case 'B' : s->numbuy++; s->volbuy += s->t->ct[i].siz; break;
          case 'b' : s->numbuy++; s->volbuy += s->t->ct[i].siz; break;
          case 'S' : s->numsell++; s->volsell += s->t->ct[i].siz; break;
          case 's' : s->numsell++; s->volsell += s->t->ct[i].siz; break;
        }
      }
      /* open output files on the first iteration of the loop */
      if (pss && NULL == ssfp) {
        ssoutfile = (char *) malloc (strlen (outfile) + strlen (".ss.csv") + 2);
        if (NULL == ssoutfile) {
          fprintf (stderr, "malloc error: ssoutfile\n"); return (TAQ_ERROR);
        }
        sprintf (ssoutfile, "%s.ss.csv", outfile);
        ssfp = fopen (ssoutfile, "w");
        if (NULL == ssfp) {
          fprintf (stderr, "file open error: %s\n", ssoutfile);
          return (TAQ_ERROR);
        }
      }
      if (pst && NULL == stfp) {
        stoutfile = (char *) malloc (strlen (outfile) + strlen (".st.csv") + 2);
        if (NULL == stoutfile) {
          fprintf (stderr, "malloc error: stoutfile\n"); return (TAQ_ERROR);
        }
        sprintf (stoutfile, "%s.st.csv", outfile);
        stfp = fopen (stoutfile, "w");
        if (NULL == stfp) {
          fprintf (stderr, "file open error: %s\n", stoutfile);
          return (TAQ_ERROR);
        }
      }

      /* write headers (only on first iteration), then write data */
      if (pss) {
        if (print_head) {
          s_header_write_ascii (ssfp, 0, ',', ',', sp);
          fprintf
            (ssfp,
             "TOTVOL,NUMBUY,VOLBUY,NUMSELL,VOLSELL,LATE,UNSIGNABLE,CANTSIGN\n");
        }
        tick_f_sum_write_ascii (ssfp, s->t, 0, ',', ',', sp);
        fprintf (ssfp, "%ld,%ld,%ld,%ld,%ld,%ld,%ld,%ld\n", 
                 s->voltot, s->numbuy, s->volbuy, s->numsell, s->volsell,
                 s->late, s->unsignable, s->cantsign);
      }
      if (pst) {
        if (print_head) {
          t_header_write_ascii (stfp, 20000101, 0, ',', ',', tp);
          fprintf (stfp, "SIGN\n");
        }
        for (i = 0; i < s->t->nt; i++) {
          trade_f_write_ascii (stfp, s->t, i, 0, ',', ',', tp);
          fprintf (stfp, "%c\n", s->sign[i]);
        }
      }
      print_head = 0;
      tick_f_free (s->t);
      signs_free (s);
      tf = NULL;
      s = NULL;
      fprintf (stderr, "ok.\n");
    }
    else fprintf (stderr, "FAILURE!\n");
  }
  /* clean up */
  if (pss && (NULL != ssfp)) fclose (ssfp);
  if (pst && (NULL != stfp)) fclose (stfp);
  fclose (dfp);

  return (0);
}


SIGNS signs_new (TICK_F t) {

  SIGNS s = NULL;
  s = (SIGNS) malloc (sizeof (*s));
  if (NULL == s) return (NULL);
  s->t = t;
  s->sign = (char *) malloc (sizeof (*s->sign) * s->t->nt);
  if (NULL == s->sign) {
    free (s);
    return (NULL);
  }
  s->numbuy = s->volbuy = s->numsell = s->volsell = 0;
  s->voltot = s->late = s->cantsign = s->unsignable = 0;
  return (s);
}

void signs_free (SIGNS s) {

  free (s->sign);
  free (s);
}


int isafter (char *cond) {

  return (memchr (cond, 'T', 4) ||
          memchr (cond, '8', 4) ||
          memchr (cond, '9', 4));
}


/*
/// -------------------
/// void tick_sign (t)
/// ===================
///    input parameters
///       TICK t : transactions data structure
///    description
///       for each trade, finds prevailing quote and infers trade direction
///       fills t->qdex with index of prevailing quote, and t->sign
///       with direction
///
///       NOTE: important differences from the original version of tick_sign
///             include the following:
///             (1) no trade reporting delay is implemented for NASD quotes;
///             (2) signs using best spread across all exchanges
///
///       Fills a time-series vector with the sign of the associated trade.
///       The following signs are assigned, using mid-quote and tick tests:
///
///       '.'    : unsigned trade   -- late or after-close
///       '?'    : unsignable trade -- no prevailing quote
///       '='    : unsigned trade   -- cannot otherwise be signed
///       'B'    : buy  -- trade price is above prevailing midquote
///       'S'    : sell -- trade price is below prevailing midquote
///       'b'    : buy  -- price at prevailing midquote, and
///                        price greater than previous trade price
///       's'    : sell -- price at prevailing midquote, and
///                        price less than previous trade price
/// ___________________
*/

void tick_sign (SIGNS s) {

  long   it, tim, best[10], iq = 0, i;
  double bid, ofr, mid;

  for (i = 0; i < 10; i++) best[i] = -1;

  /* first pass: loop over all trades to sign using midquote test */
  for(it = 0; it < s->t->nt; it++) {

    /* check trade condition codes to ensure it's not an opening or
       after-close crossing, or out of sequence trade */
    if(!isafter (s->t->ct[it].cond) &&
       !memchr (s->t->ct[it].cond, 'L', 4) &&
       !memchr (s->t->ct[it].cond, 'O', 4) &&
       !memchr (s->t->ct[it].cond, 'Z', 4) &&
       s->t->ct[it].prc > 0.0) {

      /* no trade reporting delay for NASD, 5 seconds for other exchanges */
      if (s->t->ct[it].ex == 'T') tim = s->t->ct[it].tim;
      else                        tim = s->t->ct[it].tim - 5;

      /* find the most current quotes at each exchange */
      for ( ; iq < s->t->nq && s->t->cq[iq].tim <= tim; iq++) {
        switch (s->t->cq[iq].ex) {
          case 'A' : best[0] = iq; break;
          case 'B' : best[1] = iq; break;
          case 'C' : best[2] = iq; break;
          case 'M' : best[3] = iq; break;
          case 'N' : best[4] = iq; break;
          case 'P' : best[5] = iq; break;
          case 'T' : best[6] = iq; break;
          case 'X' : best[7] = iq; break;
          case 'O' : best[8] = iq; break;
          case 'W' : best[9] = iq; break;
        }
      }

      /* calculate current best spread across all exchanges */
      for (bid = 0.0, i = 0; i < 10; i++) {
        if (best[i] >= 0 && (bid == 0.0 || bid < s->t->cq[best[i]].bid))
          bid = s->t->cq[best[i]].bid;
        }
      for (ofr = 0.0, i = 0; i < 10; i++) {
        if (best[i] >= 0 && (ofr == 0.0 || ofr > s->t->cq[best[i]].ofr))
          ofr = s->t->cq[best[i]].ofr;
      }

      /* use midquote test */
      if (bid > 0.0 && ofr > 0.0) {

        mid = (ofr + bid) / 2.0;
        if (s->t->ct[it].prc == mid) s->sign[it] = '=';
        else s->sign[it] = (s->t->ct[it].prc > mid ? 'B' : 'S');
      }
      else s->sign[it] = '?';
    }
    else s->sign[it] = '.';
  }

  /* second pass: loop over all trades that could not be signed
     using midquote test.  We use tick-test here. */
  for (it = 1; it < s->t->nt; it++) {

   /* do not sign using tick-test if it's an after-close trade or price
      did not change */
    if (!isafter (s->t->ct[it].cond) &&
        !memchr (s->t->ct[it].cond, 'Z', 4) &&
        s->t->ct[it].prc > 0.0 && s->t->ct[it-1].prc > 0.0 &&
        s->t->ct[it].prc != s->t->ct[it-1].prc) {

      /* do not sign if already signed */
      if (!isalpha (s->sign[it]))
        s->sign[it] = (s->t->ct[it].prc > s->t->ct[it-1].prc ? 'b' : 's');
    }
  }
}


void print_taqsign_help () {

  fprintf (stdout, "NAME\n\ttaqsign - sign TAQ trade and output to ");
  fprintf (stdout, "comma-separated text (csv) files\n\n");

  fprintf (stdout, "SYNOPSIS\n\ttaqsign [options]\n\n");

  fprintf (stdout, "DESCRIPTION\n");
  fprintf (stdout, "\ttaqsign signs trades as buyer/seller-initiated for ");
  fprintf (stdout, "specified stocks\n\tusing TAQ trading data and a ");
  fprintf (stdout, "variation of the algorithm published by\n\tCharles Lee ");
  fprintf (stdout, "and Mark Ready in The Journal of Finance in 1991.");
  fprintf (stdout, "\n\tOutput is generated for the list of stock ");
  fprintf (stdout, "symbols/cusips and dates\n\tread from standard input.  ");
  fprintf (stdout, "\n\tThe following is example input:\n");
  fprintf (stdout, "\t\tIBM      20020604\n");
  fprintf (stdout, "\t\t59491810 19961001\n");
  fprintf (stdout, "\tThe stock symbols should be at most 7 characters long; ");
  fprintf (stdout, "the cusips should\n\tbe *exactly* 8 characters long.  ");
  fprintf (stdout, "The date should be in YYYYMMDD format.\n");
  fprintf (stdout, "\tThe program writes the ordinary summary data as well as");
  fprintf (stdout, " total daily\n\tvolume, signed volume, and numbers of ");
  fprintf (stdout, "variously signed trades to\n\tOUTPUTFILE.ss.csv.  ");
  fprintf (stdout, "Signed trades are written to OUTPUTFILE.st.csv.");
  fprintf (stdout, "\n\tBoth files contain comma-separated text data.\n\n");

  fprintf (stdout, "COMMAND-LINE OPTIONS\n\n");
  fprintf (stdout, "\t-p TAQPATH\n");
  fprintf (stdout, "\t\tSet the location of TAQ data to TAQPATH.\n");
  fprintf (stdout, "\t\tDefault is present working directory.\n\n");
  fprintf (stdout, "\t-d DATEPATH\n");
  fprintf (stdout, "\t\tSet the location of date2.dat file to DATEPATH.\n");
  fprintf (stdout, "\t\tDefault is present working directory.\n\n");
  fprintf (stdout, "\t-n OUTPUTFILE\n");
  fprintf (stdout, "\t\tSet the prefix of the output files.  ");
  fprintf (stdout, "Default is `out'.\n");

  fprintf (stdout, "\t-s\n");
  fprintf (stdout, "\t\tWrite summary.\n");
  fprintf (stdout, "\t-t\n");
  fprintf (stdout, "\t\tWrite signed trade records.\n");

  fprintf (stdout, "EXAMPLES\n");
  fprintf (stdout, "\tTo write signed trade data to myout.st.csv ");
  fprintf (stdout, "with date2.dat located\n\tin setup/ and");
  fprintf (stdout, "TAQ data files located in data/:\n");
  fprintf (stdout, "\t\ttaqsign -n myout -d setup/ -p data/ -t\n\n");
  fprintf (stdout, "\tTo write signed summary and trade data:\n");
  fprintf (stdout, "\t\ttaqsign -s -t\n\n");
}
