/******************************
   Library:  TAQAccess_cd.c
   author:   Boulat A. Bash
   email:    boulat@alum.dartmouth.org
   created:  July      2000
   updated:  February  2004
   comments: C Routines to process TAQ CD
             Based on taq.c/tick.c created June 1996 by Terence M. Lim
             Particular thanks to Bob Burnham
               for work on migration to TAQ2 (TAQ format after 01-01-2000).
******************************/

#include "TAQAccess_cd.h"

/*-----------------------------------------
  local variables
------------------------------------------*/

static char buf[256];
static NASDINDEX  nasd1028[6000];
static int       nnasd1028=0;
static FILE     *nasd1028fp=NULL;

/*-----------------------------------------
   Read in data records from TAQ CD
------------------------------------------*/

/*
/// -------------------------------
/// int taq_cd_init(p, m, d, f, cd)
/// ===============================
///    input parameters
///       char  *p : TAQ CD-ROM path (e.g. /cdrom/cdrom0/ or l:\\)
///       int    m : month for CD (yyyymm, e.g. 199301)
///       char   d : disk designation (a, b, c, ...)
///    output parameter
///       TAQCD cd : structure containing TAQ CD ready for access
///    returns
///              1 : successful
///              0 : unsuccessful
/// _______________________________
*/
int taq_cd_init (char *p, int m, char d, TAQCD *cd) {
  char   ctbegfile[3],  cqbegfile[3],
       mastbegfile[5], divbegfile[4];

  strcpy (cd->path, p);
  cd->month    = m;

  cd->disk = d;
  cd->idxlen = 22;  /* index length hasn't changed overtime... */
  if (cd->month < 200001) {
    
    cd->ctbinlen = 22;
    if (cd->month < 199308) cd->cqbinlen=23;
    else                    cd->cqbinlen=27;
    cd->mastlen  = 91;
    cd->divlen   = 53;
    
    sprintf (  ctbegfile, "%s", "ct");
    sprintf (  cqbegfile, "%s", "cq");
    sprintf (mastbegfile, "%s", "mast");
    sprintf ( divbegfile, "%s", "div");
    
    m = m % 10000;  /* convert month to YYMM before 200001 */
  }
  else {
    
    cd->ctbinlen = 29;
    cd->cqbinlen = 39;
    cd->mastlen  = 95;
    cd->divlen   = 55;
    
    sprintf (  ctbegfile, "%s", "t");
    sprintf (  cqbegfile, "%s", "q");
    sprintf (mastbegfile, "%s", "m");
    sprintf ( divbegfile, "%s", "d");
  }

  sprintf (buf, "%s%s%d%c.bin", cd->path, ctbegfile, m, cd->disk);
  cd->ctbinfp = fopen(buf,"rb");
  sprintf (buf, "%s%s%d%c.idx", cd->path, ctbegfile, m, cd->disk);
  cd->ctidxfp = fopen(buf,"rb");
  sprintf (buf, "%s%s%d%c.bin", cd->path, cqbegfile, m, cd->disk);
  cd->cqbinfp = fopen(buf,"rb");
  sprintf (buf, "%s%s%d%c.idx", cd->path, cqbegfile, m, cd->disk);
  cd->cqidxfp = fopen(buf,"rb");
  sprintf (buf, "%s%s%d.tab",   cd->path, mastbegfile, m);
  cd->mastfp = fopen(buf,"r");
  sprintf (buf, "%s%s%d.tab",   cd->path, divbegfile, m);
  cd->divfp = fopen(buf,"r");

  if (cd->ctbinfp && cd->ctidxfp && 
      cd->cqbinfp && cd->cqidxfp &&
      cd->mastfp  && cd->divfp     ) {

    return (1);
  }
  fprintf(stderr,"Could not access volume %d in %s\n",m,p);
  return (0);
}

/*
/// -------------------
/// void taq_cd_cleanup(cd)
/// ===================
///    input parameter
///       TAQCD cd : structure containing open TAQ CD
///    returns
///       1 : successful
///       0 : unsuccessful
/// ___________________
*/
void taq_cd_cleanup (TAQCD *cd)
{
  fclose (cd->ctbinfp);
  fclose (cd->ctidxfp);
  fclose (cd->cqbinfp);
  fclose (cd->cqidxfp);
  fclose (cd->mastfp);
  fclose (cd->divfp);

  cd->ctbinfp = cd->ctidxfp =
  cd->cqbinfp = cd->cqidxfp =
  cd->mastfp  = cd->divfp   = NULL;
}

/*
/// -------------------
/// int read_stockday (dfp, p, ident, date, begtime, endtime,
///                    dbt, dbq, isfloat, t)
/// ===================
///    input parameters
///       FILE     dfp : open file pointer to date2.dat (NOTE: always use most
///                      recent date2.dat)
///       char      *p : TAQ CD-ROM path (e.g. /cdrom/cdrom0/ or l:\\)
///       char  *ident : stock id to retrieve: can be either ticker symbol
///                      of CUSIP.  First 8 characters of the latter are used.
///       int     date : trading date to be retrieved, in YYYYMMDD format
///       int  begtime : beginning of time period to read trades/quotes from
///       int  endtime : end of time period to read trades/quotes from
///       
///       int      dbt : flag to drop bad trades
///       int      dbq : flag to drop bad quotes
///
///       int  isfloat : if true, returned t is a pointer to TICK_F structure
///                      otherwise t points for TICK structure
///    output
///       void *t      : tick data (TICK_F or TICK depending on isfloat)
///    returns
///                  1 : successful
///                  0 : unsuccessful
///    description
///       reads stock's transactions and quotes for a day into TICK_F or TICK
///       structure (depending on whether isfloat is true or not).
///       WARNING: This code contains flagrant abuse of C weak type system!
///                Kids, do not try this at home.
/// ___________________
*/
int read_stockday (FILE *dfp, char *p,
                   char *ident, int date, int begtime, int endtime,
                   int dbt, int dbq,
                   int isfloat, void *t) {
  
  int ret;
  long    i, cqidxb, cqidxe, ctidxb, ctidxe, uot, denom;
  long    mdate, pdate = 0;
  char    d, id[8], tabsymbol[7], tabcusip[8];
  char    tabname[30];
  IDX     ctidx, cqidx;
  TAQCD   cd;
  MASTTAB masttab;

  /* Initialize */
  memset (id,        '\0',  8);
  mkstring (id, ident, (strlen (ident) >  8 ?  8 : strlen (ident)));

  /* Read index locations from DATE.DAT */

  rewind(dfp);
  while (fscanf(dfp, "%s", buf) == 1) {

    /* ALWAYS use most recent date2.dat!!! (and, thus, YYYYMMDD format...) */
    if (atoi(buf) == date) {
    
      fscanf (dfp, "%ld %ld %ld %ld", &cqidxb, &cqidxe, &ctidxb, &ctidxe);
      if(date>=19951231) {
        fscanf(dfp,"%1s", &d); /* no disk letter before 1996 */
        d = tolower (d);       /* UNIX files are lower case, WIN ignores case */
      }
      else               d = '\0';
      fscanf (dfp, "%s", buf);   /* discard format field, we use raw dates */
      
      if (!taq_cd_init(p, date / 100, d, &cd)) return (0);

      /* Read latest, applicable Master Table record */
      while (getmasttab (cd, &masttab)) {

        strnul (masttab.symbol, 7);
        strnul (masttab.cusip,  12);
        mdate = masttab.datef;

        /* pick by either ticker symbol or CUSIP */
        /* we use 8-character CUSIP for compatibility with CRSP */
        if (((memcmp (masttab.symbol, id, 7) == 0) || 
             (memcmp (masttab.cusip,  id, 8) == 0)   ) &&
            mdate <=  date                             &&
            mdate >= pdate) {

          uot    = mastuot(masttab.uot);
          denom  = mastdenom(masttab.denom);
          memset (tabsymbol, '\0', 7);
          mkstring (tabsymbol, masttab.symbol, 7);
          memset (tabcusip,  '\0', 8);
          mkstring (tabcusip,  masttab.cusip,  8);
          memset (tabname,   '\0', 30);
          mkstring (tabname,   masttab.name,  30);
          pdate  = mdate;
        }
      }
      
      /* Find trade and quote indices */
      for (i = ctidxb; i <= ctidxe; i++) {
        getctidx (i, cd, &ctidx);
        if (!memcmp (tabsymbol, ctidx.symbol, 7)) break;
      }
      if (i > ctidxe) memset (&ctidx, 0, sizeof (ctidx));

      for (i = cqidxb; i <= cqidxe; i++) {
        getcqidx (i, cd, &cqidx);
        if (!memcmp (tabsymbol, cqidx.symbol, 7)) break;
      }
      if (i > cqidxe) memset (&cqidx, 0, sizeof (cqidx));
      
      /* Read the data into appropriate data structure */
      if (isfloat) ret = tick_f_read_cd (tabsymbol, tabcusip, ctidx, cqidx, cd, 
                                         begtime, endtime, '\0', date, uot, 
                                         tabname, dbt, dbq, (TICK_F *) t);
      else         ret = tick_read_cd   (tabsymbol, tabcusip, ctidx, cqidx, cd, 
                                         begtime, endtime, '\0', date, uot,
                                         tabname, dbt, dbq, (TICK *) t);
      taq_cd_cleanup (&cd);
      return (ret);
    }
  }
  fprintf(stderr, "Date %d not found in DATE.DAT\n", date);
  return (0);
}

/*
/// -------------------
/// void tick_read_cd (symbol, cusip, ctidx, cqidx, cd, begtime, endtime, ex,
///                    date, uot, dbt, dbq, t)
/// ===================
///    input parameters
///      char *symbol : ticker symbol to retrieve
///      char  *cusip : CUSIP
///       IDX   ctidx : trade index record
///       IDX   cqidx : quote index record
///       TAQCD    cd : structure containing open TAQ CD
///       
///       int begtime : beginning of time period to read trades/quotes from
///       int endtime : end of time period to read trades/quotes from
///           
///       int      ex : primary exchange (from master tab file)
///       int    date : trading date to be retreived, in YYYYMMDD format
///       long    uot : number of shares in round lot (from master tab file)
///      char   *name : company name (from master tab file)
///
///       int     dbt : flag to drop bad trades
///       int     dbq : flag to drop bad quotes
///    output
///       TICK     *t : tick data (memory allocated if NULL)
///    returns
///                 1 : successful
///                 0 : unsuccessful
///    description
///       writes raw TAQ CD data into TICK structure;
///       includes a kludge (by Terence Lim) to handle Nasdaq trades on 10-28-97
/// ___________________
*/
int tick_read_cd (char *symbol, char *cusip, IDX ctidx, IDX cqidx, TAQCD cd,
                   int begtime, int endtime, int ex, long date, long uot,
                   char *name, int dbt, int dbq,
                   TICK *t) {

  long i;
  int nasdflag;
  CTBIN ctbin;
  CQBIN cqbin;

  /* NASD 10-28-1997 kludge */
  sprintf (buf, "%s%s", cd.path, "nasd1028.txt");
  nasdflag = getnasdctidx(buf, &ctidx);

  /* Initialize TICK structure */
  if(*t != NULL) {

    tick_free(*t);
    *t = NULL;
  }

  *t = tick_new(ctidx.erec - ctidx.brec + 1, cqidx.erec - cqidx.brec + 1);
  if (*t == NULL) {
    fprintf (stderr, "Failed to allocate memory for t in tick_read_cd()");
    return (0);
  }

  memcpy ((*t)->symbol, symbol, 7);
  memcpy ((*t)->cusip,  cusip,  8);
  memcpy ((*t)->name,   name,   30);
  (*t)->date = date;
  (*t)->uot  = uot;

  /* read trades into TICK stricture */
  for (i = ctidx.brec; i <= ctidx.erec && ctidx.erec; i++) {

    if   (nasdflag >= 0) { /* NASD 10-28-1997 kludge */
      if (!getnasdctbin (i, cd, &ctbin))
        taqerr("getnasdctbin");
    }
    else {
      if (!getctbin (i, cd, &ctbin))
        taqerr("getctbin");
    }

    if ((0 == dbt || okctbin (ctbin)) &&
        ctbin.ttim >= begtime && ctbin.ttim <= endtime) {

      (*t)->ct[(*t)->nt].tim  = ctbin.ttim;
      (*t)->ct[(*t)->nt].prc  = ctbin.price * (date < 20000101 ? 1000 : 0);
      (*t)->ct[(*t)->nt].siz  = ctbin.siz;
      (*t)->ct[(*t)->nt].tseq = ctbin.tseq;
      (*t)->ct[(*t)->nt].g127 = ctbin.g127;
      (*t)->ct[(*t)->nt].corr = ctbin.corr;
      memcpy((*t)->ct[(*t)->nt].cond, ctbin.cond,4);
      (*t)->ct[(*t)->nt].ex   = ctbin.ex;
      (*t)->nt++;
    }
  }
  /* read quotes into TICK stricture */
  for (i = cqidx.brec; i <= cqidx.erec && cqidx.erec; i++) {

    if (!getcqbin(i, cd, &cqbin)) { taqerr("getcqbin"); }
    if ((dbq == 0 || okcqbin (cqbin, date)) &&
        cqbin.qtim >= begtime && cqbin.qtim <= endtime) {

      (*t)->cq[(*t)->nq].tim    = cqbin.qtim;
      (*t)->cq[(*t)->nq].bid    = cqbin.bid * (date < 20000101 ? 1000 : 0);
      (*t)->cq[(*t)->nq].ofr    = cqbin.ofr * (date < 20000101 ? 1000 : 0);
      (*t)->cq[(*t)->nq].qseq   = cqbin.qseq;
      (*t)->cq[(*t)->nq].bidsiz = cqbin.bidsiz;
      (*t)->cq[(*t)->nq].ofrsiz = cqbin.ofrsiz;
      (*t)->cq[(*t)->nq].mode   = cqbin.mode;
      (*t)->cq[(*t)->nq].ex     = cqbin.ex;
      memcpy((*t)->cq[(*t)->nq].mmid, cqbin.mmid,4);
      (*t)->nq++;
    }
  }
  return (1);
}

/*
/// -------------------
/// void tick_f_read_cd (symbol, cusip, ctidx, cqidx, cd, begtime, endtime, ex,
///                      date, uot, dbt, dbq, t)
/// ===================
///    input parameters
///      char *symbol : ticker symbol to retrieve
///      char  *cusip : CUSIP
///       IDX   ctidx : trade index record
///       IDX   cqidx : quote index record
///       TAQCD    cd : structure containing open TAQ CD
///       
///       int begtime : beginning of time period to read trades/quotes from
///       int endtime : end of time period to read trades/quotes from
///           
///       int      ex : primary exchange (from master tab file)
///       int    date : trading date to be retreived, in YYYYMMDD format
///       long    uot : number of shares in round lot (from master tab file)
///      char   *name : company name (from master tab file)
///
///       int     dbt : flag to drop bad trades
///       int     dbq : flag to drop bad quotes
///    output
///       TICK_F   *t : tick data (memory allocated if NULL)
///    returns
///                 1 : successful
///                 0 : unsuccessful
///    description
///       writes raw TAQ CD data into TICK_F structure;
///       includes a kludge by Terence Lim to handle NASDAQ trades on 10-28-1997
/// ___________________
*/
int tick_f_read_cd (char *symbol, char *cusip, IDX ctidx, IDX cqidx, TAQCD cd,
                    int begtime, int endtime, int ex, long date, long uot,
                    char *name, int dbt, int dbq,
                    TICK_F *t) {

  long i;
  int nasdflag;
  CTBIN ctbin;
  CQBIN cqbin;

  /* NASD 10-28-1997 kludge */
  sprintf (buf, "%s%s", cd.path, "nasd1028.txt");
  nasdflag = getnasdctidx(buf, &ctidx);

  /* Initialize TICK_F structure */
  if(*t != NULL) {

    tick_f_free(*t);
    *t = NULL;
  }

  *t = tick_f_new(ctidx.erec - ctidx.brec + 1, cqidx.erec - cqidx.brec + 1);
  if (*t == NULL) {
    fprintf (stderr, "Failed to allocate memory for t in tick_f_read_cd()");
    return (0);
  }

  memcpy ((*t)->symbol, symbol, 7);
  memcpy ((*t)->cusip,  cusip,  8);
  memcpy ((*t)->name,   name,   30);
  (*t)->date = date;
  (*t)->uot  = uot;

  /* read trades into TICK_F stricture */
  for (i = ctidx.brec; i <= ctidx.erec && ctidx.erec; i++) {

    if   (nasdflag >= 0) { /* NASD 10-28-1997 kludge */
      if (!getnasdctbin (i, cd, &ctbin))
        taqerr("getnasdctbin");
    }
    else {
      if (!getctbin (i, cd, &ctbin))
        taqerr("getctbin");
    }

    if ((0 == dbt || okctbin (ctbin)) &&
        ctbin.ttim >= begtime && ctbin.ttim <= endtime) {

      (*t)->ct[(*t)->nt].tim  = ctbin.ttim;
      (*t)->ct[(*t)->nt].prc  = (double) ctbin.price / 
                                  (date < 20000101 ? 10000.0 : 100000000.0);
      (*t)->ct[(*t)->nt].siz  = ctbin.siz;
      (*t)->ct[(*t)->nt].tseq = ctbin.tseq;
      (*t)->ct[(*t)->nt].g127 = ctbin.g127;
      (*t)->ct[(*t)->nt].corr = ctbin.corr;
      memcpy((*t)->ct[(*t)->nt].cond, ctbin.cond,4);
      (*t)->ct[(*t)->nt].ex   = ctbin.ex;
      (*t)->nt++;
    }
  }
  /* read quotes into TICK_F stricture */
  for (i = cqidx.brec; i <= cqidx.erec && cqidx.erec; i++) {

    if (!getcqbin(i, cd, &cqbin)) { taqerr("getcqbin"); }
    if ((dbq == 0 || okcqbin (cqbin, date)) &&
        cqbin.qtim >= begtime && cqbin.qtim <= endtime) {

      (*t)->cq[(*t)->nq].tim    = cqbin.qtim;
      (*t)->cq[(*t)->nq].bid    = cqbin.bid /
                                    (date < 20000101 ? 10000.0 : 100000000.0);
      (*t)->cq[(*t)->nq].ofr    = cqbin.ofr /
                                    (date < 20000101 ? 10000.0 : 100000000.0);
      (*t)->cq[(*t)->nq].qseq   = cqbin.qseq;
      (*t)->cq[(*t)->nq].bidsiz = cqbin.bidsiz;
      (*t)->cq[(*t)->nq].ofrsiz = cqbin.ofrsiz;
      (*t)->cq[(*t)->nq].mode   = cqbin.mode;
      (*t)->cq[(*t)->nq].ex     = cqbin.ex;
      memcpy((*t)->cq[(*t)->nq].mmid, cqbin.mmid,4);
      (*t)->nq++;
    }
  }
  return (1);
}


/*-----------------------------------------
   Filter raw TAQ CD records
  ----------------------------------------*/

int okctbin (CTBIN ctbin) {
  return ((ctbin.corr <= 2) &&
          (ctbin.price > 0) &&
          (ctbin.ttim  > 0) &&
          (ctbin.siz   > 0));
}

int okcqbin (CQBIN cqbin, long date) {
  return (okspread (cqbin.bid * (date < 20000101 ? 1000 : 0),
                    cqbin.ofr * (date < 20000101 ? 1000 : 0)) &&
          (cqbin.qtim > 0));
}


/*-----------------------------------------
   Read raw records on TAQ CD
------------------------------------------*/

/*
/// -------------------
/// int getctbin(irec, cd, ctbin)
/// ===================
///    input parameters
///       long   irec : TAQ CD record location (starts from record 1, not 0)
///       TAQCD    cd : structure containing open TAQ CD
///    output parameter
///       CTBIN ctbin : trade record read from TAQ CD
///    returns
///                 1 : successful
///                 0 : unsuccessful
/// ___________________
*/
int getctbin(long irec, TAQCD cd, CTBIN *ctbin)
{
  if(irec <= 0 || 
     fseek(cd.ctbinfp, ((long) ((irec - 1) * cd.ctbinlen)), 0)) return(0);

  if(fread(buf, cd.ctbinlen, 1, cd.ctbinfp) == 1) {
    
    if (cd.month < 200001) {

      ctbin->ttim    =             mklong  (&(buf[ 0]));
      ctbin->price   = (LONGLONG)  mklong  (&(buf[ 4]));
      ctbin->siz     =             mklong  (&(buf[ 8]));
      ctbin->tseq    =             mklong  (&(buf[12]));
      ctbin->g127    =             mkshort (&(buf[16]));
      ctbin->corr    =             mkshort (&(buf[18]));
      memset(ctbin->cond, '\0', 4);
      mkstring (ctbin->cond, &(buf[20]), 1);
      ctbin->ex      =                        buf[21];
    }
    else {
      
      ctbin->ttim    =             mklong  (&(buf[ 0]));
      ctbin->price   =             mkvlong (&(buf[ 4]));
      ctbin->siz     =             mklong  (&(buf[12]));
      ctbin->tseq    =             mklong  (&(buf[16]));
      ctbin->g127    =             mkshort (&(buf[20]));
      ctbin->corr    =             mkshort (&(buf[22]));
      memset(ctbin->cond, '\0', 4);
      mkstring (ctbin->cond, &(buf[24]), 4);
      ctbin->ex      =                        buf[28];
    }
    return(1);
  }
  return(0);
}


/*
/// -------------------
/// int getcqbin(irec, cd, cqbin)
/// ===================
///    input parameters
///       long   irec : TAQ CD record location (starts from record 1, not 0)
///       TAQCD    cd : structure containing open TAQ CD
///    output parameter
///       CQBIN cqbin : quote record read from TAQ CD
///    returns
///                 1 : successful
///                 0 : unsuccessful
/// ___________________
*/
int getcqbin(long irec, TAQCD cd, CQBIN *cqbin)
{
  if(irec<=0||fseek(cd.cqbinfp,((long) ((irec-1)*cd.cqbinlen)),0)) return(0);
  if(fread(buf,cd.cqbinlen,1,cd.cqbinfp)==1) {

    if (cd.month < 200001) {

      cqbin->qtim   =             mklong  (&(buf[ 0]));
      cqbin->bid    = (LONGLONG)  mklong  (&(buf[ 4]));
      cqbin->ofr    = (LONGLONG)  mklong  (&(buf[ 8]));
      cqbin->qseq   =             mklong  (&(buf[12]));
      cqbin->bidsiz =             mkshort (&(buf[16]));
      cqbin->ofrsiz =             mkshort (&(buf[18]));
      cqbin->mode   =             mkshort (&(buf[20]));
      cqbin->ex     =                        buf[22];
      memset(cqbin->mmid, '\0', 4);
      mkstring (cqbin->mmid, &(buf[23]), 4);
    }
    else {

      cqbin->qtim   =             mklong  (&(buf[ 0]));
      cqbin->bid    =             mkvlong (&(buf[ 4]));
      cqbin->ofr    =             mkvlong (&(buf[12]));
      cqbin->qseq   =             mklong  (&(buf[20]));
      cqbin->bidsiz =             mkshort (&(buf[24]));
      cqbin->ofrsiz =             mkshort (&(buf[28]));
      cqbin->mode   =             mkshort (&(buf[32]));
      cqbin->ex     =                        buf[34];
      memset(cqbin->mmid, '\0', 4);
      mkstring (cqbin->mmid, &(buf[35]), 4);
    }
    return (1);
  }
  return (0);
}

/*
/// -------------------
/// int getcqidx(irec, cd, cqidx)
/// ===================
///    input parameters
///       long irec : TAQ CD record location (starts from record 1, not 0)
///       TAQCD  cd : structure containing open TAQ CD
///    output parameter
///       IDX cqidx : quote index record read from TAQ CD
///    returns
///               1 : successful
///               0 : unsuccessful
/// ___________________
*/
int getcqidx (long irec, TAQCD cd, IDX *cqidx)
{
  fseek (cd.cqidxfp, ((long) ((irec-1) * cd.idxlen)), 0);
  if (fread (buf, cd.idxlen, 1, cd.cqidxfp) == 1) {

    mkstring (cqidx->symbol,  buf, 7);
    cqidx->tdate = mklong (&(buf[10]));
    cqidx->brec  = mklong (&(buf[14]));
    cqidx->erec  = mklong (&(buf[18]));
    return (1);
  }
  return (0);
}


/*
/// -------------------
/// int getctidx(irec, cd, ctidx)
/// ===================
///    input parameters
///       long irec : TAQ CD record location (starts from record 1, not 0)
///       TAQCD cd  : structure containing open TAQ CD
///    output parameter
///       IDX ctidx : trade index record read from TAQ CD
///    returns
///               1 : successful
///               0 : unsuccessful
/// ___________________
*/
int getctidx (long irec, TAQCD cd, IDX *ctidx)
{
  fseek (cd.ctidxfp, ((long) ((irec-1) * cd.idxlen)), 0);
  if (fread (buf,cd.idxlen,1,cd.ctidxfp)==1) {
    mkstring (ctidx->symbol,  buf, 7);
    ctidx->tdate = mklong (&(buf[10]));
    ctidx->brec  = mklong (&(buf[14]));
    ctidx->erec  = mklong (&(buf[18]));
    return(1);
  }
  return(0);
}


/*
/// -------------------
/// int getdivtab(fp,d)
/// ===================
///    input parameters
///      FILE  *fp : file pointer of dividend table file
///    output parameters
///      DIVTAB *d : previously allocated space to contain a dividend table record
///    description
///       reads in a dividend table record
///    returns
///              1 : successful
///              0 : unsuccessful
/// ___________________
*/
int getdivtab(TAQCD cd,DIVTAB *d)
{
  char tmp[16];

  if(fread(buf,cd.divlen,1,cd.divfp)==1) {

    mkstring (d->symbol,&(buf[0]) ,7);
    mkstring (d->cusip ,&(buf[10]),12);

    memset (tmp, '\0', 16);
    mkstring (tmp,      &(buf[23]),10);
    d->div    = atof (tmp);
    d->div   /= 100000.0;

    memset (tmp, '\0', 16);
    mkstring (tmp,      &(buf[34]),10);
    d->split  = atof (tmp);
    d->split /= 100000.0;

    /* before TAQ2 (200001), tab files used YYMMDD format */
    memset (tmp, '\0', 16);
    if (cd.month < 200001) {
      mkstring (tmp,    &(buf[45]), 6);
      d->datef  = atoi(tmp) + 19000000;
    }
    else {
      mkstring (tmp,    &(buf[45]), 8);
      d->datef  = atoi(tmp);
    }
    return (1);
  }
  return (0);
}

/*
/// -------------------
/// int getmasttab(cd,m)
/// ===================
///    input parameters
///      TAQCD   cd  : structure containing open TAQ CD
///    output parameters
///      MASTTAB *m  : previously allocated space to contain master table record
///    description
///       reads in a master table record
///    returns
///                1 : successful
///                0 : unsuccessful
/// ___________________
*/
int getmasttab (TAQCD cd,MASTTAB *m)
{
  static char tmp[16];

  if(fread(buf, cd.mastlen, 1, cd.mastfp) == 1) {

    memset   (m->symbol, '\0',  7); /* better safe than sorry */
    memset   (m->cusip,  '\0', 12);
    memset   (m->name,   '\0', 30);

    mkstring (m->symbol,&(buf[0]) , 7);
    mkstring (m->name,  &(buf[10]),30);
    mkstring (m->cusip, &(buf[40]),12);
    memcpy   (m->etx,   &(buf[52]),10);
    m->its              = buf[62];
    memcpy   (m->icode, &(buf[63]), 4);

    memset (tmp, '\0', 16);
    mkstring (tmp,      &(buf[67]),10);
    m->sharesout = atoi (tmp);

    memcpy   (m->uot,   &(buf[77]), 4);
    m->denom            = buf[81];
    m->type             = buf[82];

    /* before TAQ2 (200001), tab files used YYMMDD format */
    memset (tmp, '\0', 16);
    if (cd.month < 200001) {
      mkstring (tmp,      &(buf[83]), 6);
      m->datef = atoi (tmp) + 19000000;
    }
    else {
      mkstring (tmp,      &(buf[83]), 8);
      m->datef = atoi (tmp);
    }
    return (1);
  }
  return (0);
}


/*-----------------------------------------
  NASD 10-28-1997 timestamp kludge
-----------------------------------------*/

/*
/// -------------------
/// int getnasdctidx(nasd1028file, ctidx)
/// ===================
///    input parameters
///       char *nasd1028file : nasd1028.txt, with correct NASDAQ 10-28-97
///                            timestamps
///       IDX          ctidx : trade index record read from TAQ CD
///    returns
///       1                  : successful
///      -1                  : unsuccessful
///    description
///       10-28-97 NASD timestamp kludge; modifies ctidx
/// ___________________
*/
int getnasdctidx(char *nasd1028file, IDX *ctidx)
{
  NASDINDEX *ptr;
  char s[16];
  int loc;
  int ctnasdlen = 47;

  if(ctidx->tdate%1000000!=971028) return(-1);

  if(nasd1028fp==NULL) {

    nasd1028fp=fopen(nasd1028file,"r");

    for(loc=1,nnasd1028=-1;fread(buf,ctnasdlen,1,nasd1028fp)==1;loc++) {
      memset (s, '\0', 16);
      mkstring(s,buf,10);

      if(nnasd1028==-1||strcmp(s,nasd1028[nnasd1028].key)) {

        nasd1028[++nnasd1028].loc=loc;
        strcpy(nasd1028[nnasd1028].key,s);
      }
    }
    nasd1028[++nnasd1028].loc=loc;
    rewind(nasd1028fp);
  }
  strcpy(nasd1028[nnasd1028].key,ctidx->symbol);
  ptr= bsearch (&(nasd1028[nnasd1028]), nasd1028, nnasd1028,
                sizeof(NASDINDEX), nasdcmp);

  if (ptr) {

    ctidx->brec = ptr->loc;
    ctidx->erec = (ptr+1)->loc-1;
    return((ptr+1)->loc-ptr->loc);
  }
  return(-1);
}

/*
/// -------------------
/// int getnasdctbin(irec, cd, ctbin)
/// ===================
///    input parameters
///       long   irec : TAQ CD record location (starts from record 1, not 0)
///       TAQCD    cd : structure containing open TAQ CD
///    output parameter
///       CTBIN ctbin : trade record read from TAQ CD
///    description
///       10-28-97 NASD timestamp kludge; 
///       reads from nasd1028.txt on 1097C TAQ CD
///    returns
///                 1 : successful
///                 0 : unsuccessful
/// ___________________
*/
int getnasdctbin(long irec, TAQCD cd, CTBIN *ctbin)
{
  static char s[32];
  int ctnasdlen = 47;

  if(irec<=0||fseek(nasd1028fp,((long) ((irec-1)*ctnasdlen)),0)) return(0);
  if(fread(buf,ctnasdlen,1,nasd1028fp)==1) {
    ctbin->ex      =                                  buf[16];
    memset (s, '\0', 32);
    ctbin->ttim    = clocktotime (atol (mkstring (s,&(buf[18]),6)));
    memset (s, '\0', 32);
    ctbin->price   = (LONGLONG)   atol (mkstring (s,&(buf[24]),9));
    memset (s, '\0', 32);
    ctbin->siz     =              atol (mkstring (s,&(buf[33]),9));
    memset (ctbin->cond, '\0', 4);
    ctbin->cond[0] =                                  buf[42];
    memset (s, '\0', 32);
    ctbin->corr    = (short)      atoi (mkstring (s,&(buf[43]),2));
    ctbin->tseq    = 0;
    ctbin->g127    = 0;
    return (1);
  }
  return (0);
}

/*
/// -------------------
/// int nasdcmp(a,b)
/// ===================
///    input parameters
///       NASDINDEX *a : NASD1028 kludge index record
///       NASDINDEX *b : NASD1028 kludge index record
///    description
///       compare routine to sort NASD1028 index records
/// ___________________
*/
int nasdcmp(const void *a,const void *b)
{
  return(strcmp(((NASDINDEX *)a)->key,((NASDINDEX *)b)->key));
}


/*-----------------------------------------
   Useful transformations of TAQ data
------------------------------------------*/

/*
/// -------------------
/// char mastetx(etx)
/// ===================
///    input parameters
///       char *etx : etx field, from master table record
///    description
///       returns character representing exchange (N,A,B,P,X,T,W)
/// ___________________
*/
char mastetx(char *etx)
{
  char ex;
  ex='?';
  if(etx[9]=='1') ex='W';
  if(etx[8]=='1') ex='T';
  if(etx[6]=='1') ex='X';
  if(etx[5]=='1') ex='P';
  if(etx[2]=='1') ex='B';
  if(etx[1]=='1') ex='A';
  if(etx[0]=='1') ex='N';
  return(ex);
}

/*
/// -----------------
/// long mastuot(uot)
/// =================
///    input parameters
///       char *uot : uot field, from master table record
///    description
///       returns round lot share size
/// _________________
*/
long mastuot(char *uot)
{
  char buf[16];
  memset (buf, '\0', 16);
  mkstring(buf, uot, 4);
  return((atol(buf) == 0 ? 100 : atol(buf)));
}

/*
/// -------------------
/// long mastdenom(char denom)
/// ===================
///    input parameters
///       char denom : denom field, from master table record
///    description
///       returns trading denomination (tick size) of the stock
///       as an integer multiplier
/// ___________________
*/
long mastdenom(char denom)
{
  switch (denom) {
  case 'B':
    return (  100);
  case '3':
    return (    8);
  case '4':
    return (   16);
  case '5':
    return (   32);
  case '6':
    return (   64);
  case '7':
    return (  128);
  case '8':
    return (  256);
  default:
    return (25600);
  }
}

/*
/// -------------------
/// char *masttype(type)
/// ===================
///    input parameters
///       char *type : type field, from master table record
///    description
///       returns static character string representing security type
/// ___________________
*/
char *masttype(char *type)
{
  static char *xtype[]={"Common","Preferred","Warrant","Right","Other",
      "Derivative"};
  return(*type>='0' && *type <= '5' ? xtype[*type-'0'] : "");
}


/*-----------------------------------------
   Utility routines
------------------------------------------*/

/*
/// -------------------
/// int idxcmp(a,b)
/// ===================
///    input parameters
///       IDX *a : TAQ CD index record
///       IDX *b : TAQ CD index record
///    description
///       compare routine to sort TAQ CD index records
/// ___________________
*/
int idxcmp(const void *a,const void *b)
{
  return(memcmp(((IDX *)a)->symbol, ((IDX *)b)->symbol, 7));
}




