/* queue.c */

#include "general.h"

void collapse_V_tags_forward(state *initial, int *p_num_states, double trans_uni_alpha) {
    static char rname[]="collapse_V_tags_forward";
    ts *state_q;
    state *next_state;

    state_q = (ts *) kmalloc(sizeof(ts));
    state_q->s1 = initial;
    state_q->next = NULL;

    while (state_q != NULL) {
        next_state = pop_from_top(&state_q);
        if (next_state == NULL)
            quit(-1, "%s: popped a state from the queue that no longer exists!\n",rname);

        if (DEBUG) fprintf(stderr, "Calling collapse_state_V_tags_forward on state %d\n", next_state->id);
        collapse_state_V_tags_forward(next_state, &state_q, p_num_states, trans_uni_alpha);
    }
}

void collapse_V_tags_backward(state *end, int *p_num_states, double trans_uni_alpha) {
    static char rname[]="collapse_V_tags_backward";
    ts *state_q;
    state *next_state;

    state_q = (ts *) kmalloc(sizeof(ts));
    state_q->s1 = end;
    state_q->next = NULL;

    while (state_q != NULL) {
        next_state = pop_from_top(&state_q);
        if (next_state == NULL)
            quit(-1, "%s: popped a state from the queue that no longer exists!\n", rname);

        if (DEBUG) fprintf(stderr, "Calling collapse_state_V_tags_backward on state %d\n", next_state->id);
        collapse_state_V_tags_backward(next_state, &state_q, p_num_states, trans_uni_alpha);
    }
}


void collapse_state_V_tags_forward(state *s1, ts **p_state_q, int *p_num_states, double trans_uni_alpha) {
    static char rname[]="collapse_state_V_tags_forward";
    trans *next_t, *trans_list, *first_t, *second_t;
    state *new, *child1, *child2;
    int match_found;

    if (s1 == NULL) return;
    if (s1->out == NULL) return;
    if (s1->seen == 1) {
        if (DEBUG) fprintf(stderr, "WARNING: Entered state %d for the second time...\n", s1->id);
        return;
    }

    /* Put all children in queue */
    first_t = s1->out;
    while (first_t != NULL) {
        if (first_t->dest->id != s1->id)
            push_on_bottom(p_state_q, first_t->dest);
        first_t = first_t->next_source;
    }

    /* Look for two destination states with the same label */
    first_t = s1->out;
    while (first_t != NULL) {

        match_found = 0;
        child1 = NULL;
        child2 = NULL;

        while (first_t != NULL && first_t->dest->id == s1->id) {
            first_t = first_t->next_source;
        }
        if (first_t == NULL) break;

        child1 = first_t->dest;
        second_t = first_t->next_source;

        while (second_t != NULL && (second_t->dest->id == s1->id || strcmp(second_t->dest->label, child1->label))) {
            second_t = second_t->next_source;
        }
        if (second_t != NULL) {
            child2 = second_t->dest;

            if (DEBUG) fprintf(stderr, "Found a match: %d %s, %d %s\n", child1->id, child1->label, child2->id, child2->label);
            match_found = 1;

            /* Get pointer to next transition now before removing the other two... */
            next_t = first_t->next_source;
            if (next_t->dest->id == child2->id) next_t = next_t->next_source;

            remove_from_q(p_state_q, child1);
            remove_from_q(p_state_q, child2);

            trans_list = NULL;
            new = merge_states(child1, child2, &trans_list, trans_uni_alpha);
            free_trans_list(trans_list);
            free_state(child2);
            free_state(child1);
            (*p_num_states)--;
            if (DEBUG) fprintf(stderr, "New state id = %d, label = %s\n", new->id, new->label);

            push_on_bottom(p_state_q, new);
            first_t = next_t;
        }
        else first_t = first_t->next_source;
    }
    s1->seen = 1;
}

void collapse_state_V_tags_backward(state *s1, ts **p_state_q, int *p_num_states, double trans_uni_alpha) {
    static char rname[]="collapse_state_V_tags_backward";
    trans *next_t, *trans_list, *first_t, *second_t;
    state *new, *child1, *child2;
    int match_found;

    if (s1 == NULL) return;
    if (s1->in == NULL) return;
    if (s1->seen == 1) {
        if (DEBUG) fprintf(stderr, "WARNING: Entered state %d for the second time...\n", s1->id);
        return;
    }

    /* Put all children in queue */
    first_t = s1->in;
    while (first_t != NULL) {
        if (first_t->source->id != s1->id)
            push_on_bottom(p_state_q, first_t->source);
        first_t = first_t->next_dest;
    }

    /* Look for two source states with the same label */
    first_t = s1->in;
    while (first_t != NULL) {

        match_found = 0;
        child1 = NULL;
        child2 = NULL;

        while (first_t != NULL && first_t->source->id == s1->id) {
            first_t = first_t->next_dest;
        }
        if (first_t == NULL) break;

        child1 = first_t->source;
        second_t = first_t->next_dest;

        while (second_t != NULL && (second_t->source->id == s1->id || strcmp(second_t->source->label, child1->label))) {
            second_t = second_t->next_dest;
        }
        if (second_t != NULL) {
            child2 = second_t->source;

            if (DEBUG) fprintf(stderr, "Found a match: %d %s, %d %s\n", child1->id, child1->label, child2->id, child2->label);
            match_found = 1;

            /* Get pointer to next transition now before removing the other two... */
            next_t = first_t->next_dest;
            if (next_t->source->id == child2->id) next_t = next_t->next_dest;

            trans_list = NULL;
            new = merge_states(child1, child2, &trans_list, trans_uni_alpha);
            free_trans_list(trans_list);

            remove_from_q(p_state_q, child1);
            remove_from_q(p_state_q, child2);

            free_state(child2);
            free_state(child1);
            (*p_num_states)--;
            if (DEBUG) fprintf(stderr, "New state id = %d, label = %s\n", new->id, new->label);

            push_on_bottom(p_state_q, new);
            first_t = next_t;
        }
        else first_t = first_t->next_dest;
    }
    s1->seen = 1;
}

void push_on_top(ts **p_state_q, state *s1) {
    static char rname[]="push_on_top";
    ts *temp;

    /* See if state is already on queue */
    temp = *p_state_q;
    while (temp != NULL) {
        if (temp->s1 == s1) return;
        temp = temp->next;
    } 

    /* State is not on queue, so add to top */
    temp = *p_state_q;
    *p_state_q = (ts *) kmalloc(sizeof(ts));
    (*p_state_q)->s1 = s1;
    (*p_state_q)->next = temp;
    return;
}

void push_on_bottom(ts **p_state_q, state *s1) {
    static char rname[]="push_on_bottom";
    ts *temp, *new, *prev;

    /* See if state is already on queue */
    temp = *p_state_q;
    prev = NULL;
    while (temp != NULL) {
        if (temp->s1 == s1) return;
        prev = temp;
        temp = temp->next;
    }

    /* State is not on queue, so add to bottom */
    new = (ts *) kmalloc(sizeof(ts));
    new->s1 = s1;
    new->next = NULL;

    if (*p_state_q == NULL) *p_state_q = new;
    else prev->next = new;
    return;
}


state *pop_from_top(ts **p_state_q) {
    static char rname[]="pop_from_top";
    ts *temp;
    state *s1;

    if (*p_state_q == NULL) return(NULL);

    temp = *p_state_q;
    *p_state_q = (*p_state_q)->next;
    s1 = temp->s1;
    free(temp);
    return(s1);
}


state *pop_from_bottom(ts **p_state_q) {
    static char rname[]="pop_from_bottom";
    ts *temp, *prev;
    state *s1;

    if (*p_state_q == NULL) return(NULL);

    prev= NULL;
    temp = *p_state_q;
    while (temp->next != NULL) {
        prev = temp;
        temp = temp->next;
    }

    prev->next = NULL;
    s1 = temp->s1;
    free(temp);
    return(s1);
}

void remove_from_q(ts **p_state_q, state *s1) {
    static char rname[]="remove_from_q";
    ts *temp, *prev, *next;
    int removed;

    removed = 0;
    prev = NULL;
    temp = *p_state_q;
    while (temp != NULL) {
        if (temp->s1 == s1) {

            if (prev == NULL) *p_state_q = (*p_state_q)->next;
            else prev->next = temp->next;

            removed = 1;

            next = temp->next;
            free(temp);
            temp = next;

        }
        else {
            prev = temp;
            temp = temp->next;
        }
    }

    return;
}


